from logging import Logger
from typing import Type, NoReturn

from httpx import Client, AsyncClient, Response

from ..exceptions import (
    NotAuthorizedException,
    NotFoundException,
    ForbiddenException,
    BadRequestException,
    NotAcceptableException,
    UnprocessableEntityException,
    UnknownException,
)
from ..models.api.attendances.responses import AttendancesByDayResponse, AttendanceResponse
from ..models.api.events.responses import ModifyEvent
from ..models.api.users.responses import UserStats


class BaseAdapter:
    def __init__(self, client_type: Type[Client] | Type[AsyncClient], api_key: str,
                 base_url: str = 'https://api.an.lafresa.org/attendance-control/v2'):
        self._client = client_type(headers={'api-key': api_key}, base_url=base_url, timeout=300, follow_redirects=True)

    def _raise_for_error(self, p_logger: Logger, resp: Response) -> NoReturn:
        """
        Преобразует HTTP-ошибку ответа в доменное исключение адаптера.
        Ожидает JSON с полем 'detail' (строка-код ошибки). При отсутствии — использует статус-код.
        """
        logger = p_logger.getChild('RE')
        status = resp.status_code
        detail: str | None = None
        body_repr: str | None = None
        try:
            data = resp.json()
            if isinstance(data, dict):
                raw_detail = data.get('detail')
                if isinstance(raw_detail, str):
                    detail = raw_detail
                else:
                    body_repr = str(data)
            else:
                body_repr = str(data)
        except Exception:
            try:
                body_repr = resp.text
            except Exception:
                body_repr = None

        # 400 Bad Request
        if status == 400:
            if detail == 'EVENT_NOT_REGISTERED':
                raise BadRequestException(BadRequestException.Entity.event_not_registered)
            raise UnknownException(f'HTTP 400: {detail or body_repr}')

        # 401 Unauthorized
        if status == 401:
            raise NotAuthorizedException(detail or 'UNAUTHORIZED')

        # 403 Forbidden
        if status == 403:
            if detail == 'INTEGRATION_NOT_ENABLED_FOR_DOMAIN':
                raise ForbiddenException(ForbiddenException.Entity.integration_not_enabled_for_domain)
            raise UnknownException(f'HTTP 403: {detail or body_repr}')

        # 404 Not Found
        if status == 404:
            if detail == 'EVENT_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.event)
            if detail == 'USER_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.user)
            raise UnknownException(f'HTTP 404: {detail or body_repr}')

        # 406 Not Acceptable
        if status == 406:
            if detail == 'CONFIDENCE_TOO_LOW':
                raise NotAcceptableException(detail, NotAcceptableException.Entity.confidence_too_low)
            raise UnknownException(f'HTTP 406: {detail or body_repr}')

        # 422 Unprocessable Entity
        if status == 422:
            raise UnprocessableEntityException(detail or 'VALIDATION_ERROR')

        # 500 Internal Server Error
        if status == 500:
            if detail == 'INTERNAL_SERVER_ERROR':
                raise UnknownException('INTERNAL_SERVER_ERROR')
            raise UnknownException(f'HTTP 500: {detail or body_repr}')

        # Everything else -> Unknown
        raise UnknownException(f'HTTP {status}: {detail or body_repr}')

    def _parse_post_events(self, p_logger: Logger, resp: Response) -> tuple[ModifyEvent, bool]:
        """
        Разбор ответа POST /events: создание или дополнение события.
        - 200: событие дополнено (модель ModifyEvent, created=False)
        - 202: событие создано (модель ModifyEvent, created=True)
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PE')
        if resp.status_code in (200, 202):
            result = ModifyEvent.model_validate(resp.json())
            created = resp.status_code == 202
            logger.debug('parsed event: event_uuid=%s, status=%s, created=%s', result.event_uuid, result.status,
                         created)
            return result, created
        self._raise_for_error(logger, resp)

    def _parse_get_attendances_by_day(self, p_logger: Logger, resp: Response) -> AttendancesByDayResponse:
        """
        Разбор ответа GET /attendances/by-day: возвращает посещения пользователя за день.
        - 200: модель AttendancesByDayResponse
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GA')
        if resp.status_code == 200:
            result = AttendancesByDayResponse.model_validate(resp.json())
            logger.debug('parsed attendances by day: user_uuid=%s, date=%s, count=%d',
                         result.user_uuid, result.date, len(result.attendances))
            return result
        self._raise_for_error(logger, resp)

    def _parse_post_attendances(self, p_logger: Logger, resp: Response) -> tuple[AttendanceResponse, bool]:
        """
        Разбор ответа POST /attendances: создание или обновление посещения.
        - 200: посещение обновлено (модель AttendanceResponse, created=False)
        - 202: посещение создано (модель AttendanceResponse, created=True)
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PA')
        if resp.status_code in (200, 202):
            result = AttendanceResponse.model_validate(resp.json())
            created = resp.status_code == 202
            logger.debug('parsed attendance: event_uuid=%s, user_uuid=%s, confidence=%s, status=%s, created=%s',
                         result.event_uuid, result.user_uuid, result.confidence, result.status, created)
            return result, created
        self._raise_for_error(logger, resp)

    def _parse_get_users_stats_by_user(self, p_logger: Logger, resp: Response) -> UserStats:
        """
        Разбор ответа GET /users/stats/by-user: возвращает статистику пользователя.
        - 200: модель StatsByUser
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GS')
        if resp.status_code == 200:
            result = UserStats.model_validate(resp.json())
            logger.debug('parsed user stats: user_uuid=%s', result.user_uuid)
            return result
        self._raise_for_error(logger, resp)
