from __future__ import annotations

from datetime import datetime
from logging import Logger
from typing import Tuple
from urllib.parse import quote

from httpx import AsyncClient, RequestError

from .base import BaseAdapter
from ..exceptions import UnknownException
from ..models.api.domains.responses import GetDomain
from ..models.api.enums.main import UserType, Integrations
from ..models.api.enums.monitoring_types import MonitoringType
from ..models.api.users.model import User
from ..models.api.users.requests import CreateIterator, ValidateUser
from ..models.api.users.responses import GetUsersStats, CreateUsersInterator
from ..models.api.integrations.responses import GetIntegrationConfig
from ..models.api.integrations.requests import SetIntegrationConfigRequest
from ..models.api.integrations import configs


class AsyncConfigsClient(BaseAdapter):
    """
    Асинхронный клиент для AN.API.CONFIGS с удобными методами.
    """

    def __init__(self, api_key: str, base_url: str = 'https://api.an.lafresa.org/configs/v1'):
        super().__init__(AsyncClient, api_key, base_url)

    async def aclose(self) -> None:
        """
        Закрывает внутренний HTTP-клиент.
        """
        try:
            await self._client.aclose()  # type: ignore[attr-defined]
        except Exception:
            # глушим закрытие; это best-effort
            pass

    # -------- Domains --------
    async def get_domain(self, p_logger: Logger, name: str) -> GetDomain:
        """
        Получить домен по имени.
        """
        logger = p_logger.getChild('GD')
        try:
            resp = await self._client.get(f'/domains/by-name/{quote(name, safe="")}')  # type: ignore[attr-defined]
            return self._parse_get_domains_by_name(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Users listing --------
    async def list_users(self, p_logger: Logger, iterator_uuid: str | None = None, page: int | None = None) -> list[User]:
        """
        Получить список пользователей (по итератору).
        """
        logger = p_logger.getChild('LU')
        params: dict[str, object] = {}
        if iterator_uuid is not None:
            params['iterator_uuid'] = iterator_uuid
        if page is not None:
            params['page_idx'] = page
        try:
            resp = await self._client.get('/users/', params=params)  # type: ignore[attr-defined]
            return self._parse_get_users(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def users_stats(self, p_logger: Logger, iterator_uuid: str | None = None) -> GetUsersStats:
        """
        Получить статистику пользователей для текущего итератора.
        """
        logger = p_logger.getChild('US')
        params: dict[str, object] = {}
        if iterator_uuid is not None:
            params['iterator_uuid'] = iterator_uuid
        try:
            resp = await self._client.get('/users/stats', params=params)  # type: ignore[attr-defined]
            return self._parse_get_users_stats(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def create_users_iterator(self, p_logger: Logger, query: str) -> CreateUsersInterator:
        """
        Создать итератор пользователей.
        """
        logger = p_logger.getChild('CI')
        model = CreateIterator(query=query)
        body = model.model_dump(mode='json')
        try:
            resp = await self._client.post('/users/iterator', json=body)  # type: ignore[attr-defined]
            return self._parse_post_users_iterator(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Single user fetch --------
    async def get_user_by_uuid(self, p_logger: Logger, uuid: str) -> User:
        """
        Получить пользователя по UUID.
        """
        logger = p_logger.getChild('GU')
        try:
            resp = await self._client.get(f'/users/by-uuid/{quote(uuid, safe="")}')  # type: ignore[attr-defined]
            return self._parse_get_users_by_uuid(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def get_user_by_email(self, p_logger: Logger, email: str) -> User:
        """
        Получить пользователя по email.
        """
        logger = p_logger.getChild('GE')
        try:
            resp = await self._client.get(f'/users/by-email/{quote(email, safe="")}')  # type: ignore[attr-defined]
            return self._parse_get_users_by_email(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Users modify --------
    async def save_user(self, p_logger: Logger, user: User) -> Tuple[User, bool]:
        """
        Апсерт пользователя (POST /users). Возвращает (user, created).
        """
        logger = p_logger.getChild('SU')
        # Готовим JSON, приводя enum к строке (имя)
        body = user.model_dump(mode='json')
        if hasattr(user, 'type'):
            try:
                body['type'] = user.type.name  # type: ignore[assignment]
            except Exception:
                pass
        try:
            resp = await self._client.post('/users/', json=body)  # type: ignore[attr-defined]
            return self._parse_post_users(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def create_user(self, p_logger: Logger, domain: str, email: str, user_type: UserType | str) -> Tuple[User, bool]:
        """
        Создать пользователя (PUT /users). Возвращает (user, created).
        """
        logger = p_logger.getChild('CU')
        # Валидируем входные данные и приводим enum к строке
        type_val = user_type.name if isinstance(user_type, UserType) else str(user_type)
        body = {'domain': domain, 'email': email, 'type': type_val}
        try:
            resp = await self._client.put('/users/', json=body)  # type: ignore[attr-defined]
            return self._parse_put_users(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def validate_user(self, p_logger: Logger, domain: str, email: str) -> Tuple[User, bool]:
        """
        Провалидировать email пользователя в домене; при необходимости создать (POST /users/validate).
        Возвращает (user, created).
        """
        logger = p_logger.getChild('VU')
        model = ValidateUser(domain=domain, email=email)
        body = model.model_dump(mode='json')
        try:
            resp = await self._client.post('/users/validate', json=body)  # type: ignore[attr-defined]
            return self._parse_post_users_validate(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def delete_user(self, p_logger: Logger, uuid: str) -> User:
        """
        Удалить пользователя по UUID.
        """
        logger = p_logger.getChild('DU')
        try:
            resp = await self._client.delete(f'/users/{quote(uuid, safe="")}')  # type: ignore[attr-defined]
            return self._parse_delete_users(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Monitoring --------
    async def record_monitoring(
        self,
        p_logger: Logger,
        integration: Integrations | str,
        entity_uuid: str,
        monitoring_type: MonitoringType | str,
        status: bool,
        timestamp: datetime | None = None,
    ) -> None:
        """
        Записать событие мониторинга интеграции.
        """
        logger = p_logger.getChild('RM')
        integ = integration.name if isinstance(integration, Integrations) else str(integration)
        mtype = monitoring_type.name if isinstance(monitoring_type, MonitoringType) else str(monitoring_type)
        body: dict[str, object] = {
            'integration': integ,
            'entity_uuid': entity_uuid,
            'monitoring_type': mtype,
            'status': status,
        }
        if timestamp is not None:
            body['timestamp'] = timestamp.isoformat()
        try:
            resp = await self._client.put('/monitoring/record', json=body)  # type: ignore[attr-defined]
            return self._parse_put_monitoring_record(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Integrations --------
    async def get_integration_config(
        self,
        p_logger: Logger,
        domain: str,
        integration: Integrations | str,
    ) -> GetIntegrationConfig:
        """
        Получить конфигурацию интеграции (GET /integrations/configs).
        """
        logger = p_logger.getChild('GIC')
        integ = integration.name if isinstance(integration, Integrations) else str(integration)
        params: dict[str, object] = {
            'domain': domain,
            'integration': integ,
        }
        try:
            resp = await self._client.get('/integrations/configs', params=params)  # type: ignore[attr-defined]
            return self._parse_get_integration_config(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def set_integration_config(
        self,
        p_logger: Logger,
        domain: str,
        integration: Integrations | str,
        payload: configs.AttendanceControlConfig,
    ) -> Tuple[GetIntegrationConfig, bool]:
        """
        Создать или обновить конфигурацию интеграции (PUT /integrations/configs).
        Возвращает (config, created).
        """
        logger = p_logger.getChild('SIC')
        integ = integration.name if isinstance(integration, Integrations) else str(integration)
        params: dict[str, object] = {
            'domain': domain,
            'integration': integ,
        }
        request = SetIntegrationConfigRequest(payload=payload)
        body = request.model_dump(mode='json')
        try:
            resp = await self._client.put('/integrations/configs', params=params, json=body)  # type: ignore[attr-defined]
            return self._parse_put_integration_config(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc
