from logging import Logger
from typing import Type, NoReturn

from httpx import Client, AsyncClient, Response

from ..exceptions import (
    NotAuthorizedException,
    NotFoundException,
    UnknownException,
)
from ..models.api.event import Event


class BaseAdapter:
    def __init__(self, client_type: Type[Client] | Type[AsyncClient], api_key: str,
                 base_url: str = 'https://api.an.lafresa.org/events/v1'):
        self._client = client_type(headers={'api-key': api_key}, base_url=base_url, timeout=300)

    def _raise_for_error(self, p_logger: Logger, resp: Response) -> NoReturn:
        """
        Преобразует HTTP-ошибку ответа в доменное исключение адаптера.
        Ожидает JSON с полем 'detail' (строка-код ошибки). При отсутствии — использует статус-код.
        """
        logger = p_logger.getChild('RE')
        status = resp.status_code
        detail: str | None = None
        body_repr: str | None = None
        try:
            data = resp.json()
            if isinstance(data, dict):
                raw_detail = data.get('detail')
                if isinstance(raw_detail, str):
                    detail = raw_detail
                else:
                    body_repr = str(data)
            else:
                body_repr = str(data)
        except Exception:
            try:
                body_repr = resp.text
            except Exception:
                body_repr = None

        # 401 Unauthorized
        if status == 401:
            raise NotAuthorizedException(detail or 'UNAUTHORIZED')

        # 404 Not Found mappings
        if status == 404:
            # По умолчанию считаем, что это событие не найдено
            # В будущем можно расширить маппинг по detail, если появятся специфичные коды
            raise NotFoundException(NotFoundException.Entity.event)

        # 400 Bad Request - на первом этапе сводим к Unknown
        if status == 400:
            raise UnknownException(f'HTTP 400: {detail or body_repr}')

        # 409 Conflict - на первом этапе сводим к Unknown
        if status == 409:
            raise UnknownException(f'HTTP 409: {detail or body_repr}')

        # 422 Unprocessable Entity - на первом этапе сводим к Unknown
        if status == 422:
            raise UnknownException(f'HTTP 422: {detail or body_repr}')

        # 500+ or everything else -> Unknown
        raise UnknownException(f'HTTP {status}: {detail or body_repr}')

    def _parse_get_event(self, p_logger: Logger, resp: Response) -> Event:
        """
        Разбор ответа GET /event/{uuid}: возвращает событие.
        - 200: модель Event
        - Иначе: возбуждает соответствующее исключение маппингом статуса/детали.
        """
        logger = p_logger.getChild('GE')
        if resp.status_code == 200:
            result = Event.model_validate(resp.json())
            logger.debug('parsed event: uuid=%s, domain=%s, title=%s', result.uuid, result.domain, result.title)
            return result
        self._raise_for_error(logger, resp)
