from __future__ import annotations

from logging import Logger
from urllib.parse import quote

from httpx import AsyncClient, RequestError

from .base import BaseAdapter
from ..exceptions import UnknownException
from ..models.api.event import Event


class AsyncEventsClient(BaseAdapter):
    """
    Асинхронный клиент для Events Manager API (AN.API.EVENTS) с удобными методами.
    """

    def __init__(self, api_key: str, base_url: str = 'https://api.an.lafresa.org/events/v1'):
        super().__init__(AsyncClient, api_key, base_url)

    # -------- Events --------
    async def get_event(self, p_logger: Logger, uuid: str) -> Event:
        """
        Получить информацию о событии по UUID.
        
        Соответствует эндпоинту GET /event/{uuid}.
        
        Args:
            p_logger: Логгер для записи операций
            uuid: Уникальный идентификатор события
            
        Returns:
            Event: Модель события
            
        Raises:
            NotFoundException: Если событие не найдено (404)
            NotAuthorizedException: Если не авторизован (401)
            UnknownException: При других ошибках
        """
        logger = p_logger.getChild('GE')
        try:
            resp = await self._client.get(f'/event/{quote(uuid, safe="")}')  # type: ignore[attr-defined]
            return self._parse_get_event(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc
