from __future__ import annotations

from datetime import date, datetime
from logging import Logger
from typing import Tuple

from httpx import AsyncClient, RequestError

from .base import BaseAdapter
from ..exceptions import UnknownException
from ..models.api.attendances.requests import CreateAttendanceRequest
from ..models.api.attendances.responses import AttendancesByDayResponse, AttendanceResponse
from ..models.api.events.requests import ModifyEvent as CreateEventRequest
from ..models.api.events.responses import ModifyEvent
from ..models.api.users.responses import UserStats


class AsyncAttendanceControlClient(BaseAdapter):
    """
    Асинхронный клиент для AN.API.ATTENDANCE-CONTROL с удобными методами.
    """

    def __init__(self, api_key: str, base_url: str = 'https://api.an.lafresa.org/attendance-control/v2'):
        super().__init__(AsyncClient, api_key, base_url)

    async def aclose(self) -> None:
        """
        Закрывает внутренний HTTP-клиент.
        """
        try:
            await self._client.aclose()  # type: ignore[attr-defined]
        except Exception:
            # глушим закрытие; это best-effort
            pass

    # -------- Events --------
    async def create_or_update_event(self, p_logger: Logger, event: CreateEventRequest) -> Tuple[ModifyEvent, bool]:
        """
        Создает или дополняет событие в БД (POST /events).
        Возвращает (event, created) где created=True если событие создано (202), False если дополнено (200).
        """
        logger = p_logger.getChild('CE')
        try:
            resp = await self._client.post('/events', content=event.model_dump_json())  # type: ignore[attr-defined]
            return self._parse_post_events(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Attendances --------
    async def create_or_update_attendance(
            self,
            p_logger: Logger,
            attendance: CreateAttendanceRequest,
    ) -> Tuple[AttendanceResponse, bool]:
        """
        Создает или обновляет посещение в БД (POST /attendances).
        Возвращает (attendance, created) где created=True если посещение создано (202), False если обновлено (200).
        """
        logger = p_logger.getChild('CA')
        try:
            resp = await self._client.post('/attendances', content=attendance.model_dump_json())  # type: ignore[attr-defined]
            return self._parse_post_attendances(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    async def get_attendances_by_day(
            self,
            p_logger: Logger,
            user_uuid: str,
            date: date | None = None,
    ) -> AttendancesByDayResponse:
        """
        Получить посещения пользователя за день (GET /attendances/by-day).
        Если дата не указана, возвращается список за текущий день.
        """
        logger = p_logger.getChild('GA')
        params: dict[str, object] = {'user_uuid': user_uuid}
        if date is not None:
            params['date'] = date.isoformat()
        try:
            resp = await self._client.get('/attendances/by-day', params=params)  # type: ignore[attr-defined]
            return self._parse_get_attendances_by_day(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc

    # -------- Users Stats --------
    async def get_user_stats(
            self,
            p_logger: Logger,
            user_uuid: str,
            range_from_dt: datetime,
            range_till_dt: datetime,
    ) -> UserStats:
        """
        Получить статистику посещаемости пользователя за указанный период (GET /users/stats/by-user).
        """
        logger = p_logger.getChild('GS')
        params: dict[str, object] = {
            'user_uuid': user_uuid,
            'range_from_dt': range_from_dt.isoformat(),
            'range_till_dt': range_till_dt.isoformat(),
        }
        try:
            resp = await self._client.get('/users/stats/by-user', params=params)  # type: ignore[attr-defined]
            return self._parse_get_users_stats_by_user(logger, resp)
        except RequestError as exc:
            raise UnknownException(f'NETWORK_ERROR: {exc}') from exc
