from httpx import Client, AsyncClient, Response
from typing import Type, NoReturn
from logging import Logger
from pydantic import TypeAdapter

from ..models.api.users.model import User
from ..models.api.users.responses import GetUsersStats, CreateUsersInterator
from ..models.api.domains.responses import GetDomain
from ..models.api.integrations.responses import GetIntegrationConfig
from ..exceptions import (
    NotAuthorizedException,
    NotFoundException,
    BadRequestException,
    UnprocessableEntityException,
    ConflictException,
    UnknownException,
)

class BaseAdapter:
    def __init__(self, client_type: Type[Client] | Type[AsyncClient], api_key: str,
                 base_url: str = 'https://api.an.lafresa.org/configs/v1'):
        self._client = client_type(headers={'api-key': api_key}, base_url=base_url, timeout=300)

    def _raise_for_error(self, p_logger: Logger, resp: Response) -> NoReturn:
        """
        Преобразует HTTP-ошибку ответа в доменное исключение адаптера.
        Ожидает JSON с полем 'detail' (строка-код ошибки). При отсутствии — использует статус-код.
        """
        logger = p_logger.getChild('RE')
        status = resp.status_code
        detail: str | None = None
        body_repr: str | None = None
        try:
            data = resp.json()
            if isinstance(data, dict):
                raw_detail = data.get('detail')
                if isinstance(raw_detail, str):
                    detail = raw_detail
                else:
                    body_repr = str(data)
            else:
                body_repr = str(data)
        except Exception:
            try:
                body_repr = resp.text
            except Exception:
                body_repr = None

        # 401 Unauthorized
        if status == 401:
            raise NotAuthorizedException(detail or 'UNAUTHORIZED')

        # 404 Not Found mappings
        if status == 404:
            if detail == 'DOMAIN_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.domain)
            if detail in ('USER_NOT_FOUND', 'USER_NOT_FOUND_FOR_INTEGRATIONS'):
                raise NotFoundException(NotFoundException.Entity.user)
            if detail == 'INTEGRATION_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.integration)
            if detail == 'INTEGRATION_UNIT_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.monitoring_unit)
            if detail == 'MONITORING_TYPE_NOT_FOUND':
                raise NotFoundException(NotFoundException.Entity.monitoring_type)
            raise UnknownException(f'HTTP 404: {detail or body_repr}')

        # 400 Bad Request mappings
        if status == 400:
            if detail == 'EMAIL_NOT_MATCH_DOMAIN_PATTERN':
                raise BadRequestException(BadRequestException.Reason.email_not_match_domain_pattern)
            raise UnknownException(f'HTTP 400: {detail or body_repr}')

        # 409 Conflict mappings
        if status == 409:
            if detail == 'USER_DOMAIN_MISMATCH':
                raise ConflictException(ConflictException.Reason.user_domain_mismatch)
            if detail == 'USER_DOMAIN_MISMATCH_FOR_INVALIDATED':
                raise ConflictException(ConflictException.Reason.user_domain_mismatch_for_invalidated)
            raise UnknownException(f'HTTP 409: {detail or body_repr}')

        # 422 Unprocessable Entity mappings
        if status == 422:
            if detail == 'INVALID_UUID':
                raise UnprocessableEntityException(UnprocessableEntityException.Reason.invalid_uuid)
            if detail == 'USER_EMAIL_CONFLICT':
                raise UnprocessableEntityException(UnprocessableEntityException.Reason.user_email_conflict)
            if detail == 'INTEGRATION_NOT_AVAILABLE_FOR_DOMAIN':
                raise UnprocessableEntityException(UnprocessableEntityException.Reason.integration_not_available_for_domain)
            raise UnknownException(f'HTTP 422: {detail or body_repr}')

        # 500+ or everything else -> Unknown
        raise UnknownException(f'HTTP {status}: {detail or body_repr}')

    def _parse_get_users(self, p_logger: Logger, resp: Response) -> list[User]:
        """
        Разбор ответа GET /users: возвращает список пользователей.
        - 200: список моделей User
        - Иначе: возбуждает соответствующее исключение маппингом статуса/детали.
        """
        logger = p_logger.getChild('GU')
        if resp.status_code == 200:
            users = TypeAdapter(list[User]).validate_python(resp.json())
            logger.debug('parsed users list: %d entries', len(users))
            return users
        self._raise_for_error(logger, resp)

    def _parse_get_users_stats(self, p_logger: Logger, resp: Response) -> GetUsersStats:
        """
        Разбор ответа GET /users/stats: возвращает агрегированную статистику.
        - 200: модель GetUsersStats
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GS')
        if resp.status_code == 200:
            result = GetUsersStats.model_validate(resp.json())
            logger.debug('parsed users stats: user_num=%d', result.user_num)
            return result
        self._raise_for_error(logger, resp)

    def _parse_post_users_iterator(self, p_logger: Logger, resp: Response) -> CreateUsersInterator:
        """
        Разбор ответа POST /users/iterator: возвращает параметры созданного итератора.
        - 200: модель CreateUsersInterator
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PI')
        if resp.status_code == 200:
            result = CreateUsersInterator.model_validate(resp.json())
            logger.debug(
                'parsed users iterator: iterator_uuid=%s, pages=%d',
                result.iterator_uuid, result.pages
            )
            return result
        self._raise_for_error(logger, resp)

    def _parse_get_users_by_uuid(self, p_logger: Logger, resp: Response) -> User:
        """
        Разбор ответа GET /users/by-uuid/{uuid}: возвращает пользователя.
        - 200: модель User
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GU')
        if resp.status_code == 200:
            result = User.model_validate(resp.json())
            logger.debug('parsed user by uuid: %s', result.uuid)
            return result
        self._raise_for_error(logger, resp)

    def _parse_get_users_by_email(self, p_logger: Logger, resp: Response) -> User:
        """
        Разбор ответа GET /users/by-email/{email}: возвращает пользователя.
        - 200: модель User
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GE')
        if resp.status_code == 200:
            result = User.model_validate(resp.json())
            logger.debug('parsed user by email: %s', result.emails.main_email)
            return result
        self._raise_for_error(logger, resp)

    def _parse_post_users(self, p_logger: Logger, resp: Response) -> tuple[User, bool]:
        """
        Разбор ответа POST /users: апсерт пользователя.
        - 200 | 201: (модель User, created: bool) — created=True если статус 201
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PO')
        if resp.status_code in (200, 201):
            result = User.model_validate(resp.json())
            created = resp.status_code == 201
            logger.debug('parsed user via POST: %s (%s), created=%s', result.uuid, result.domain, created)
            return result, created
        self._raise_for_error(logger, resp)

    def _parse_put_users(self, p_logger: Logger, resp: Response) -> tuple[User, bool]:
        """
        Разбор ответа PUT /users: создание пользователя.
        - 200 | 201: (модель User, created: bool) — created=True если статус 201
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PU')
        if resp.status_code in (200, 201):
            result = User.model_validate(resp.json())
            created = resp.status_code == 201
            logger.debug('parsed user via PUT: %s (%s), created=%s', result.uuid, result.domain, created)
            return result, created
        self._raise_for_error(logger, resp)

    def _parse_post_users_validate(self, p_logger: Logger, resp: Response) -> tuple[User, bool]:
        """
        Разбор ответа POST /users/validate: валидация/создание пользователя.
        - 200 | 201: (модель User, created: bool) — created=True если статус 201
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('VA')
        if resp.status_code in (200, 201):
            result = User.model_validate(resp.json())
            created = resp.status_code == 201
            logger.debug('parsed user via VALIDATE: %s (%s), created=%s', result.uuid, result.domain, created)
            return result, created
        self._raise_for_error(logger, resp)

    def _parse_delete_users(self, p_logger: Logger, resp: Response) -> User:
        """
        Разбор ответа DELETE /users/{uuid}: удаление пользователя.
        - 200: модель User (содержит удалённого пользователя)
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('DL')
        if resp.status_code == 200:
            result = User.model_validate(resp.json())
            logger.debug('parsed deleted user: %s', result.uuid)
            return result
        self._raise_for_error(logger, resp)

    def _parse_get_domains_by_name(self, p_logger: Logger, resp: Response) -> GetDomain:
        """
        Разбор ответа GET /domains/by-name/{name}: возвращает домен.
        - 200: модель GetDomain
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GDN')
        if resp.status_code == 200:
            result = GetDomain.model_validate(resp.json())
            logger.debug('parsed domain: %s', result.name)
            return result
        self._raise_for_error(logger, resp)

    def _parse_put_monitoring_record(self, p_logger: Logger, resp: Response) -> None:
        """
        Разбор ответа PUT /monitoring/record: запись события мониторинга.
        - 200: успех (возвращает None)
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PR')
        if resp.status_code == 200:
            logger.debug('parsed monitoring record result: OK')
            return None
        self._raise_for_error(logger, resp)

    def _parse_get_integration_config(self, p_logger: Logger, resp: Response) -> GetIntegrationConfig:
        """
        Разбор ответа GET /integrations/configs: возвращает конфигурацию интеграции.
        - 200: модель GetIntegrationConfig
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('GIC')
        if resp.status_code == 200:
            result = GetIntegrationConfig.model_validate(resp.json())
            logger.debug(
                'parsed integration config: domain=%s, integration=%s',
                result.domain,
                result.integration,
            )
            return result
        self._raise_for_error(logger, resp)

    def _parse_put_integration_config(self, p_logger: Logger, resp: Response) -> tuple[GetIntegrationConfig, bool]:
        """
        Разбор ответа PUT /integrations/configs: создание или обновление конфигурации интеграции.
        - 200 | 201: (модель GetIntegrationConfig, created: bool) — created=True если статус 201
        - Иначе: возбуждает соответствующее исключение.
        """
        logger = p_logger.getChild('PIC')
        if resp.status_code in (200, 201):
            result = GetIntegrationConfig.model_validate(resp.json())
            created = resp.status_code == 201
            logger.debug(
                'parsed integration config via PUT: domain=%s, integration=%s, created=%s',
                result.domain,
                result.integration,
                created,
            )
            return result, created
        self._raise_for_error(logger, resp)

    