"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonitoredCertificate = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const certificate_monitor_1 = require("./certificate-monitor");
/**
 * Monitors an ACM certificate's expiration date. Tracks the days to expiry
 * metric published by ACM (until the certificate has expired), and also
 * inspects the certificate used by the provided `domainName` endpoint for
 * serving HTTPS webpages.
 *
 * These ensure ACM certificates are rotated by the time automated renewal would
 * have happened (60 to 45 days prior to expiration), and that the endpoint is
 * updated to a renewed certificate in due time.
 */
class MonitoredCertificate extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.props = props;
        this.endpointMetricName = (_a = props.metricName) !== null && _a !== void 0 ? _a : 'DaysToExpiry';
        this.endpointMetricNamespace = (_b = props.metricNamespace) !== null && _b !== void 0 ? _b : core_1.Stack.of(this).stackName;
        const dynamicMonitor = new certificate_monitor_1.CertificateMonitor(this, 'Monitor', {
            description: `Monitors the days to expiry of the certificate used to serve ${props.domainName}`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                HTTPS_ENDPOINT: props.domainName,
                METRIC_NAME: this.endpointMetricName,
                METRIC_NAMESPACE: this.endpointMetricNamespace,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(5),
        });
        const rule = new aws_events_1.Rule(this, 'Schedule', {
            description: `Executes the ${dynamicMonitor.functionName} function every 30 minutes`,
            schedule: aws_events_1.Schedule.rate(core_1.Duration.minutes(30)),
            targets: [new aws_events_targets_1.LambdaFunction(dynamicMonitor)],
        });
        this.alarmAcmCertificateExpiresSoon = this.metricAcmCertificateDaysToExpiry()
            .createAlarm(this, 'ACMAlarm', {
            alarmDescription: `The ACM certificate ${props.certificate.certificateArn} will expire in less than 45 days!`,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 45,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        this.alarmEndpointCertificateExpiresSoon = this.metricEndpointCertificateDaysToExpiry()
            .createAlarm(this, 'EndpointAlarm', {
            alarmDescription: `The certificate used to serve ${props.domainName} will expire in less than 45 days!`,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 45,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        // Ensure we don't alarm before the function's trigger has been created...
        this.alarmEndpointCertificateExpiresSoon.node.addDependency(rule);
    }
    /**
     * The remaining days before the monitored certificate expires, as far as ACM
     * is concerned. This metric is no longer emitted after the certificate has
     * expired (alarms should treat missing data as `<= 0`).
     */
    metricAcmCertificateDaysToExpiry(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.days(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            dimensions: { CertificateArn: this.props.certificate.certificateArn },
            namespace: 'AWS/CertificateManager',
            metricName: 'DaysToExpiry',
            region: 'us-east-1',
        });
    }
    /**
     * The remaining days before the certificate served by the configured
     * `domainName` expires. This metric is published as 0 if the certificate has
     * already expired.
     */
    metricEndpointCertificateDaysToExpiry(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.days(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            dimensions: { DomainName: this.props.domainName },
            namespace: this.endpointMetricNamespace,
            metricName: this.endpointMetricName,
        });
    }
}
exports.MonitoredCertificate = MonitoredCertificate;
//# sourceMappingURL=data:application/json;base64,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