"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.shellOutWithOutput = exports.shellOut = void 0;
const child_process_1 = require("child_process");
/**
 * Executes the specified command in a sub-shell, and asserts success.
 */
function shellOut(cmd, ...args) {
    return new Promise((ok, ko) => {
        const child = child_process_1.spawn(cmd, args, { stdio: ['ignore', 'inherit', 'inherit'] });
        child.once('error', ko);
        child.once('close', (code, signal) => {
            if (code === 0) {
                return ok();
            }
            const reason = code != null
                ? `exit code ${code}`
                : `signal ${signal}`;
            ko(new Error(`Command "${cmd} ${args.join(' ')}" failed with ${reason}`));
        });
    });
}
exports.shellOut = shellOut;
/**
 * Executes the specified command in a sub-shell. Instead of asserting success,
 * this captures all data sent to `STDOUT` and returns that, with the command's
 * exit code or signal.
 */
function shellOutWithOutput(cmd, ...args) {
    return new Promise((ok, ko) => {
        const child = child_process_1.spawn(cmd, args, { stdio: ['ignore', 'pipe', 'inherit'] });
        const chunks = new Array();
        child.stdout.on('data', (chunk) => {
            chunks.push(Buffer.from(chunk));
        });
        child.once('error', ko);
        child.once('close', (exitCode, signal) => {
            const stdout = Buffer.concat(chunks);
            ok({ exitCode, signal, stdout });
        });
    });
}
exports.shellOutWithOutput = shellOutWithOutput;
//# sourceMappingURL=data:application/json;base64,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