"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.MISSING_DOCUMENTATION_KEY_PATTERN = exports.missingDocumentationKey = exports.CATALOG_KEY = exports.NOT_SUPPORTED_SUFFIX = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_CSHARP = exports.DOCS_KEY_SUFFIX_JAVA = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_JAVA = docsKeySuffix(language_1.DocumentationLanguage.JAVA);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_CSHARP = docsKeySuffix(language_1.DocumentationLanguage.CSHARP);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * The keys for missing documentation lists.
 *
 * @param language the language for which missing documentation is requested.
 */
function missingDocumentationKey(language) {
    return `missing-objects/${language.name}-documentation.json`;
}
exports.missingDocumentationKey = missingDocumentationKey;
/**
 * The key pattern for objects containing missing documentation lists.
 */
exports.MISSING_DOCUMENTATION_KEY_PATTERN = 'missing-objects/*-documentation.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,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