"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const _lambda_architecture_1 = require("../_lambda-architecture");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const transliterator_1 = require("../transliterator");
const needs_catalog_update_1 = require("./needs-catalog-update");
const redrive_state_machine_1 = require("./redrive-state-machine");
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).next(new aws_stepfunctions_1.Succeed(this, 'Sent to DLQ'));
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' });
        const needsCatalogUpdateFunction = new needs_catalog_update_1.NeedsCatalogUpdate(this, 'NeedsCatalogUpdate', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Orchestration/NeedsCatalogUpdate] Determines whether a package version requires a catalog update',
            environment: { CATALOG_BUCKET_NAME: props.bucket.bucketName, CATALOG_OBJECT_KEY: constants_1.CATALOG_KEY },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        props.bucket.grantRead(needsCatalogUpdateFunction);
        // Check whether the catalog needs updating. If so, trigger addToCatalog.
        const addToCatalogIfNeeded = new tasks.LambdaInvoke(this, 'Check whether catalog needs udpating', {
            lambdaFunction: needsCatalogUpdateFunction,
            payloadResponseOnly: true,
            resultPath: '$.catalogNeedsUpdating',
        })
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(new aws_stepfunctions_1.Choice(this, 'Is catalog update needed?')
            .when(aws_stepfunctions_1.Condition.booleanEquals('$.catalogNeedsUpdating', true), addToCatalog)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        })
            .next(new aws_stepfunctions_1.Pass(this, 'Prepare doc-gen ECS Command', {
            parameters: { 'command.$': 'States.Array(States.JsonToString($))' },
            resultPath: '$.docGen',
        }))
            .next(this.transliterator.createEcsRunTask(this, 'Generate docs', {
            cluster: this.ecsCluster,
            inputPath: '$.docGen.command',
            resultPath: '$.docGenOutput',
            // Expect this to complete within one hour
            timeout: core_1.Duration.hours(1),
            vpcSubnets: props.vpcSubnets,
        })
            // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
            .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
            .addRetry({
            errors: [
                'ECS.AmazonECSException',
                'ECS.InvalidParameterException',
                'jsii-docgen.NpmError.E429',
                'jsii-codgen.NpmError.EPROTO',
            ],
            ...THROTTLE_RETRY_POLICY,
        })
            .addRetry({
            errors: ['jsii-docgen.NpmError.ETARGET'],
            // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
            backoffRate: 2,
            interval: core_1.Duration.minutes(5),
            maxAttempts: 3,
        })
            .addRetry({ maxAttempts: 3 })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.Timeout'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(addToCatalogIfNeeded));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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