"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const codeArtifactProps = (function () {
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            return undefined;
        }
        const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
        const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
        const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
        return { endpoint, domain, domainOwner, apiEndpoint };
    })();
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const assembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            constructFramework = detectConstructFramework(assembly);
            const { license, name, version } = assembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        if (codeArtifactProps) {
            console.log('Publishing to the internal CodeArtifact...');
            try {
                const { publishConfig } = packageJsonObj;
                if (publishConfig) {
                    console.log('Not publishing to CodeArtifact due to the presence of publishConfig in package.json: ', publishConfig);
                }
                else {
                    await code_artifact_lambda_shared_1.codeArtifactPublishPackage(Buffer.from(tarball.Body), codeArtifactProps);
                }
            }
            catch (err) {
                console.error('Failed publishing to CodeArtifact: ', err);
            }
        }
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public, max-age=86400, must-revalidate, s-maxage=300, proxy-revalidate',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public, max-age: 86400, must-revalidate, s-maxage=300, proxy-revalidate',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf' || packageName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,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