"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const config_file_1 = require("../../config-file");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const _lambda_architecture_1 = require("../_lambda-architecture");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const configFilename = 'config.json';
        const config = new config_file_1.ConfigFile(configFilename, JSON.stringify({
            packageLinks: (_a = props.packageLinks) !== null && _a !== void 0 ? _a : [],
            packageTags: (_b = props.packageTags) !== null && _b !== void 0 ? _b : [],
        }));
        const configBucket = new aws_s3_1.Bucket(this, 'ConfigBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'DeployIngestionConfiguration', {
            sources: [aws_s3_deployment_1.Source.asset(config.dir)],
            destinationBucket: configBucket,
        });
        const environment = {
            AWS_EMF_ENVIRONMENT: 'Local',
            BUCKET_NAME: props.bucket.bucketName,
            CONFIG_BUCKET_NAME: configBucket.bucketName,
            CONFIG_FILE_KEY: configFilename,
            STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
        };
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.publishingRepositoryNpmEndpoint;
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
        }
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment,
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        configBucket.grantRead(handler);
        props.bucket.grantWrite(this.function);
        (_d = props.codeArtifact) === null || _d === void 0 ? void 0 : _d.grantPublishToRepository(handler);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            memorySize: 10240,
            tracing: aws_lambda_1.Tracing.ACTIVE,
            timeout: core_1.Duration.minutes(3),
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        // Need to physical-name the state machine so it can self-invoke.
        const stateMachineName = stateMachineNameFrom(this.node.path);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a ContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.ContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.ContinuationToken'),
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        listBucket.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_tasks_1.StepFunctionsStartExecution(this, 'Continue as new', {
            associateWithParent: true,
            stateMachine: aws_stepfunctions_1.StateMachine.fromStateMachineArn(this, 'ThisStateMachine', core_1.Stack.of(this).formatArn({
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                service: 'states',
                resource: 'stateMachine',
                resourceName: stateMachineName,
            })),
            input: aws_stepfunctions_1.TaskInput.fromObject({ ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken') }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })).afterwards({ includeOtherwise: true })
            .next(process));
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            stateMachineName,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(stateMachine);
        props.queue.grantSendMessages(stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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