"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const clients = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const client_lambda_shared_1 = require("./client.lambda-shared");
const constants_1 = require("./constants");
const s3 = clients.s3();
const sqs = clients.sqs();
const lambda = clients.lambda();
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event) {
    var _a;
    console.log(`Event: ${JSON.stringify(event)}`);
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setDimensions();
        const ruleCount = Object.keys(client.map).length;
        metrics.putMetric("DenyListRuleCount" /* DENY_LIST_RULE_COUNT */, ruleCount, aws_embedded_metrics_1.Unit.Count);
    })();
    const packageData = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_PACKAGE_DATA_BUCKET_NAME);
    const pruneQueue = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_QUEUE_URL);
    const keyPrefix = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_PACKAGE_DATA_KEY_PREFIX);
    const objectsFound = new Array();
    for (const nameVersion of Object.keys(client.map)) {
        const prefix = `${keyPrefix}${nameVersion}/`;
        console.log(`Querying bucket ${packageData} for all objects with prefix ${prefix}`);
        // for each prefix, query the package data bucket for all objects with that
        // prefix and delete them
        let continuation = undefined;
        do {
            const req = {
                Bucket: packageData,
                Prefix: prefix,
                ContinuationToken: continuation,
            };
            const result = await s3.listObjectsV2(req).promise();
            continuation = result.NextContinuationToken;
            // queue all objects for deletion
            for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
                if (!object.Key) {
                    continue;
                }
                const sendMessageRequest = {
                    QueueUrl: pruneQueue,
                    MessageBody: object.Key,
                };
                console.log(JSON.stringify({ sendMessageRequest }));
                const sendMessageResponse = await sqs.sendMessage(sendMessageRequest).promise();
                console.log(JSON.stringify({ sendMessageResponse }));
                objectsFound.push(object.Key);
            }
        } while (continuation);
        // trigger the "on change" handler objects were found and we have a handler
        const onChangeFunctionName = process.env[constants_1.ENV_PRUNE_ON_CHANGE_FUNCTION_NAME];
        if (onChangeFunctionName && objectsFound.length > 0) {
            console.log(`Triggering a on-change handler: ${onChangeFunctionName}`);
            const onChangeCallbackRequest = {
                FunctionName: onChangeFunctionName,
                InvocationType: 'Event',
            };
            console.log(JSON.stringify({ onChangeCallbackRequest }));
            const onChangeCallbackResponse = await lambda.invoke(onChangeCallbackRequest).promise();
            console.log(JSON.stringify({ onChangeCallbackResponse }));
        }
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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