"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const missing_documentation_widget_1 = require("./backend/inventory/missing-documentation-widget");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:primary:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder.function)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkCount({ label: 'No Construct Framework' }),
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkVersionCount({ label: 'No Construct Framework Version' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:primary:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:primary:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Regenerate All Documentation](${deep_link_1.stateMachineUrl(props.orchestration.regenerateAllDocumentation)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:primary:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                ...(props.packageStats ? renderPackageStatsWidgets(props.packageStats) : []),
            ],
        });
    }
    *catalogOverviewLanguageSections({ inventory, orchestration, packageData }) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Documentation Generation',
                    '',
                    `[button:primary:Transliterator Logs](${deep_link_1.logGroupUrl(orchestration.transliterator.logGroup)})`,
                    `[button:Transliterator ECS Cluster](${deep_link_1.ecsClusterUrl(orchestration.ecsCluster)})`,
                ].join('\n'),
            }),
        ];
        const mFargateUsage = new aws_cloudwatch_1.Metric({
            dimensionsMap: {
                Class: 'None',
                Resource: 'OnDemand',
                Service: 'Fargate',
                Type: 'Resource',
            },
            metricName: 'ResourceCount',
            namespace: 'AWS/Usage',
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
        });
        yield [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Fargate Resources',
                left: [
                    mFargateUsage.with({ label: 'Fargate Usage (On-Demand)' }),
                    new aws_cloudwatch_1.MathExpression({
                        expression: 'SERVICE_QUOTA(mFargateUsage)',
                        label: 'Fargate Quota (On-Demand)',
                        usingMetrics: { mFargateUsage },
                    }),
                ],
                leftYAxis: { min: 0 },
                right: [
                    orchestration.metricEcsCpuUtilization({ label: 'CPU Utilization' }),
                    orchestration.metricEcsMemoryUtilization({ label: 'Memory Utilization' }),
                ],
                rightYAxis: { label: 'Percent', min: 0, max: 100, showUnits: false },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'ECS Resources',
                left: [
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkRxBytes({ label: 'Received Bytes' })),
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkTxBytes({ label: 'Transmitted Bytes' })),
                ],
                leftYAxis: { min: 0 },
                right: [
                    metric_utils_1.fillMetric(orchestration.metricEcsTaskCount({ label: 'Task Count' })),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new missing_documentation_widget_1.MissingDocumentationWidget(this, `MissingDocs-${language.name}`, {
                    bucket: packageData,
                    height: 6,
                    language,
                    title: 'Package Versions missing documentation',
                    width: 12,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
function renderPackageStatsWidgets(packageStats) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Package Stats',
                    '',
                    `[button:primary:Package Stats Object](${deep_link_1.s3ObjectUrl(packageStats.bucket, packageStats.statsKey)})`,
                    `[button:Package Stats Function](${deep_link_1.lambdaFunctionUrl(packageStats.handler)})`,
                    `[button:Package Stats Logs](${deep_link_1.lambdaSearchLogGroupUrl(packageStats.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Stats Recorded',
                left: [
                    metric_utils_1.fillMetric(packageStats.metricPackagesCount({ label: 'Packages with stats' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    packageStats.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '15 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(15).toSeconds(),
                    }],
            }),
        ],
    ];
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFja2VuZC1kYXNoYm9hcmQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvYmFja2VuZC1kYXNoYm9hcmQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNERBQTBKO0FBRTFKLHdDQUFvRDtBQUlwRCxtR0FBOEY7QUFHOUYsd0RBQWtFO0FBQ2xFLDJDQUFnSjtBQUNoSixpREFBNEM7QUFjNUMsTUFBYSxnQkFBaUIsU0FBUSxnQkFBUztJQUM3QyxZQUFtQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE0QjtRQUMzRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksMEJBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzlCLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxjQUFjLEVBQUUsK0JBQWMsQ0FBQyxJQUFJO1lBQ25DLEtBQUssRUFBRSxNQUFNO1lBQ2IsT0FBTyxFQUFFO2dCQUNQO29CQUNFLElBQUksMkJBQVUsQ0FBQzt3QkFDYixNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxRQUFRLEVBQUU7NEJBQ1Isb0JBQW9COzRCQUNwQixFQUFFOzRCQUNGLGlDQUFpQyx1QkFBVyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsR0FBRzs0QkFDbEUsNEJBQTRCLDZCQUFpQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxHQUFHOzRCQUM3Riw2QkFBNkIsNkJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsR0FBRzs0QkFDM0Usb0NBQW9DLG1DQUF1QixDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLEdBQUc7eUJBQ3pGLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztxQkFDYixDQUFDO2lCQUNIO2dCQUNEO29CQUNFLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsY0FBYzt3QkFDckIsSUFBSSxFQUFFOzRCQUNKLEtBQUssQ0FBQyxTQUFTLENBQUMsb0JBQW9CLENBQUMsRUFBRSxLQUFLLEVBQUUsWUFBWSxFQUFFLENBQUM7NEJBQzdELEtBQUssQ0FBQyxTQUFTLENBQUMseUJBQXlCLENBQUMsRUFBRSxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQzs0QkFDeEUsS0FBSyxDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxDQUFDOzRCQUNwRSxLQUFLLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxDQUFDO3lCQUMxRDt3QkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3FCQUN0QixDQUFDO29CQUNGLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsZ0JBQWdCO3dCQUN2QixJQUFJLEVBQUU7NEJBQ0osS0FBSyxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQzs0QkFDOUQsS0FBSyxDQUFDLFNBQVMsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxDQUFDOzRCQUN6RSxLQUFLLENBQUMsU0FBUyxDQUFDLGlDQUFpQyxDQUFDLEVBQUUsS0FBSyxFQUFFLGtCQUFrQixFQUFFLENBQUM7NEJBQ2hGLEtBQUssQ0FBQyxTQUFTLENBQUMsZ0NBQWdDLENBQUMsRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQzt5QkFDL0U7d0JBQ0QsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTt3QkFDckIsS0FBSyxFQUFFOzRCQUNMLEtBQUssQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLG9DQUFvQyxDQUFDLEVBQUUsS0FBSyxFQUFFLHdCQUF3QixFQUFFLENBQUM7NEJBQzVHLEtBQUssQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsS0FBSyxFQUFFLGdDQUFnQyxFQUFFLENBQUM7eUJBQzVIO3dCQUNELFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7cUJBQ3ZCLENBQUM7aUJBQ0g7Z0JBQ0QsR0FBRyxJQUFJLENBQUMsK0JBQStCLENBQUMsS0FBSyxDQUFDO2dCQUM5QyxHQUFHLDJCQUEyQixDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7Z0JBQ3BEO29CQUNFLElBQUksMkJBQVUsQ0FBQzt3QkFDYixNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxRQUFRLEVBQUU7NEJBQ1Isc0JBQXNCOzRCQUN0QixFQUFFOzRCQUNGLCtCQUErQiw2QkFBaUIsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxHQUFHOzRCQUM3RSxxQ0FBcUMsbUNBQXVCLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsR0FBRzs0QkFDekYsZ0JBQWdCLHVCQUFXLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsR0FBRzt5QkFDaEUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO3FCQUNiLENBQUM7aUJBQ0g7Z0JBQ0Q7b0JBQ0UsSUFBSSw0QkFBVyxDQUFDO3dCQUNkLE1BQU0sRUFBRSxDQUFDO3dCQUNULEtBQUssRUFBRSxFQUFFO3dCQUNULEtBQUssRUFBRSxpQkFBaUI7d0JBQ3hCLElBQUksRUFBRTs0QkFDSix5QkFBVSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7NEJBQ2hGLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7eUJBQ3ZFO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztxQkFDNUIsQ0FBQztvQkFDRixJQUFJLDRCQUFXLENBQUM7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLGFBQWE7d0JBQ3BCLElBQUksRUFBRTs0QkFDSixLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyx3Q0FBd0MsQ0FBQyxFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDOzRCQUMxSCxLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQywyQ0FBMkMsQ0FBQyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO3lCQUM3SDt3QkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3dCQUNyQixLQUFLLEVBQUU7NEJBQ0wsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsbUNBQW1DLENBQUMsRUFBRSxLQUFLLEVBQUUsb0JBQW9CLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzt5QkFDeEg7d0JBQ0QsZ0JBQWdCLEVBQUUsQ0FBQztnQ0FDakIsS0FBSyxFQUFFLFNBQVM7Z0NBQ2hCLEtBQUssRUFBRSxZQUFZO2dDQUNuQixLQUFLLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxTQUFTLEVBQUU7NkJBQ3hDLENBQUM7d0JBQ0YsVUFBVSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTt3QkFDdEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO3FCQUM1QixDQUFDO2lCQUNIO2dCQUNEO29CQUNFLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsZUFBZTt3QkFDdEIsSUFBSSxFQUFFOzRCQUNKLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDLENBQUM7NEJBQ2xGLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7NEJBQzlFLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDLENBQUM7NEJBQ3BGLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFLEtBQUssRUFBRSxxQkFBcUIsRUFBRSxDQUFDLENBQUM7NEJBQ2hHLHlCQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDLENBQUM7eUJBQ3BGO3dCQUNELFNBQVMsRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFO3dCQUN2RCxPQUFPLEVBQUUsSUFBSTtxQkFDZCxDQUFDO29CQUNGLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsY0FBYzt3QkFDckIsSUFBSSxFQUFFOzRCQUNKLEtBQUssQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLHdDQUF3QyxDQUFDLEVBQUUsS0FBSyxFQUFFLGtCQUFrQixFQUFFLENBQUM7NEJBQ3ZHLEtBQUssQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsS0FBSyxFQUFFLG9CQUFvQixFQUFFLENBQUM7eUJBQzdHO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLEtBQUssRUFBRTs0QkFDTCxLQUFLLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBQyxtQ0FBbUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDO3lCQUNyRzt3QkFDRCxnQkFBZ0IsRUFBRSxDQUFDO2dDQUNqQixLQUFLLEVBQUUsU0FBUztnQ0FDaEIsS0FBSyxFQUFFLFNBQVM7Z0NBQ2hCLEtBQUssRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsRUFBRTs2QkFDckMsRUFBRTtnQ0FDRCxLQUFLLEVBQUUsU0FBUztnQ0FDaEIsS0FBSyxFQUFFLHlCQUF5QjtnQ0FDaEMsS0FBSyxFQUFFLGVBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsU0FBUyxFQUFFOzZCQUNyQyxDQUFDO3dCQUNGLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3RCLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztxQkFDNUIsQ0FBQztpQkFDSDtnQkFDRDtvQkFDRSxJQUFJLDJCQUFVLENBQUM7d0JBQ2IsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsUUFBUSxFQUNOOzRCQUNFLGlCQUFpQjs0QkFDakIsRUFBRTs0QkFDRixrQ0FBa0MsMkJBQWUsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxHQUFHOzRCQUN0RixnQkFBZ0IsdUJBQVcsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxHQUFHOzRCQUNuRSx3QkFBd0IsNkJBQWlCLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsR0FBRzs0QkFDakYseUNBQXlDLDJCQUFlLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQywwQkFBMEIsQ0FBQyxHQUFHO3lCQUM1RyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7cUJBQ2YsQ0FBQztpQkFDSDtnQkFDRDtvQkFDRSxJQUFJLDRCQUFXLENBQUM7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLDBCQUEwQjt3QkFDakMsSUFBSSxFQUFFOzRCQUNKLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7NEJBQ2hGLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7NEJBQ3BGLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7NEJBQ2hGLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7NEJBQzlFLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7NEJBQ3BGLHlCQUFVLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7eUJBQ3BGO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLEtBQUssRUFBRTs0QkFDTCxLQUFLLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLENBQUM7eUJBQ25FO3dCQUNELFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7cUJBQ3ZCLENBQUM7b0JBQ0YsSUFBSSw0QkFBVyxDQUFDO3dCQUNkLE1BQU0sRUFBRSxDQUFDO3dCQUNULEtBQUssRUFBRSxFQUFFO3dCQUNULEtBQUssRUFBRSxtQkFBbUI7d0JBQzFCLElBQUksRUFBRTs0QkFDSixLQUFLLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyx3Q0FBd0MsQ0FBQyxFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxDQUFDOzRCQUMzRyxLQUFLLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQywyQ0FBMkMsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDO3lCQUNqSDt3QkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3dCQUNyQixLQUFLLEVBQUU7NEJBQ0wsS0FBSyxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsbUNBQW1DLENBQUMsRUFBRSxLQUFLLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQzt5QkFDekc7d0JBQ0QsZ0JBQWdCLEVBQUUsQ0FBQztnQ0FDakIsS0FBSyxFQUFFLFNBQVM7Z0NBQ2hCLEtBQUssRUFBRSxTQUFTO2dDQUNoQixLQUFLLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxTQUFTLEVBQUU7NkJBQ3JDLEVBQUU7Z0NBQ0QsS0FBSyxFQUFFLFNBQVM7Z0NBQ2hCLEtBQUssRUFBRSx5QkFBeUI7Z0NBQ2hDLEtBQUssRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsRUFBRTs2QkFDckMsQ0FBQzt3QkFDRixVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3dCQUN0QixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7cUJBQzVCLENBQUM7aUJBQ0g7Z0JBRUQsWUFBWTtnQkFDWixpREFBaUQ7Z0JBQ2pEO29CQUNFLElBQUksMkJBQVUsQ0FBQzt3QkFDYixNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxRQUFRLEVBQ047NEJBQ0UsYUFBYTs0QkFDYixFQUFFOzRCQUNGLHFDQUFxQyx1QkFBVyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLEdBQUc7NEJBQ3BHLDJCQUEyQiw2QkFBaUIsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRzs0QkFDbEYsdUJBQXVCLG1DQUF1QixDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHOzRCQUNwRix5QkFBeUIsdUJBQVcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsR0FBRzs0QkFDbkUsd0JBQXdCLG1DQUF1QixDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxHQUFHO3lCQUN2RixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7cUJBQ2YsQ0FBQztpQkFDSDtnQkFDRDtvQkFDRSxJQUFJLDRCQUFXLENBQUM7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLFdBQVc7d0JBQ2xCLElBQUksRUFBRTs0QkFDSix5QkFBVSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLENBQUMsRUFBRSxRQUFRLENBQUM7NEJBQzVFLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxFQUFFLEtBQUssRUFBRSxlQUFlLEVBQUUsQ0FBQzt5QkFDckY7d0JBQ0QsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTt3QkFDckIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO3FCQUM1QixDQUFDO29CQUNGLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsdUJBQXVCO3dCQUM5QixJQUFJLEVBQUU7NEJBQ0oseUJBQVUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxLQUFLLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQzs0QkFDekYseUJBQVUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7eUJBQ2hGO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztxQkFDNUIsQ0FBQztpQkFDSDtnQkFFRCxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDN0U7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sQ0FBQywrQkFBK0IsQ0FBQyxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsV0FBVyxFQUF5QjtRQUN2RyxNQUFNO1lBQ0osSUFBSSwyQkFBVSxDQUFDO2dCQUNiLE1BQU0sRUFBRSxDQUFDO2dCQUNULEtBQUssRUFBRSxFQUFFO2dCQUNULFFBQVEsRUFBRTtvQkFDUiw0QkFBNEI7b0JBQzVCLEVBQUU7b0JBQ0Ysd0NBQXdDLHVCQUFXLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsR0FBRztvQkFDN0YsdUNBQXVDLHlCQUFhLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxHQUFHO2lCQUNsRixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7YUFDYixDQUFDO1NBQ0gsQ0FBQztRQUNGLE1BQU0sYUFBYSxHQUFHLElBQUksdUJBQU0sQ0FBQztZQUMvQixhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLE1BQU07Z0JBQ2IsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLE9BQU8sRUFBRSxTQUFTO2dCQUNsQixJQUFJLEVBQUUsVUFBVTthQUNqQjtZQUNELFVBQVUsRUFBRSxlQUFlO1lBQzNCLFNBQVMsRUFBRSxXQUFXO1lBQ3RCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87U0FDN0IsQ0FBQyxDQUFDO1FBRUgsTUFBTTtZQUNKLElBQUksNEJBQVcsQ0FBQztnQkFDZCxNQUFNLEVBQUUsQ0FBQztnQkFDVCxLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsbUJBQW1CO2dCQUMxQixJQUFJLEVBQUU7b0JBQ0osYUFBYSxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSwyQkFBMkIsRUFBRSxDQUFDO29CQUMxRCxJQUFJLCtCQUFjLENBQUM7d0JBQ2pCLFVBQVUsRUFBRSw4QkFBOEI7d0JBQzFDLEtBQUssRUFBRSwyQkFBMkI7d0JBQ2xDLFlBQVksRUFBRSxFQUFFLGFBQWEsRUFBRTtxQkFDaEMsQ0FBQztpQkFDSDtnQkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO2dCQUNyQixLQUFLLEVBQUU7b0JBQ0wsYUFBYSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLENBQUM7b0JBQ25FLGFBQWEsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxDQUFDO2lCQUMxRTtnQkFDRCxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFO2FBQ3JFLENBQUM7WUFDRixJQUFJLDRCQUFXLENBQUM7Z0JBQ2QsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLGVBQWU7Z0JBQ3RCLElBQUksRUFBRTtvQkFDSix5QkFBVSxDQUFDLGFBQWEsQ0FBQyx1QkFBdUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7b0JBQzlFLHlCQUFVLENBQUMsYUFBYSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsS0FBSyxFQUFFLG1CQUFtQixFQUFFLENBQUMsQ0FBQztpQkFDbEY7Z0JBQ0QsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTtnQkFDckIsS0FBSyxFQUFFO29CQUNMLHlCQUFVLENBQUMsYUFBYSxDQUFDLGtCQUFrQixDQUFDLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUM7aUJBQ3RFO2dCQUNELFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7YUFDdkIsQ0FBQztTQUNILENBQUM7UUFFRixLQUFLLE1BQU0sUUFBUSxJQUFJLGdDQUFxQixDQUFDLEdBQUcsRUFBRTtZQUNoRCxNQUFNO2dCQUNKLElBQUksMkJBQVUsQ0FBQztvQkFDYixNQUFNLEVBQUUsQ0FBQztvQkFDVCxLQUFLLEVBQUUsRUFBRTtvQkFDVCxRQUFRLEVBQUUsZ0JBQWdCLFFBQVEsQ0FBQyxRQUFRLEVBQUUsRUFBRTtpQkFDaEQsQ0FBQzthQUNILENBQUM7WUFDRixNQUFNO2dCQUNKLElBQUksNEJBQVcsQ0FBQztvQkFDZCxNQUFNLEVBQUUsQ0FBQztvQkFDVCxLQUFLLEVBQUUsQ0FBQztvQkFDUixLQUFLLEVBQUUsa0JBQWtCO29CQUN6QixJQUFJLEVBQUU7d0JBQ0osU0FBUyxDQUFDLGtDQUFrQyxDQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDO3dCQUNoRyxTQUFTLENBQUMsb0NBQW9DLENBQUMsUUFBUSxFQUFFLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLENBQUM7d0JBQ3BHLFNBQVMsQ0FBQyxnQ0FBZ0MsQ0FBQyxRQUFRLEVBQUUsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQztxQkFDN0Y7b0JBQ0QsU0FBUyxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRTtvQkFDL0IsSUFBSSxFQUFFLGdDQUFlLENBQUMsR0FBRztpQkFDMUIsQ0FBQztnQkFDRixJQUFJLHlEQUEwQixDQUFDLElBQUksRUFBRSxlQUFlLFFBQVEsQ0FBQyxJQUFJLEVBQUUsRUFBRTtvQkFDbkUsTUFBTSxFQUFFLFdBQVc7b0JBQ25CLE1BQU0sRUFBRSxDQUFDO29CQUNULFFBQVE7b0JBQ1IsS0FBSyxFQUFFLHdDQUF3QztvQkFDL0MsS0FBSyxFQUFFLEVBQUU7aUJBQ1YsQ0FBQztnQkFDRixJQUFJLDRCQUFXLENBQUM7b0JBQ2QsTUFBTSxFQUFFLENBQUM7b0JBQ1QsS0FBSyxFQUFFLENBQUM7b0JBQ1IsS0FBSyxFQUFFLDRCQUE0QjtvQkFDbkMsSUFBSSxFQUFFO3dCQUNKLFNBQVMsQ0FBQyw2QkFBNkIsQ0FBQyxRQUFRLEVBQUUsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQzt3QkFDM0YsU0FBUyxDQUFDLCtCQUErQixDQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxhQUFhLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDO3dCQUMvRixTQUFTLENBQUMsMkJBQTJCLENBQUMsUUFBUSxFQUFFLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLENBQUM7cUJBQ3hGO29CQUNELFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUU7b0JBQy9CLElBQUksRUFBRSxnQ0FBZSxDQUFDLEdBQUc7aUJBQzFCLENBQUM7YUFDSCxDQUFDO1NBQ0g7SUFDSCxDQUFDO0NBQ0Y7QUFsV0QsNENBa1dDO0FBRUQsUUFBUSxDQUFDLENBQUMsMkJBQTJCLENBQUMsY0FBeUM7O0lBQzdFLEtBQUssTUFBTSxhQUFhLElBQUksY0FBYyxFQUFFO1FBQzFDLE1BQU07WUFDSixJQUFJLDJCQUFVLENBQUM7Z0JBQ2IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsUUFBUSxFQUFFO29CQUNSLEtBQUssYUFBYSxDQUFDLElBQUksRUFBRTtvQkFDekIsRUFBRTtvQkFDRixHQUFHLE9BQUMsYUFBYSxDQUFDLEtBQUssbUNBQUksRUFBRSxDQUFDO3lCQUMzQixHQUFHLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLEVBQUUsRUFBRSxDQUFDLElBQUksT0FBTyxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsUUFBUSxJQUFJLElBQUksS0FBSyxHQUFHLEdBQUcsQ0FBQztpQkFDakcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO2FBQ2IsQ0FBQztTQUNILENBQUM7UUFDRixLQUFLLENBQUMsQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUM7S0FDdkM7SUFDRCxPQUFPO0FBQ1QsQ0FBQztBQUVELFNBQVMseUJBQXlCLENBQUMsWUFBMEI7SUFDM0QsT0FBTztRQUNMO1lBQ0UsSUFBSSwyQkFBVSxDQUFDO2dCQUNiLE1BQU0sRUFBRSxDQUFDO2dCQUNULEtBQUssRUFBRSxFQUFFO2dCQUNULFFBQVEsRUFDTjtvQkFDRSxpQkFBaUI7b0JBQ2pCLEVBQUU7b0JBQ0YseUNBQXlDLHVCQUFXLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsUUFBUSxDQUFDLEdBQUc7b0JBQ25HLG1DQUFtQyw2QkFBaUIsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEdBQUc7b0JBQzdFLCtCQUErQixtQ0FBdUIsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEdBQUc7aUJBQ2hGLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQzthQUNmLENBQUM7U0FDSDtRQUNEO1lBQ0UsSUFBSSw0QkFBVyxDQUFDO2dCQUNkLE1BQU0sRUFBRSxDQUFDO2dCQUNULEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSxrQ0FBa0M7Z0JBQ3pDLElBQUksRUFBRTtvQkFDSix5QkFBVSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxxQkFBcUIsRUFBRSxDQUFDLEVBQUUsUUFBUSxDQUFDO2lCQUN6RjtnQkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO2FBQ3RCLENBQUM7WUFDRixJQUFJLDRCQUFXLENBQUM7Z0JBQ2QsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLHFCQUFxQjtnQkFDNUIsSUFBSSxFQUFFO29CQUNKLFlBQVksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxDQUFDO2lCQUMzRDtnQkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO2dCQUNyQixnQkFBZ0IsRUFBRSxDQUFDO3dCQUNqQixLQUFLLEVBQUUsU0FBUzt3QkFDaEIsS0FBSyxFQUFFLDZCQUE2Qjt3QkFDcEMsS0FBSyxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsU0FBUyxFQUFFO3FCQUN4QyxDQUFDO2FBQ0gsQ0FBQztTQUNIO0tBQ0YsQ0FBQztBQUNKLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBEYXNoYm9hcmQsIEdyYXBoV2lkZ2V0LCBHcmFwaFdpZGdldFZpZXcsIFRleHRXaWRnZXQsIElXaWRnZXQsIE1hdGhFeHByZXNzaW9uLCBNZXRyaWMsIFN0YXRpc3RpYywgUGVyaW9kT3ZlcnJpZGUgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IERlbnlMaXN0IH0gZnJvbSAnLi9iYWNrZW5kL2RlbnktbGlzdCc7XG5pbXBvcnQgeyBJbmdlc3Rpb24gfSBmcm9tICcuL2JhY2tlbmQvaW5nZXN0aW9uJztcbmltcG9ydCB7IEludmVudG9yeSB9IGZyb20gJy4vYmFja2VuZC9pbnZlbnRvcnknO1xuaW1wb3J0IHsgTWlzc2luZ0RvY3VtZW50YXRpb25XaWRnZXQgfSBmcm9tICcuL2JhY2tlbmQvaW52ZW50b3J5L21pc3NpbmctZG9jdW1lbnRhdGlvbi13aWRnZXQnO1xuaW1wb3J0IHsgT3JjaGVzdHJhdGlvbiB9IGZyb20gJy4vYmFja2VuZC9vcmNoZXN0cmF0aW9uJztcbmltcG9ydCB7IFBhY2thZ2VTdGF0cyB9IGZyb20gJy4vYmFja2VuZC9wYWNrYWdlLXN0YXRzJztcbmltcG9ydCB7IERvY3VtZW50YXRpb25MYW5ndWFnZSB9IGZyb20gJy4vYmFja2VuZC9zaGFyZWQvbGFuZ3VhZ2UnO1xuaW1wb3J0IHsgZWNzQ2x1c3RlclVybCwgbGFtYmRhRnVuY3Rpb25VcmwsIGxhbWJkYVNlYXJjaExvZ0dyb3VwVXJsLCBsb2dHcm91cFVybCwgczNPYmplY3RVcmwsIHNxc1F1ZXVlVXJsLCBzdGF0ZU1hY2hpbmVVcmwgfSBmcm9tICcuL2RlZXAtbGluayc7XG5pbXBvcnQgeyBmaWxsTWV0cmljIH0gZnJvbSAnLi9tZXRyaWMtdXRpbHMnO1xuaW1wb3J0IHsgUGFja2FnZVNvdXJjZUJpbmRSZXN1bHQgfSBmcm9tICcuL3BhY2thZ2Utc291cmNlJztcblxuZXhwb3J0IGludGVyZmFjZSBCYWNrZW5kRGFzaGJvYXJkUHJvcHMge1xuICByZWFkb25seSBkYXNoYm9hcmROYW1lPzogc3RyaW5nO1xuICByZWFkb25seSBwYWNrYWdlU291cmNlczogUGFja2FnZVNvdXJjZUJpbmRSZXN1bHRbXTtcbiAgcmVhZG9ubHkgaW5nZXN0aW9uOiBJbmdlc3Rpb247XG4gIHJlYWRvbmx5IG9yY2hlc3RyYXRpb246IE9yY2hlc3RyYXRpb247XG4gIHJlYWRvbmx5IGludmVudG9yeTogSW52ZW50b3J5O1xuICByZWFkb25seSBkZW55TGlzdDogRGVueUxpc3Q7XG4gIHJlYWRvbmx5IHBhY2thZ2VEYXRhOiBJQnVja2V0O1xuICByZWFkb25seSBwYWNrYWdlU3RhdHM/OiBQYWNrYWdlU3RhdHM7XG59XG5cbmV4cG9ydCBjbGFzcyBCYWNrZW5kRGFzaGJvYXJkIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBCYWNrZW5kRGFzaGJvYXJkUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbmV3IERhc2hib2FyZCh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkYXNoYm9hcmROYW1lOiBwcm9wcy5kYXNoYm9hcmROYW1lLFxuICAgICAgcGVyaW9kT3ZlcnJpZGU6IFBlcmlvZE92ZXJyaWRlLkFVVE8sXG4gICAgICBzdGFydDogJy1QMVcnLCAvLyBTaG93IDEgd2VlayBieSBkZWZhdWx0XG4gICAgICB3aWRnZXRzOiBbXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgVGV4dFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgICAgICB3aWR0aDogMjQsXG4gICAgICAgICAgICBtYXJrZG93bjogW1xuICAgICAgICAgICAgICAnIyBDYXRhbG9nIE92ZXJ2aWV3JyxcbiAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgIGBbYnV0dG9uOnByaW1hcnk6UGFja2FnZSBEYXRhXSgke3MzT2JqZWN0VXJsKHByb3BzLnBhY2thZ2VEYXRhKX0pYCxcbiAgICAgICAgICAgICAgYFtidXR0b246Q2F0YWxvZyBCdWlsZGVyXSgke2xhbWJkYUZ1bmN0aW9uVXJsKHByb3BzLm9yY2hlc3RyYXRpb24uY2F0YWxvZ0J1aWxkZXIuZnVuY3Rpb24pfSlgLFxuICAgICAgICAgICAgICBgW2J1dHRvbjpJbnZlbnRvcnkgQ2FuYXJ5XSgke2xhbWJkYUZ1bmN0aW9uVXJsKHByb3BzLmludmVudG9yeS5mdW5jdGlvbil9KWAsXG4gICAgICAgICAgICAgIGBbYnV0dG9uOlNlYXJjaCBDYW5hcnkgTG9nIEdyb3VwXSgke2xhbWJkYVNlYXJjaExvZ0dyb3VwVXJsKHByb3BzLmludmVudG9yeS5mdW5jdGlvbil9KWAsXG4gICAgICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnQ2F0YWxvZyBTaXplJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgcHJvcHMuaW52ZW50b3J5Lm1ldHJpY1N1Ym1vZHVsZUNvdW50KHsgbGFiZWw6ICdTdWJtb2R1bGVzJyB9KSxcbiAgICAgICAgICAgICAgcHJvcHMuaW52ZW50b3J5Lm1ldHJpY1BhY2thZ2VWZXJzaW9uQ291bnQoeyBsYWJlbDogJ1BhY2thZ2UgVmVyc2lvbnMnIH0pLFxuICAgICAgICAgICAgICBwcm9wcy5pbnZlbnRvcnkubWV0cmljUGFja2FnZU1ham9yQ291bnQoeyBsYWJlbDogJ1BhY2thZ2UgTWFqb3JzJyB9KSxcbiAgICAgICAgICAgICAgcHJvcHMuaW52ZW50b3J5Lm1ldHJpY1BhY2thZ2VDb3VudCh7IGxhYmVsOiAnUGFja2FnZXMnIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGxlZnRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICAgICAgdGl0bGU6ICdDYXRhbG9nIElzc3VlcycsXG4gICAgICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgICAgIHByb3BzLmludmVudG9yeS5tZXRyaWNVbmtub3duT2JqZWN0Q291bnQoeyBsYWJlbDogJ1Vua25vd24nIH0pLFxuICAgICAgICAgICAgICBwcm9wcy5pbnZlbnRvcnkubWV0cmljTWlzc2luZ0Fzc2VtYmx5Q291bnQoeyBsYWJlbDogJ01pc3NpbmcgQXNzZW1ibHknIH0pLFxuICAgICAgICAgICAgICBwcm9wcy5pbnZlbnRvcnkubWV0cmljTWlzc2luZ1BhY2thZ2VNZXRhZGF0YUNvdW50KHsgbGFiZWw6ICdNaXNzaW5nIE1ldGFkYXRhJyB9KSxcbiAgICAgICAgICAgICAgcHJvcHMuaW52ZW50b3J5Lm1ldHJpY01pc3NpbmdQYWNrYWdlVGFyYmFsbENvdW50KHsgbGFiZWw6ICdNaXNzaW5nIFRhcmJhbGwnIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGxlZnRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICAgIHJpZ2h0OiBbXG4gICAgICAgICAgICAgIHByb3BzLm9yY2hlc3RyYXRpb24uY2F0YWxvZ0J1aWxkZXIubWV0cmljTWlzc2luZ0NvbnN0cnVjdEZyYW1ld29ya0NvdW50KHsgbGFiZWw6ICdObyBDb25zdHJ1Y3QgRnJhbWV3b3JrJyB9KSxcbiAgICAgICAgICAgICAgcHJvcHMub3JjaGVzdHJhdGlvbi5jYXRhbG9nQnVpbGRlci5tZXRyaWNNaXNzaW5nQ29uc3RydWN0RnJhbWV3b3JrVmVyc2lvbkNvdW50KHsgbGFiZWw6ICdObyBDb25zdHJ1Y3QgRnJhbWV3b3JrIFZlcnNpb24nIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJpZ2h0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICAgIC4uLnRoaXMuY2F0YWxvZ092ZXJ2aWV3TGFuZ3VhZ2VTZWN0aW9ucyhwcm9wcyksXG4gICAgICAgIC4uLnJlbmRlclBhY2thZ2VTb3VyY2VzV2lkZ2V0cyhwcm9wcy5wYWNrYWdlU291cmNlcyksXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgVGV4dFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgICAgICB3aWR0aDogMjQsXG4gICAgICAgICAgICBtYXJrZG93bjogW1xuICAgICAgICAgICAgICAnIyBJbmdlc3Rpb24gRnVuY3Rpb24nLFxuICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgYFtidXR0b246SW5nZXN0aW9uIEZ1bmN0aW9uXSgke2xhbWJkYUZ1bmN0aW9uVXJsKHByb3BzLmluZ2VzdGlvbi5mdW5jdGlvbil9KWAsXG4gICAgICAgICAgICAgIGBbYnV0dG9uOnByaW1hcnk6U2VhcmNoIExvZyBHcm91cF0oJHtsYW1iZGFTZWFyY2hMb2dHcm91cFVybChwcm9wcy5pbmdlc3Rpb24uZnVuY3Rpb24pfSlgLFxuICAgICAgICAgICAgICBgW2J1dHRvbjpETFFdKCR7c3FzUXVldWVVcmwocHJvcHMuaW5nZXN0aW9uLmRlYWRMZXR0ZXJRdWV1ZSl9KWAsXG4gICAgICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnRnVuY3Rpb24gSGVhbHRoJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5pbmdlc3Rpb24uZnVuY3Rpb24ubWV0cmljSW52b2NhdGlvbnMoeyBsYWJlbDogJ0ludm9jYXRpb25zJyB9KSksXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMocHJvcHMuaW5nZXN0aW9uLmZ1bmN0aW9uLm1ldHJpY0Vycm9ycyh7IGxhYmVsOiAnRXJyb3JzJyB9KSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgbGVmdFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDYsXG4gICAgICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgICAgICB0aXRsZTogJ0lucHV0IFF1ZXVlJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgcHJvcHMuaW5nZXN0aW9uLnF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc1Zpc2libGUoeyBsYWJlbDogJ1Zpc2libGUgTWVzc2FnZXMnLCBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgICAgICAgIHByb3BzLmluZ2VzdGlvbi5xdWV1ZS5tZXRyaWNBcHByb3hpbWF0ZU51bWJlck9mTWVzc2FnZXNOb3RWaXNpYmxlKHsgbGFiZWw6ICdIaWRkZW4gTWVzc2FnZXMnLCBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgbGVmdFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcmlnaHQ6IFtcbiAgICAgICAgICAgICAgcHJvcHMuaW5nZXN0aW9uLnF1ZXVlLm1ldHJpY0FwcHJveGltYXRlQWdlT2ZPbGRlc3RNZXNzYWdlKHsgbGFiZWw6ICdPbGRlc3QgTWVzc2FnZSBBZ2UnLCBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcmlnaHRBbm5vdGF0aW9uczogW3tcbiAgICAgICAgICAgICAgY29sb3I6ICcjZmZhNTAwJyxcbiAgICAgICAgICAgICAgbGFiZWw6ICcxMCBNaW51dGVzJyxcbiAgICAgICAgICAgICAgdmFsdWU6IER1cmF0aW9uLm1pbnV0ZXMoMTApLnRvU2Vjb25kcygpLFxuICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICByaWdodFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnSW5wdXQgUXVhbGl0eScsXG4gICAgICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMocHJvcHMuaW5nZXN0aW9uLm1ldHJpY0ludmFsaWRBc3NlbWJseSh7IGxhYmVsOiAnSW52YWxpZCBBc3NlbWJsaWVzJyB9KSksXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMocHJvcHMuaW5nZXN0aW9uLm1ldHJpY0ludmFsaWRUYXJiYWxsKHsgbGFiZWw6ICdJbnZhbGlkIFRhcmJhbGwnIH0pKSxcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5pbmdlc3Rpb24ubWV0cmljSW5lbGlnaWJsZUxpY2Vuc2UoeyBsYWJlbDogJ0luZWxpZ2libGUgTGljZW5zZScgfSkpLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHByb3BzLmluZ2VzdGlvbi5tZXRyaWNNaXNtYXRjaGVkSWRlbnRpdHlSZWplY3Rpb25zKHsgbGFiZWw6ICdNaXNtYXRjaGVkIElkZW50aXR5JyB9KSksXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMocHJvcHMuaW5nZXN0aW9uLm1ldHJpY0ZvdW5kTGljZW5zZUZpbGUoeyBsYWJlbDogJ0ZvdW5kIExpY2Vuc2UgZmlsZScgfSkpLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGxlZnRZQXhpczogeyBsYWJlbDogJ0NvdW50JywgbWluOiAwLCBzaG93VW5pdHM6IGZhbHNlIH0sXG4gICAgICAgICAgICBzdGFja2VkOiB0cnVlLFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDYsXG4gICAgICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgICAgICB0aXRsZTogJ0RlYWQgTGV0dGVycycsXG4gICAgICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgICAgIHByb3BzLmluZ2VzdGlvbi5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSh7IGxhYmVsOiAnVmlzaWJsZSBNZXNzYWdlcycgfSksXG4gICAgICAgICAgICAgIHByb3BzLmluZ2VzdGlvbi5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzTm90VmlzaWJsZSh7IGxhYmVsOiAnSW52aXNpYmxlIE1lc3NhZ2VzJyB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgICAgICByaWdodDogW1xuICAgICAgICAgICAgICBwcm9wcy5pbmdlc3Rpb24uZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlQWdlT2ZPbGRlc3RNZXNzYWdlKHsgbGFiZWw6ICdPbGRlc3QgTWVzc2FnZSBBZ2UnIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJpZ2h0QW5ub3RhdGlvbnM6IFt7XG4gICAgICAgICAgICAgIGNvbG9yOiAnI2ZmN2YwZScsXG4gICAgICAgICAgICAgIGxhYmVsOiAnMTAgZGF5cycsXG4gICAgICAgICAgICAgIHZhbHVlOiBEdXJhdGlvbi5kYXlzKDEwKS50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIH0sIHtcbiAgICAgICAgICAgICAgY29sb3I6ICcjZmYwMDAwJyxcbiAgICAgICAgICAgICAgbGFiZWw6ICcxNCBkYXlzIChETFEgUmV0ZW50aW9uKScsXG4gICAgICAgICAgICAgIHZhbHVlOiBEdXJhdGlvbi5kYXlzKDE0KS50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIH1dLFxuICAgICAgICAgICAgcmlnaHRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgICAgW1xuICAgICAgICAgIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogMixcbiAgICAgICAgICAgIHdpZHRoOiAyNCxcbiAgICAgICAgICAgIG1hcmtkb3duOlxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgJyMgT3JjaGVzdHJhdGlvbicsXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgYFtidXR0b246cHJpbWFyeTpTdGF0ZSBNYWNoaW5lXSgke3N0YXRlTWFjaGluZVVybChwcm9wcy5vcmNoZXN0cmF0aW9uLnN0YXRlTWFjaGluZSl9KWAsXG4gICAgICAgICAgICAgICAgYFtidXR0b246RExRXSgke3Nxc1F1ZXVlVXJsKHByb3BzLm9yY2hlc3RyYXRpb24uZGVhZExldHRlclF1ZXVlKX0pYCxcbiAgICAgICAgICAgICAgICBgW2J1dHRvbjpSZWRyaXZlIERMUV0oJHtsYW1iZGFGdW5jdGlvblVybChwcm9wcy5vcmNoZXN0cmF0aW9uLnJlZHJpdmVGdW5jdGlvbil9KWAsXG4gICAgICAgICAgICAgICAgYFtidXR0b246UmVnZW5lcmF0ZSBBbGwgRG9jdW1lbnRhdGlvbl0oJHtzdGF0ZU1hY2hpbmVVcmwocHJvcHMub3JjaGVzdHJhdGlvbi5yZWdlbmVyYXRlQWxsRG9jdW1lbnRhdGlvbil9KWAsXG4gICAgICAgICAgICAgIF0uam9pbignXFxuJyksXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICAgICAgdGl0bGU6ICdTdGF0ZSBNYWNoaW5lIEV4ZWN1dGlvbnMnLFxuICAgICAgICAgICAgbGVmdDogW1xuICAgICAgICAgICAgICBmaWxsTWV0cmljKHByb3BzLm9yY2hlc3RyYXRpb24uc3RhdGVNYWNoaW5lLm1ldHJpY1N0YXJ0ZWQoeyBsYWJlbDogJ1N0YXJ0ZWQnIH0pKSxcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5vcmNoZXN0cmF0aW9uLnN0YXRlTWFjaGluZS5tZXRyaWNTdWNjZWVkZWQoeyBsYWJlbDogJ1N1Y2NlZWRlZCcgfSkpLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHByb3BzLm9yY2hlc3RyYXRpb24uc3RhdGVNYWNoaW5lLm1ldHJpY0Fib3J0ZWQoeyBsYWJlbDogJ0Fib3J0ZWQnIH0pKSxcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5vcmNoZXN0cmF0aW9uLnN0YXRlTWFjaGluZS5tZXRyaWNGYWlsZWQoeyBsYWJlbDogJ0ZhaWxlZCcgfSkpLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHByb3BzLm9yY2hlc3RyYXRpb24uc3RhdGVNYWNoaW5lLm1ldHJpY1Rocm90dGxlZCh7IGxhYmVsOiAnVGhyb3R0bGVkJyB9KSksXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMocHJvcHMub3JjaGVzdHJhdGlvbi5zdGF0ZU1hY2hpbmUubWV0cmljVGltZWRPdXQoeyBsYWJlbDogJ1RpbWVkIE91dCcgfSkpLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGxlZnRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICAgIHJpZ2h0OiBbXG4gICAgICAgICAgICAgIHByb3BzLm9yY2hlc3RyYXRpb24uc3RhdGVNYWNoaW5lLm1ldHJpY1RpbWUoeyBsYWJlbDogJ0R1cmF0aW9uJyB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByaWdodFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDYsXG4gICAgICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgICAgICB0aXRsZTogJ0RlYWQgTGV0dGVyIFF1ZXVlJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgcHJvcHMub3JjaGVzdHJhdGlvbi5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSh7IGxhYmVsOiAnVmlzaWJsZSBNZXNzYWdlcycgfSksXG4gICAgICAgICAgICAgIHByb3BzLm9yY2hlc3RyYXRpb24uZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc05vdFZpc2libGUoeyBsYWJlbDogJ0ludmlzaWJsZSBNZXNzYWdlcycgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgbGVmdFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcmlnaHQ6IFtcbiAgICAgICAgICAgICAgcHJvcHMub3JjaGVzdHJhdGlvbi5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVBZ2VPZk9sZGVzdE1lc3NhZ2UoeyBsYWJlbDogJ09sZGVzdCBNZXNzYWdlIEFnZScgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcmlnaHRBbm5vdGF0aW9uczogW3tcbiAgICAgICAgICAgICAgY29sb3I6ICcjZmY3ZjBlJyxcbiAgICAgICAgICAgICAgbGFiZWw6ICcxMCBkYXlzJyxcbiAgICAgICAgICAgICAgdmFsdWU6IER1cmF0aW9uLmRheXMoMTApLnRvU2Vjb25kcygpLFxuICAgICAgICAgICAgfSwge1xuICAgICAgICAgICAgICBjb2xvcjogJyNmZjAwMDAnLFxuICAgICAgICAgICAgICBsYWJlbDogJzE0IGRheXMgKERMUSBSZXRlbnRpb24pJyxcbiAgICAgICAgICAgICAgdmFsdWU6IER1cmF0aW9uLmRheXMoMTQpLnRvU2Vjb25kcygpLFxuICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICByaWdodFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuXG4gICAgICAgIC8vIGRlbnkgbGlzdFxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgVGV4dFdpZGdldCh7XG4gICAgICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgICAgICB3aWR0aDogMjQsXG4gICAgICAgICAgICBtYXJrZG93bjpcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICcjIERlbnkgTGlzdCcsXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgYFtidXR0b246cHJpbWFyeTpEZW55IExpc3QgT2JqZWN0XSgke3MzT2JqZWN0VXJsKHByb3BzLmRlbnlMaXN0LmJ1Y2tldCwgcHJvcHMuZGVueUxpc3Qub2JqZWN0S2V5KX0pYCxcbiAgICAgICAgICAgICAgICBgW2J1dHRvbjpQcnVuZSBGdW5jdGlvbl0oJHtsYW1iZGFGdW5jdGlvblVybChwcm9wcy5kZW55TGlzdC5wcnVuZS5wcnVuZUhhbmRsZXIpfSlgLFxuICAgICAgICAgICAgICAgIGBbYnV0dG9uOlBydW5lIExvZ3NdKCR7bGFtYmRhU2VhcmNoTG9nR3JvdXBVcmwocHJvcHMuZGVueUxpc3QucHJ1bmUucHJ1bmVIYW5kbGVyKX0pYCxcbiAgICAgICAgICAgICAgICBgW2J1dHRvbjpEZWxldGUgUXVldWVdKCR7c3FzUXVldWVVcmwocHJvcHMuZGVueUxpc3QucHJ1bmUucXVldWUpfSlgLFxuICAgICAgICAgICAgICAgIGBbYnV0dG9uOkRlbGV0ZSBMb2dzXSgke2xhbWJkYVNlYXJjaExvZ0dyb3VwVXJsKHByb3BzLmRlbnlMaXN0LnBydW5lLmRlbGV0ZUhhbmRsZXIpfSlgLFxuICAgICAgICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnRGVueSBMaXN0JyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5kZW55TGlzdC5tZXRyaWNEZW55TGlzdFJ1bGVzKHsgbGFiZWw6ICdSdWxlcycgfSksICdSRVBFQVQnKSxcbiAgICAgICAgICAgICAgcHJvcHMuZGVueUxpc3QucHJ1bmUucXVldWUubWV0cmljTnVtYmVyT2ZNZXNzYWdlc0RlbGV0ZWQoeyBsYWJlbDogJ0RlbGV0ZWQgRmlsZXMnIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGxlZnRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICAgICAgdGl0bGU6ICdQcnVuZSBGdW5jdGlvbiBIZWFsdGgnLFxuICAgICAgICAgICAgbGVmdDogW1xuICAgICAgICAgICAgICBmaWxsTWV0cmljKHByb3BzLmRlbnlMaXN0LnBydW5lLnBydW5lSGFuZGxlci5tZXRyaWNJbnZvY2F0aW9ucyh7IGxhYmVsOiAnSW52b2NhdGlvbnMnIH0pKSxcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhwcm9wcy5kZW55TGlzdC5wcnVuZS5wcnVuZUhhbmRsZXIubWV0cmljRXJyb3JzKHsgbGFiZWw6ICdFcnJvcnMnIH0pKSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG5cbiAgICAgICAgLi4uKHByb3BzLnBhY2thZ2VTdGF0cyA/IHJlbmRlclBhY2thZ2VTdGF0c1dpZGdldHMocHJvcHMucGFja2FnZVN0YXRzKSA6IFtdKSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlICpjYXRhbG9nT3ZlcnZpZXdMYW5ndWFnZVNlY3Rpb25zKHsgaW52ZW50b3J5LCBvcmNoZXN0cmF0aW9uLCBwYWNrYWdlRGF0YSB9OiBCYWNrZW5kRGFzaGJvYXJkUHJvcHMpOiBHZW5lcmF0b3I8SVdpZGdldFtdPiB7XG4gICAgeWllbGQgW1xuICAgICAgbmV3IFRleHRXaWRnZXQoe1xuICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgIHdpZHRoOiAyNCxcbiAgICAgICAgbWFya2Rvd246IFtcbiAgICAgICAgICAnIyBEb2N1bWVudGF0aW9uIEdlbmVyYXRpb24nLFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGBbYnV0dG9uOnByaW1hcnk6VHJhbnNsaXRlcmF0b3IgTG9nc10oJHtsb2dHcm91cFVybChvcmNoZXN0cmF0aW9uLnRyYW5zbGl0ZXJhdG9yLmxvZ0dyb3VwKX0pYCxcbiAgICAgICAgICBgW2J1dHRvbjpUcmFuc2xpdGVyYXRvciBFQ1MgQ2x1c3Rlcl0oJHtlY3NDbHVzdGVyVXJsKG9yY2hlc3RyYXRpb24uZWNzQ2x1c3Rlcil9KWAsXG4gICAgICAgIF0uam9pbignXFxuJyksXG4gICAgICB9KSxcbiAgICBdO1xuICAgIGNvbnN0IG1GYXJnYXRlVXNhZ2UgPSBuZXcgTWV0cmljKHtcbiAgICAgIGRpbWVuc2lvbnNNYXA6IHtcbiAgICAgICAgQ2xhc3M6ICdOb25lJyxcbiAgICAgICAgUmVzb3VyY2U6ICdPbkRlbWFuZCcsXG4gICAgICAgIFNlcnZpY2U6ICdGYXJnYXRlJyxcbiAgICAgICAgVHlwZTogJ1Jlc291cmNlJyxcbiAgICAgIH0sXG4gICAgICBtZXRyaWNOYW1lOiAnUmVzb3VyY2VDb3VudCcsXG4gICAgICBuYW1lc3BhY2U6ICdBV1MvVXNhZ2UnLFxuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICB9KTtcblxuICAgIHlpZWxkIFtcbiAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGhlaWdodDogNixcbiAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICB0aXRsZTogJ0ZhcmdhdGUgUmVzb3VyY2VzJyxcbiAgICAgICAgbGVmdDogW1xuICAgICAgICAgIG1GYXJnYXRlVXNhZ2Uud2l0aCh7IGxhYmVsOiAnRmFyZ2F0ZSBVc2FnZSAoT24tRGVtYW5kKScgfSksXG4gICAgICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgIGV4cHJlc3Npb246ICdTRVJWSUNFX1FVT1RBKG1GYXJnYXRlVXNhZ2UpJyxcbiAgICAgICAgICAgIGxhYmVsOiAnRmFyZ2F0ZSBRdW90YSAoT24tRGVtYW5kKScsXG4gICAgICAgICAgICB1c2luZ01ldHJpY3M6IHsgbUZhcmdhdGVVc2FnZSB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgIHJpZ2h0OiBbXG4gICAgICAgICAgb3JjaGVzdHJhdGlvbi5tZXRyaWNFY3NDcHVVdGlsaXphdGlvbih7IGxhYmVsOiAnQ1BVIFV0aWxpemF0aW9uJyB9KSxcbiAgICAgICAgICBvcmNoZXN0cmF0aW9uLm1ldHJpY0Vjc01lbW9yeVV0aWxpemF0aW9uKHsgbGFiZWw6ICdNZW1vcnkgVXRpbGl6YXRpb24nIH0pLFxuICAgICAgICBdLFxuICAgICAgICByaWdodFlBeGlzOiB7IGxhYmVsOiAnUGVyY2VudCcsIG1pbjogMCwgbWF4OiAxMDAsIHNob3dVbml0czogZmFsc2UgfSxcbiAgICAgIH0pLFxuICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgIHRpdGxlOiAnRUNTIFJlc291cmNlcycsXG4gICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICBmaWxsTWV0cmljKG9yY2hlc3RyYXRpb24ubWV0cmljRWNzTmV0d29ya1J4Qnl0ZXMoeyBsYWJlbDogJ1JlY2VpdmVkIEJ5dGVzJyB9KSksXG4gICAgICAgICAgZmlsbE1ldHJpYyhvcmNoZXN0cmF0aW9uLm1ldHJpY0Vjc05ldHdvcmtUeEJ5dGVzKHsgbGFiZWw6ICdUcmFuc21pdHRlZCBCeXRlcycgfSkpLFxuICAgICAgICBdLFxuICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgIHJpZ2h0OiBbXG4gICAgICAgICAgZmlsbE1ldHJpYyhvcmNoZXN0cmF0aW9uLm1ldHJpY0Vjc1Rhc2tDb3VudCh7IGxhYmVsOiAnVGFzayBDb3VudCcgfSkpLFxuICAgICAgICBdLFxuICAgICAgICByaWdodFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgfSksXG4gICAgXTtcblxuICAgIGZvciAoY29uc3QgbGFuZ3VhZ2Ugb2YgRG9jdW1lbnRhdGlvbkxhbmd1YWdlLkFMTCkge1xuICAgICAgeWllbGQgW1xuICAgICAgICBuZXcgVGV4dFdpZGdldCh7XG4gICAgICAgICAgaGVpZ2h0OiAxLFxuICAgICAgICAgIHdpZHRoOiAyNCxcbiAgICAgICAgICBtYXJrZG93bjogYCMjIExhbmd1YWdlOiAke2xhbmd1YWdlLnRvU3RyaW5nKCl9YCxcbiAgICAgICAgfSksXG4gICAgICBdO1xuICAgICAgeWllbGQgW1xuICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICB3aWR0aDogNixcbiAgICAgICAgICB0aXRsZTogJ1BhY2thZ2UgVmVyc2lvbnMnLFxuICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgIGludmVudG9yeS5tZXRyaWNTdXBwb3J0ZWRQYWNrYWdlVmVyc2lvbkNvdW50KGxhbmd1YWdlLCB7IGxhYmVsOiAnQXZhaWxhYmxlJywgY29sb3I6ICcjMmNhMDJjJyB9KSxcbiAgICAgICAgICAgIGludmVudG9yeS5tZXRyaWNVbnN1cHBvcnRlZFBhY2thZ2VWZXJzaW9uQ291bnQobGFuZ3VhZ2UsIHsgbGFiZWw6ICdVbnN1cHBvcnRlZCcsIGNvbG9yOiAnIzk0NjdiZCcgfSksXG4gICAgICAgICAgICBpbnZlbnRvcnkubWV0cmljTWlzc2luZ1BhY2thZ2VWZXJzaW9uQ291bnQobGFuZ3VhZ2UsIHsgbGFiZWw6ICdNaXNzaW5nJywgY29sb3I6ICcjZDYyNzI4JyB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIGxlZnRZQXhpczogeyBzaG93VW5pdHM6IGZhbHNlIH0sXG4gICAgICAgICAgdmlldzogR3JhcGhXaWRnZXRWaWV3LlBJRSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBNaXNzaW5nRG9jdW1lbnRhdGlvbldpZGdldCh0aGlzLCBgTWlzc2luZ0RvY3MtJHtsYW5ndWFnZS5uYW1lfWAsIHtcbiAgICAgICAgICBidWNrZXQ6IHBhY2thZ2VEYXRhLFxuICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICBsYW5ndWFnZSxcbiAgICAgICAgICB0aXRsZTogJ1BhY2thZ2UgVmVyc2lvbnMgbWlzc2luZyBkb2N1bWVudGF0aW9uJyxcbiAgICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICB3aWR0aDogNixcbiAgICAgICAgICB0aXRsZTogJ1BhY2thZ2UgVmVyc2lvbiBTdWJtb2R1bGVzJyxcbiAgICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgICBpbnZlbnRvcnkubWV0cmljU3VwcG9ydGVkU3VibW9kdWxlQ291bnQobGFuZ3VhZ2UsIHsgbGFiZWw6ICdBdmFpbGFibGUnLCBjb2xvcjogJyMyY2EwMmMnIH0pLFxuICAgICAgICAgICAgaW52ZW50b3J5Lm1ldHJpY1Vuc3VwcG9ydGVkU3VibW9kdWxlQ291bnQobGFuZ3VhZ2UsIHsgbGFiZWw6ICdVbnN1cHBvcnRlZCcsIGNvbG9yOiAnIzk0NjdiZCcgfSksXG4gICAgICAgICAgICBpbnZlbnRvcnkubWV0cmljTWlzc2luZ1N1Ym1vZHVsZUNvdW50KGxhbmd1YWdlLCB7IGxhYmVsOiAnTWlzc2luZycsIGNvbG9yOiAnI2Q2MjcyOCcgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBsZWZ0WUF4aXM6IHsgc2hvd1VuaXRzOiBmYWxzZSB9LFxuICAgICAgICAgIHZpZXc6IEdyYXBoV2lkZ2V0Vmlldy5QSUUsXG4gICAgICAgIH0pLFxuICAgICAgXTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24qIHJlbmRlclBhY2thZ2VTb3VyY2VzV2lkZ2V0cyhwYWNrYWdlU291cmNlczogUGFja2FnZVNvdXJjZUJpbmRSZXN1bHRbXSk6IEdlbmVyYXRvcjxJV2lkZ2V0W10sIHVuZGVmaW5lZCwgdW5kZWZpbmVkPiB7XG4gIGZvciAoY29uc3QgcGFja2FnZVNvdXJjZSBvZiBwYWNrYWdlU291cmNlcykge1xuICAgIHlpZWxkIFtcbiAgICAgIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiAyLFxuICAgICAgICB3aWR0aDogMjQsXG4gICAgICAgIG1hcmtkb3duOiBbXG4gICAgICAgICAgYCMgJHtwYWNrYWdlU291cmNlLm5hbWV9YCxcbiAgICAgICAgICAnJyxcbiAgICAgICAgICAuLi4ocGFja2FnZVNvdXJjZS5saW5rcyA/PyBbXSlcbiAgICAgICAgICAgIC5tYXAoKHsgbmFtZSwgcHJpbWFyeSwgdXJsIH0pID0+IGBbJHtwcmltYXJ5ID8gJ2J1dHRvbjpwcmltYXJ5JyA6ICdidXR0b24nfToke25hbWV9XSgke3VybH0pYCksXG4gICAgICAgIF0uam9pbignXFxuJyksXG4gICAgICB9KSxcbiAgICBdO1xuICAgIHlpZWxkKiBwYWNrYWdlU291cmNlLmRhc2hib2FyZFdpZGdldHM7XG4gIH1cbiAgcmV0dXJuO1xufVxuXG5mdW5jdGlvbiByZW5kZXJQYWNrYWdlU3RhdHNXaWRnZXRzKHBhY2thZ2VTdGF0czogUGFja2FnZVN0YXRzKTogSVdpZGdldFtdW10ge1xuICByZXR1cm4gW1xuICAgIFtcbiAgICAgIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiAyLFxuICAgICAgICB3aWR0aDogMjQsXG4gICAgICAgIG1hcmtkb3duOlxuICAgICAgICAgIFtcbiAgICAgICAgICAgICcjIFBhY2thZ2UgU3RhdHMnLFxuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBgW2J1dHRvbjpwcmltYXJ5OlBhY2thZ2UgU3RhdHMgT2JqZWN0XSgke3MzT2JqZWN0VXJsKHBhY2thZ2VTdGF0cy5idWNrZXQsIHBhY2thZ2VTdGF0cy5zdGF0c0tleSl9KWAsXG4gICAgICAgICAgICBgW2J1dHRvbjpQYWNrYWdlIFN0YXRzIEZ1bmN0aW9uXSgke2xhbWJkYUZ1bmN0aW9uVXJsKHBhY2thZ2VTdGF0cy5oYW5kbGVyKX0pYCxcbiAgICAgICAgICAgIGBbYnV0dG9uOlBhY2thZ2UgU3RhdHMgTG9nc10oJHtsYW1iZGFTZWFyY2hMb2dHcm91cFVybChwYWNrYWdlU3RhdHMuaGFuZGxlcil9KWAsXG4gICAgICAgICAgXS5qb2luKCdcXG4nKSxcbiAgICAgIH0pLFxuICAgIF0sXG4gICAgW1xuICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgIHRpdGxlOiAnTnVtYmVyIG9mIFBhY2thZ2UgU3RhdHMgUmVjb3JkZWQnLFxuICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgZmlsbE1ldHJpYyhwYWNrYWdlU3RhdHMubWV0cmljUGFja2FnZXNDb3VudCh7IGxhYmVsOiAnUGFja2FnZXMgd2l0aCBzdGF0cycgfSksICdSRVBFQVQnKSxcbiAgICAgICAgXSxcbiAgICAgICAgbGVmdFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgfSksXG4gICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBoZWlnaHQ6IDYsXG4gICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgdGl0bGU6ICdJbnZvY2F0aW9uIER1cmF0aW9uJyxcbiAgICAgICAgbGVmdDogW1xuICAgICAgICAgIHBhY2thZ2VTdGF0cy5oYW5kbGVyLm1ldHJpY0R1cmF0aW9uKHsgbGFiZWw6ICdEdXJhdGlvbicgfSksXG4gICAgICAgIF0sXG4gICAgICAgIGxlZnRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgcmlnaHRBbm5vdGF0aW9uczogW3tcbiAgICAgICAgICBjb2xvcjogJyNmZmE1MDAnLFxuICAgICAgICAgIGxhYmVsOiAnMTUgbWludXRlcyAoTGFtYmRhIHRpbWVvdXQpJyxcbiAgICAgICAgICB2YWx1ZTogRHVyYXRpb24ubWludXRlcygxNSkudG9TZWNvbmRzKCksXG4gICAgICAgIH1dLFxuICAgICAgfSksXG4gICAgXSxcbiAgXTtcbn1cbiJdfQ==