"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../../filesystem");
const certificate_1 = require("../certificate");
const distinguished_name_1 = require("../distinguished-name");
const exec = util_1.promisify(child_process.exec);
let tmpDir;
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
beforeAll(async () => {
    tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
});
afterAll(async () => {
    const unlinks = [];
    const filenames = await fs.promises.readdir(tmpDir);
    for (const file of filenames) {
        unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
    }
    await Promise.all(unlinks);
    await fs.promises.rmdir(tmpDir);
});
test('generate self-signed', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    const crtFileName = path.join(tmpDir, 'ss-ca.crt');
    const keyFileName = path.join(tmpDir, 'ss-ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PW: passphrase } });
    const keyVerification = keyOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 3 * 365);
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certificate.certChain).toEqual('');
    expect(certVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toMatch(/Subject: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toContain('Version: 3 (0x2)');
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toMatch(/Private-Key: \(2048 bit(, \d+ primes)?\)/);
});
test('generate self-signed with expiry', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase, 5 * 365);
    const crtFileName = path.join(tmpDir, 'ss-ca.crt');
    const keyFileName = path.join(tmpDir, 'ss-ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 5 * 365);
    // THEN
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
});
test('generate signed certificate', async () => {
    // GIVEN
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, undefined, ca);
    const crtFileName = path.join(tmpDir, 'signed.crt');
    const crtChainFileName = path.join(tmpDir, 'chain.crt');
    const keyFileName = path.join(tmpDir, 'signed.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    if (certificate.certChain) {
        await filesystem_1.writeAsciiFile(crtChainFileName, certificate.certChain);
    }
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const certChainOut = await exec(`openssl x509 -noout -text -in ${crtChainFileName}`);
    const certChainVerification = certChainOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const keyVerification = keyOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 3 * 365);
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    // The cert chain should contain the signing cert
    expect(certificate.certChain).toContain(ca.cert);
    // The cert should not contain any chain
    expect(certificate.cert.indexOf('-----BEGIN CERTIFICATE-----')).toBe(certificate.cert.lastIndexOf('-----BEGIN CERTIFICATE-----'));
    expect(certificate.certChain).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.certChain).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toMatch(/Subject: CN\s*=\s*CertCN, O\s*=\s*CertO, OU\s*=\s*CertOU/);
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
    expect(certChainVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certChainVerification).toMatch(/Subject: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certChainVerification).toContain('Public-Key: (2048 bit)');
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toMatch(/Private-Key: \(2048 bit(, \d+ primes)?\)/);
});
test('generate signed certificate with expiry', async () => {
    // GIVEN
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, 5 * 365, ca);
    const crtFileName = path.join(tmpDir, 'signed.crt');
    const crtChainFileName = path.join(tmpDir, 'chain.crt');
    const keyFileName = path.join(tmpDir, 'signed.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    if (certificate.certChain) {
        await filesystem_1.writeAsciiFile(crtChainFileName, certificate.certChain);
    }
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 5 * 365);
    // THEN
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
});
test('convert to PKCS #12', async () => {
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, undefined, ca);
    const pkcs12Passphrase = 'test_passphrase';
    // WHEN
    const pkcs12Data = await certificate.toPkcs12(pkcs12Passphrase);
    const fileName = path.join(tmpDir, 'cert.p12');
    await filesystem_1.writeBinaryFile(fileName, pkcs12Data);
    let pkcs12Validation;
    // If the PKCS12 passphrase does not match, openssl will return a non-zero exit code and fail the test
    // This was tested for both OpenSSL 1.0.x and 1.1.x.
    pkcs12Validation = await exec(`openssl pkcs12 -in ${fileName} -info -nodes -passin env:PW`, { env: { PATH: process.env.PATH, PW: pkcs12Passphrase } });
    const validationOut = pkcs12Validation.stdout;
    // THEN
    // Must have the certificate's cert
    expect(validationOut).toMatch(/subject=\/?CN\s*=\s*CertCN[/,]\s*O\s*=\s*CertO[/,]\s*OU\s*=\s*CertOU\n{1,2}issuer=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}-----BEGIN CERTIFICATE-----/);
    // Must have the CA cert
    expect(validationOut).toMatch(/subject=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}issuer=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}-----BEGIN CERTIFICATE-----/);
    // Must have the decrypted private key
    expect(validationOut).toContain('-----BEGIN PRIVATE KEY-----');
});
test('decrypt private key', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    // WHEN
    const decryptedKey = await certificate_1.Certificate.decryptKey(certificate.key, passphrase);
    const crtFileName = path.join(tmpDir, 'ca.crt');
    const keyFileName = path.join(tmpDir, 'ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const expectedDecryptedKeyOut = await exec(`openssl rsa -in ${keyFileName} -passin env:PW`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const expectedDecryptedKey = expectedDecryptedKeyOut.stdout;
    // THEN
    expect(decryptedKey).toEqual(expectedDecryptedKey);
    // Must have the decrypted private key
    // OpenSSL 1.0.x: -----BEGIN RSA PRIVATE KEY-----
    // OpenSSL 1.1.x: -----BEGIN PRIVATE KEY-----
    expect(decryptedKey).toMatch(/-----BEGIN (?:RSA )?PRIVATE KEY-----/);
});
//# sourceMappingURL=data:application/json;base64,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