"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
describe('Version', () => {
    describe('.isGreaterThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '10.0.9.2',
                    secondVersion: '10.1.9.2',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ], [
                'greater than',
                {
                    firstVersion: '2.0.0.1',
                    secondVersion: '2.0.0.0',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isGreaterThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    test('.isGreaterThan constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 1, 9, 2]);
        const result = lhs.isGreaterThan(lib_1.Version.parse('10.0.9.2'));
        expect(result).toEqual(true);
    });
    describe('.isLessThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'greater minor version',
                {
                    firstVersion: '10.1.9.2',
                    secondVersion: '10.0.9.2',
                    expectedValue: false,
                },
            ], [
                'greater patch version',
                {
                    firstVersion: '1.1.1.2',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '2.0.0.0',
                    secondVersion: '2.0.0.1',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isLessThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isEqual', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: true,
                },
            ], [
                'unequal',
                {
                    firstVersion: '2.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isEqual(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    test('.isLessThan using constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 0, 9, 2]);
        const result = lhs.isLessThan(lib_1.Version.parse('10.1.9.2'));
        expect(result).toEqual(true);
    });
    describe('throws exception for invalid components', () => {
        test.each([
            [
                'incorrect component count',
                {
                    version: [10, 1, 9],
                    expectedException: /Invalid version format. Version should contain exactly 4 components./,
                },
            ], [
                'negative value',
                {
                    version: [10, -1, 9, 2],
                    expectedException: /Invalid version format. None of the version components can be negative./,
                },
            ], [
                'decimal value',
                {
                    version: [10, 1, 9.2, 2],
                    expectedException: /Invalid version format. None of the version components can contain decimal values./,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            expect(() => new lib_1.Version(version)).toThrow(expectedException);
        });
    });
    describe('components are mapped to correct properties', () => {
        // GIVEN
        const versionComponents = [10, 1, 9, 2];
        let version;
        // WHEN
        beforeEach(() => {
            version = new lib_1.Version(versionComponents);
        });
        // THEN
        test('majorVersion', () => {
            expect(version.majorVersion).toEqual(versionComponents[0]);
        });
        test('minorVersion', () => {
            expect(version.minorVersion).toEqual(versionComponents[1]);
        });
        test('releaseVersion', () => {
            expect(version.releaseVersion).toEqual(versionComponents[2]);
        });
        test('patchVersion', () => {
            expect(version.patchVersion).toEqual(versionComponents[3]);
        });
    });
    describe('.parse throws exception', () => {
        test.each([
            [
                'ending with .',
                {
                    version: '10.1.9.',
                    expectedException: /Invalid version format/,
                },
            ], [
                'empty string',
                {
                    version: '',
                    expectedException: /Invalid version format/,
                },
            ], [
                'negative value',
                {
                    version: '10.-1.9.2',
                    expectedException: /Invalid version format/,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            expect(() => lib_1.Version.parse(version)).toThrow(expectedException);
        });
    });
    test('.parse() works', () => {
        // GIVEN
        const versionString = '10.1.9.2';
        // WHEN
        const version = lib_1.Version.parse(versionString);
        // THEN
        expect(version.majorVersion).toBe(10);
        expect(version.minorVersion).toBe(1);
        expect(version.releaseVersion).toBe(9);
        expect(version.patchVersion).toBe(2);
    });
});
//# sourceMappingURL=data:application/json;base64,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