"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const lib_1 = require("../lib");
const secrets_management_1 = require("../lib/secrets-management");
class MockUserData extends aws_ec2_1.UserData {
    constructor() {
        super();
        this.addCommands = jest.fn();
        this.addCommands = jest.fn();
        this.addOnExitCommands = jest.fn();
        this.render = jest.fn();
        this.addExecuteFileCommand = jest.fn();
        this.addS3DownloadCommand = jest.fn();
        this.addSignalOnExitCommand = jest.fn();
    }
}
class MockDeploymentInstance extends deployment_instance_1.DeploymentInstance {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mockUserData = new MockUserData();
    }
    get userData() {
        return this.mockUserData;
    }
}
const DEADLINE_CLIENT_SUBNET_NAME = 'DeadlineClient';
const RENDER_QUEUE_ALB_SUBNET_NAME = 'RenderQueueALB';
describe('SecretsManagementIdentityRegistration', () => {
    let app;
    let dependencyStack;
    let deploymentInstanceStack;
    let stack;
    let vpc;
    let version;
    let repository;
    let deploymentInstance;
    let deploymentInstanceRole;
    let renderQueueSubnets;
    let target;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        dependencyStack = new aws_cdk_lib_1.Stack(app, 'DependencyStack');
        deploymentInstanceStack = new aws_cdk_lib_1.Stack(app, 'DeploymentInstanceStack');
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc', {
            subnetConfiguration: [
                {
                    name: RENDER_QUEUE_ALB_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
                    cidrMask: 28,
                },
                {
                    name: 'Public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
                {
                    name: DEADLINE_CLIENT_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
            ],
        });
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        deploymentInstance = new MockDeploymentInstance(deploymentInstanceStack, 'DeploymentInstance', {
            vpc,
        });
        renderQueueSubnets = vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME });
    });
    describe('when Repository uses secrets management', () => {
        beforeEach(() => {
            // GIVEN
            repository = new lib_1.Repository(dependencyStack, 'Repository', {
                version,
                vpc,
                secretsManagementSettings: {
                    enabled: true,
                },
            });
            jest.spyOn(repository, 'configureClientInstance');
            // Get a reference to the DeploymentInstance's IAM role L1 resource
            deploymentInstanceRole = (deploymentInstance
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
        });
        function createTarget() {
            target = new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistration', {
                deploymentInstance,
                renderQueueSubnets,
                repository,
                version,
            });
        }
        describe('Deadline Client installer', () => {
            test('grant S3 read to client installer', () => {
                // WHEN
                createTarget();
                // THEN
                assertions_1.Template.fromStack(deploymentInstanceStack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    's3:GetObject*',
                                    's3:GetBucket*',
                                    's3:List*',
                                ],
                                Effect: 'Allow',
                                Resource: assertions_1.Match.arrayWith([...deploymentInstanceStack.resolve([
                                        version.linuxInstallers.client.s3Bucket.bucketArn,
                                        version.linuxInstallers.client.s3Bucket.arnForObjects(version.linuxInstallers.client.objectKey),
                                    ])]),
                            },
                        ]),
                    },
                    Roles: [
                        deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                    ],
                });
            });
            test('downloads and executes Client installer', () => {
                // GIVEN
                const clientInstallerLocalFilename = 'clientInstallerLocalFilename';
                const userData = deploymentInstance.userData;
                userData.addS3DownloadCommand.mockReturnValueOnce(clientInstallerLocalFilename);
                // WHEN
                createTarget();
                // THEN
                expect(userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: version.linuxInstallers.client.s3Bucket,
                    bucketKey: version.linuxInstallers.client.objectKey,
                });
                expect(userData.addCommands).toHaveBeenCalledWith(`chmod +x "${clientInstallerLocalFilename}"`);
                expect(userData.addCommands).toHaveBeenCalledWith([
                    // This is required b/c USER and HOME environment variables are not defined when running
                    // user-data
                    'sudo', '--login',
                    // Run the Deadline Client installer
                    `"${clientInstallerLocalFilename}"`,
                    '--mode', 'unattended',
                    '--connectiontype', 'Remote',
                    '--proxyrootdir', '127.0.0.1:8080',
                    '--noguimode', 'true',
                    '--slavestartup', 'false',
                    '--launcherdaemon', 'false',
                    '--restartstalled', 'true',
                    '--autoupdateoverride', 'False',
                ].join(' '));
            });
        });
        test('grants DeploymentInstance role permissions to describe subnets', () => {
            // WHEN
            createTarget();
            // THEN
            assertions_1.Template.fromStack(deploymentInstanceStack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: 'ec2:DescribeSubnets',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ]),
                },
                Roles: [stack.resolve(deploymentInstanceRole.ref)],
            });
        });
        test('configures direct repository connection', () => {
            // WHEN
            createTarget();
            // THEN
            expect(repository.configureClientInstance).toHaveBeenCalledWith({
                host: deploymentInstance,
                mountPoint: expect.any(String),
            });
        });
        test('grants DeploymentInstance read access to the Deadline Secrets Management admin credentials secret', () => {
            // WHEN
            createTarget();
            // THEN
            assertions_1.Template.fromStack(deploymentInstanceStack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: deploymentInstanceStack.resolve(repository.secretsManagementSettings.credentials.secretArn),
                        },
                    ]),
                },
                Roles: [
                    deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                ],
            });
        });
        describe('Identity registration settings script', () => {
            function getIdentityRegistrationSettingsScript() {
                return target.node.findChild('ConfigureIdentityRegistrationSettingScript');
            }
            test('DeploymentInstance granted S3 read access', () => {
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                assertions_1.Template.fromStack(deploymentInstanceStack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    's3:GetObject*',
                                    's3:GetBucket*',
                                    's3:List*',
                                ],
                                Effect: 'Allow',
                                Resource: deploymentInstanceStack.resolve([
                                    identityRegistrationSettingsScript.bucket.bucketArn,
                                    identityRegistrationSettingsScript.bucket.arnForObjects('*'),
                                ]),
                            },
                        ]),
                    },
                    Roles: [deploymentInstanceStack.resolve(deploymentInstanceRole.ref)],
                });
            });
            test('DeploymentInstance downloads script', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce('deadlineClientLocalPath');
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath);
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                expect(deploymentInstance.userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: identityRegistrationSettingsScript.bucket,
                    bucketKey: identityRegistrationSettingsScript.s3ObjectKey,
                    localFile: expect.stringMatching(/^\/home\/ec2-user\//),
                });
            });
            test('DeploymentInstance sets ownership and executable permissions for ec2-user', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                // WHEN
                createTarget();
                // THEN
                expect(deploymentInstance.userData.addCommands).toHaveBeenCalledWith(`chmod +x ${identityRegistrationSettingsScriptLocalPath}`, `chown ec2-user:ec2-user ${identityRegistrationSettingsScriptLocalPath}`);
            });
        });
        describe('.addSubnetIdentityRegistrationSetting(...)', () => {
            describe.each([
                [lib_1.SecretsManagementRole.SERVER],
                [lib_1.SecretsManagementRole.CLIENT],
            ])('identityRole=%s', (identityRole) => {
                describe.each([
                    [lib_1.SecretsManagementRegistrationStatus.PENDING],
                    [lib_1.SecretsManagementRegistrationStatus.REGISTERED],
                    [lib_1.SecretsManagementRegistrationStatus.REVOKED],
                ])('registrationStatus=%s', (registrationStatus) => {
                    test('executes identity registration settings configuration script with proper arguments', () => {
                        // GIVEN
                        const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                        (deploymentInstance.userData.addS3DownloadCommand
                            .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                            .mockReturnValueOnce('efsMountScriptLocalPath')
                            .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                            .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                        const clientStack = new aws_cdk_lib_1.Stack(app, 'ClientStack');
                        createTarget();
                        // WHEN
                        target.addSubnetIdentityRegistrationSetting({
                            dependent: new constructs_1.Construct(clientStack, 'DeadlineClient'),
                            registrationStatus,
                            role: identityRole,
                            vpc,
                            vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                        });
                        // THEN
                        const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                            return deploymentInstanceStack.resolve(call);
                        });
                        const expectedCall = [{
                                'Fn::Join': [
                                    '',
                                    [
                                        // Command is run as "ec2-user" which has the database credentials stored
                                        `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                        stack.resolve(aws_cdk_lib_1.Fn.join(' ', [
                                            '--region',
                                            stack.region,
                                            // The Deadline Secrets Management admin credentials secret ARN is passed
                                            '--credentials',
                                            `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                            // The Render Queue's ALB subnets are passed as --connection-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                                .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                            // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                                .subnetIds.map(subnetID => {
                                                return `--source-subnet "${subnetID},${identityRole},${registrationStatus}"`;
                                            })),
                                        ])),
                                    ],
                                ],
                            }];
                        expect(resolvedCalls).toContainEqual(expectedCall);
                    });
                });
            });
            test('throws execption when using Administrator role', () => {
                // GIVEN
                createTarget();
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: new constructs_1.Construct(stack, 'Dependent'),
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: lib_1.SecretsManagementRole.ADMINISTRATOR,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError('The Administrator role cannot be set using a Deadline identity registration setting');
            });
            test('throws when two rules for same source subnet with different roles', () => {
                // GIVEN
                const client1 = new constructs_1.Construct(stack, 'client1');
                const client2 = new constructs_1.Construct(stack, 'client2');
                const existingRole = lib_1.SecretsManagementRole.SERVER;
                const newRole = lib_1.SecretsManagementRole.CLIENT;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: existingRole,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: newRole,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with role "${existingRole}" but another caller requested "${newRole}"`);
            });
            test('throws when two rules for same source subnet with different registration statuses', () => {
                // GIVEN
                const client1 = new constructs_1.Construct(stack, 'client1');
                const client2 = new constructs_1.Construct(stack, 'client2');
                const role = lib_1.SecretsManagementRole.CLIENT;
                const existingStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const newStatus = lib_1.SecretsManagementRegistrationStatus.PENDING;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: existingStatus,
                    role,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: newStatus,
                        role,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with registrationStatus "${existingStatus}" but another caller requested "${newStatus}"`);
            });
            test('de-duplicates subnets', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                const clientStack = new aws_cdk_lib_1.Stack(app, 'ClientStack');
                const client1 = new constructs_1.Construct(clientStack, 'client1');
                const client2 = new constructs_1.Construct(clientStack, 'client2');
                createTarget();
                const baseProps = {
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                };
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client1,
                });
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client2,
                });
                // THEN
                const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                    return deploymentInstanceStack.resolve(call);
                });
                const expectedCall = [{
                        'Fn::Join': [
                            '',
                            [
                                // Command is run as "ec2-user" which has the database credentials stored
                                `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                stack.resolve(aws_cdk_lib_1.Fn.join(' ', [
                                    '--region',
                                    stack.region,
                                    // The Deadline Secrets Management admin credentials secret ARN is passed
                                    '--credentials',
                                    `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                    // The Render Queue's ALB subnets are passed as --connection-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                        .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                    // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                        .subnetIds.map(subnetID => {
                                        return `--source-subnet "${subnetID},${baseProps.role},${baseProps.registrationStatus}"`;
                                    })),
                                ])),
                            ],
                        ],
                    }];
                expect(resolvedCalls).toContainEqual(expectedCall);
            });
            test('warns about dedicated subnets when render queue ALB and source subnets match', () => {
                // GIVEN
                createTarget();
                const dependent = new constructs_1.Construct(stack, 'Dependent');
                const registrationStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const role = lib_1.SecretsManagementRole.CLIENT;
                const vpcSubnets = {
                    subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME,
                };
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    dependent,
                    registrationStatus,
                    role,
                    vpc,
                    vpcSubnets,
                });
                assertions_1.Annotations.fromStack(stack).hasWarning(`/${dependent.node.path}`, `Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`);
            });
        });
        test('Repository with no admin credentials throws an error', () => {
            // GIVEN
            class RepositoryNoCredentials extends lib_1.Repository {
                constructor(scope, id, props) {
                    super(scope, id, props);
                    this.secretsManagementSettings = {
                        enabled: true,
                        credentials: undefined,
                    };
                }
            }
            repository = new RepositoryNoCredentials(dependencyStack, 'RepoNoCreds', {
                version,
                vpc,
            });
            // WHEN
            const when = createTarget;
            // THEN
            expect(when).toThrowError('Repository does not contain secrets management credentials');
        });
    });
    test('when Repository disables secrets management throws an exception', () => {
        // GIVEN
        repository = new lib_1.Repository(stack, 'Repository', {
            version,
            vpc,
            secretsManagementSettings: {
                enabled: false,
            },
        });
        // WHEN
        function when() {
            new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistrationSettings', {
                deploymentInstance,
                renderQueueSubnets: vpc.selectSubnets({
                    subnetGroupName: 'RenderQueueALB',
                }),
                repository,
                version,
            });
        }
        // THEN
        expect(when).toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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