"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const __1 = require("../..");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const test_helper_1 = require("./test-helper");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let version;
    let renderQueueVersion;
    // GIVEN
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        dependencyStack = new aws_cdk_lib_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueVersion = new lib_1.VersionQuery(stack, 'Version');
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    describe('basic', () => {
        let repository;
        let renderQueue;
        beforeEach(() => {
            repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
            });
            renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
        });
        test('creates cluster', () => {
            // THEN
            assertions_1.Template.fromStack(stack).resourceCountIs('AWS::ECS::Cluster', 1);
        });
        test('creates service', () => {
            // THEN
            assertions_1.Template.fromStack(stack).resourceCountIs('AWS::ECS::Service', 1);
        });
        test('creates task definition', () => {
            // THEN
            assertions_1.Template.fromStack(stack).resourceCountIs('AWS::ECS::TaskDefinition', 1);
        });
        test('closed ingress by default', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', assertions_1.Match.not({
                // The openListener=true option would create an ingress rule in the listener's SG.
                // make sure that we don't have that.
                // DDN - intentionally broke
                SecurityGroupIngress: assertions_1.Match.anyValue(),
            }));
        });
        test('creates load balancer with default values', () => {
            // THEN
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assertions_1.Match.arrayWith([
                    {
                        Key: 'deletion_protection.enabled',
                        Value: 'true',
                    },
                ]),
                Scheme: 'internal',
            }, 1);
        });
        test('creates a log group with default prefix of "/renderfarm/"', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
                LogGroupName: '/renderfarm/RenderQueue',
                RetentionInDays: 3,
            });
        });
        test('configure the container log driver', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    assertions_1.Match.objectLike({
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    'Fn::GetAtt': [
                                        assertions_1.Match.stringLikeRegexp('^RenderQueueLogGroupWrapper.*'),
                                        'LogGroupName',
                                    ],
                                },
                                'awslogs-stream-prefix': 'RCS',
                                'awslogs-region': { Ref: 'AWS::Region' },
                            },
                        },
                    }),
                ],
            });
        });
        test('child dependencies added', () => {
            // GIVEN
            const host = new aws_ec2_1.Instance(stack, 'Host', {
                vpc,
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            });
            // WHEN
            renderQueue.addChildDependency(host);
            // THEN
            assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
                DependsOn: assertions_1.Match.arrayWith([
                    'RenderQueueAlbEc2ServicePatternService5B6692FB',
                    'RenderQueueLBPublicListenerBBF15D5F',
                    'RenderQueueRCSTaskA9AE70D3',
                    'RenderQueueWaitForStableService4B92A8D2',
                ]),
            });
        });
        describe('renderQueueSize.min', () => {
            describe('defaults to 1', () => {
                function assertSpecifiesMinSize(stackToAssert) {
                    assertions_1.Template.fromStack(stackToAssert).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                        MinSize: '1',
                    });
                }
                test('renderQueueSize unspecified', () => {
                    // THEN
                    assertSpecifiesMinSize(stack);
                });
                test('renderQueueSize.min unspecified', () => {
                    // GIVEN
                    const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
                    // WHEN
                    new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                        images,
                        repository,
                        version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                        vpc,
                        renderQueueSize: {},
                    });
                    // THEN
                    assertSpecifiesMinSize(isolatedStack);
                });
            });
            // Asserts that at least one RCS container and ASG instance must be created.
            test('throws error when minimum size is 0', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    renderQueueSize: {
                        min: 0,
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueueTest', props);
                })
                    // THEN
                    .toThrow('renderQueueSize.min capacity must be at least 1: got 0');
            });
            // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
            test('validates Deadline pre 10.1.10 has min value of at most 1', () => {
                // GIVEN
                const min = 2;
                const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
                const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
                const props = {
                    images,
                    repository,
                    version: versionOld,
                    vpc,
                    renderQueueSize: {
                        min,
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(newStack, 'RenderQueueTest', props);
                })
                    // THEN
                    .toThrow(`renderQueueSize.min for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${min}`);
            });
            // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
            test.each([
                [1],
                [2],
                [10],
            ])('configures minimum number of ASG instances to %d', (min) => {
                // GIVEN
                const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {
                        min,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                // THEN
                assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: min.toString(),
                });
            });
        });
        describe('renderQueueSize.max', () => {
            describe('defaults to 1', () => {
                function assertSpecifiesMaxSize(stackToAssert) {
                    assertions_1.Template.fromStack(stackToAssert).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                        MaxSize: '1',
                    });
                }
                test('renderQueueSize unspecified', () => {
                    // THEN
                    assertSpecifiesMaxSize(stack);
                });
                test('renderQueueSize.max unspecified', () => {
                    // GIVEN
                    const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
                    // WHEN
                    new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                        images,
                        repository,
                        version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                        vpc,
                        renderQueueSize: {},
                    });
                    // THEN
                    assertSpecifiesMaxSize(isolatedStack);
                });
            });
            // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
            test('validates Deadline pre 10.1.10 has max value of at most 1', () => {
                // GIVEN
                const max = 2;
                const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
                const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
                const props = {
                    images,
                    repository,
                    version: versionOld,
                    vpc,
                    renderQueueSize: {
                        max,
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(newStack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(`renderQueueSize.max for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${max}`);
            });
            // Asserts that when the renderQueueSize.max prop is specified, the underlying ASG's max property is set accordingly.
            test.each([
                [1],
                [2],
                [10],
            ])('configures maximum number of ASG instances to %d', (max) => {
                // GIVEN
                const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {
                        max,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                // THEN
                assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: max.toString(),
                });
            });
        });
        describe('renderQueueSize.desired', () => {
            describe('defaults', () => {
                test('unset ASG desired', () => {
                    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                        DesiredCapacity: assertions_1.Match.absent(),
                    });
                    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                        DesiredCount: 1,
                    });
                });
            });
            test('validates Deadline pre 10.1.10 has desired value of at most 1', () => {
                // GIVEN
                const desired = 2;
                const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
                const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
                const props = {
                    images,
                    repository,
                    version: versionOld,
                    vpc,
                    renderQueueSize: {
                        desired,
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(newStack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(`renderQueueSize.desired for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${desired}`);
            });
            test.each([
                [1],
                [2],
                [10],
            ])('is specified to %d', (desired) => {
                // GIVEN
                const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {
                        desired,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                // THEN
                assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: desired.toString(),
                });
                // THEN
                assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::ECS::Service', {
                    DesiredCount: desired,
                });
            });
        });
        test('creates WaitForStableService by default', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_WaitForStableService', {
                cluster: stack.resolve(renderQueue.cluster.clusterArn),
                // eslint-disable-next-line dot-notation
                services: [stack.resolve(renderQueue['pattern'].service.serviceArn)],
            });
        });
        test('Does not enable filesystem cache by default', () => {
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::AutoScaling::LaunchConfiguration', {
                UserData: {
                    'Fn::Base64': {
                        'Fn::Join': [
                            '',
                            assertions_1.Match.arrayWith([
                                assertions_1.Match.stringLikeRegexp('.*# RenderQueue file caching enabled.*'),
                            ]),
                        ],
                    },
                },
            }, 0);
        });
        test('runs as RCS user', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({ User: '1000:1000' }),
                ]),
            });
        });
        test('.backendConnections is associated with ASG security group rules', () => {
            // GIVEN
            const instance = new aws_ec2_1.Instance(dependencyStack, 'BackendConnectionInstance', {
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux(),
                vpc,
            });
            const portNumber = 5555;
            const port = aws_ec2_1.Port.tcp(portNumber);
            const asgSecurityGroup = renderQueue.asg.connections.securityGroups[0];
            // WHEN
            renderQueue.backendConnections.allowFrom(instance, port);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                Description: `from ${instance.connections.securityGroups[0].uniqueId}:${portNumber}`,
                GroupId: stack.resolve(asgSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(instance.connections.securityGroups[0].securityGroupId),
                FromPort: portNumber,
                ToPort: portNumber,
            });
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let repository;
            beforeEach(() => {
                // GIVEN
                repository = new lib_1.Repository(dependencyStack, 'Repo', {
                    version,
                    vpc,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {},
                };
                // WHEN
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                });
            });
            test('to HTTPS externally between clients and ALB', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                });
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let repository;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                repository = new lib_1.Repository(dependencyStack, 'Repo', {
                    version,
                    vpc,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
                caCertPemLogicalId = stack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = stack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = stack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                });
            });
            test('creates RCS cert', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assertions_1.Match.arrayWith(['Arn']),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'Stack/RenderQueue/TlsCaCertPem',
                    },
                });
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = stack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    'secretsmanager:GetSecretValue',
                                    'secretsmanager:DescribeSecret',
                                ],
                                Effect: 'Allow',
                                Resource: {
                                    'Fn::GetAtt': [
                                        caCertPemLogicalId,
                                        'Cert',
                                    ],
                                },
                            },
                            {
                                Action: [
                                    'secretsmanager:GetSecretValue',
                                    'secretsmanager:DescribeSecret',
                                ],
                                Effect: 'Allow',
                                Resource: {
                                    'Fn::GetAtt': [
                                        caCertPkcsLogicalId,
                                        'Cert',
                                    ],
                                },
                            },
                            {
                                Action: [
                                    'secretsmanager:GetSecretValue',
                                    'secretsmanager:DescribeSecret',
                                ],
                                Effect: 'Allow',
                                Resource: { Ref: caCertPkcsPassphraseLogicalId },
                            },
                        ]),
                        Version: '2012-10-17',
                    },
                    Roles: assertions_1.Match.arrayWith([{ Ref: taskRoleLogicalId }]),
                });
            });
            test('configures environment variables for cert secret URIs', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assertions_1.Match.arrayWith([
                        assertions_1.Match.objectLike({
                            Environment: assertions_1.Match.arrayWith([
                                {
                                    Name: 'RCS_TLS_CA_CERT_URI',
                                    Value: {
                                        'Fn::GetAtt': [
                                            caCertPemLogicalId,
                                            'Cert',
                                        ],
                                    },
                                },
                                {
                                    Name: 'RCS_TLS_CERT_URI',
                                    Value: {
                                        'Fn::GetAtt': [
                                            caCertPkcsLogicalId,
                                            'Cert',
                                        ],
                                    },
                                },
                                {
                                    Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                                    Value: { Ref: caCertPkcsPassphraseLogicalId },
                                },
                            ]),
                        }),
                    ]),
                });
            });
        });
        describe('when internal protocol is HTTP', () => {
            let repository;
            beforeEach(() => {
                // GIVEN
                repository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                });
                const props = {
                    images,
                    repository: repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        externalTLS: { enabled: false },
                    },
                };
                // WHEN
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509Generator', 0);
            });
            test('ALB connects with HTTP to port 8080', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                });
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let repository;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                repository = new lib_1.Repository(dependencyStack, 'Repo', {
                    version,
                    vpc,
                    // Cannot have secrets management unless external TLS is enabled on the RQ,
                    // so we disable it to allow for testing.
                    secretsManagementSettings: { enabled: false },
                });
                const zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        hostname: 'renderqueue',
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                });
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                });
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assertions_1.Match.arrayWith([{
                            CertificateArn: CERT_ARN,
                        }]),
                });
            });
            test('raises an error when a cert is specified without a hosted zone', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueueTest', props);
                })
                    // THEN
                    .toThrow(/The hostname for the render queue must be defined if supplying your own certificates./);
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZoneNoName', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                    hostname: { zone },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueueTest', props);
                })
                    // THEN
                    .toThrow(/A hostname must be supplied if a certificate is supplied, with the common name of the certificate matching the hostname \+ domain name/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            describe('passing cases', () => {
                let repository;
                let zone;
                const ZONE_NAME = 'renderfarm.local';
                const HOSTNAME = 'server';
                beforeEach(() => {
                    // GIVEN
                    repository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                        vpc,
                        version,
                        // Cannot have secrets management unless external TLS is enabled on the RQ,
                        // so we disable it to allow for testing.
                        secretsManagementSettings: { enabled: false },
                    });
                    zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                        vpc,
                        zoneName: ZONE_NAME,
                    });
                    const caCert = new __1.X509CertificatePem(stack, 'CaCert', {
                        subject: {
                            cn: `ca.${ZONE_NAME}`,
                        },
                    });
                    const serverCert = new __1.X509CertificatePem(stack, 'ServerCert', {
                        subject: {
                            cn: `${HOSTNAME}.${ZONE_NAME}`,
                        },
                        signingCertificate: caCert,
                    });
                    const props = {
                        images,
                        repository,
                        version: renderQueueVersion,
                        vpc,
                        trafficEncryption: {
                            externalTLS: {
                                rfdkCertificate: serverCert,
                            },
                            internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        },
                        hostname: {
                            zone,
                            hostname: HOSTNAME,
                        },
                    };
                    // WHEN
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                });
                test('sets the listener port to 4433', () => {
                    // THEN
                    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                        Port: 4433,
                    });
                });
                test('sets the listener protocol to HTTPS', () => {
                    // THEN
                    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                        Protocol: 'HTTPS',
                    });
                });
                test('Imports Cert to ACM', () => {
                    // THEN
                    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_AcmImportedCertificate', {
                        X509CertificatePem: {
                            Cert: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Cert',
                                ],
                            },
                            Key: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Key',
                                ],
                            },
                            Passphrase: {
                                Ref: 'ServerCertPassphraseE4C3CB38',
                            },
                            CertChain: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'CertChain',
                                ],
                            },
                        },
                    });
                });
            });
            describe('failure cases,', () => {
                test('Throws when missing cert chain', () => {
                    const ZONE_NAME = 'renderfarm.local';
                    const HOSTNAME = 'server';
                    // GIVEN
                    const repository = new lib_1.Repository(dependencyStack, 'Repo', {
                        version,
                        vpc,
                    });
                    const zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                        vpc,
                        zoneName: ZONE_NAME,
                    });
                    const rootCert = new __1.X509CertificatePem(stack, 'RootCert', {
                        subject: {
                            cn: `ca.${ZONE_NAME}`,
                        },
                    });
                    const props = {
                        images,
                        repository,
                        version: renderQueueVersion,
                        vpc,
                        trafficEncryption: {
                            externalTLS: {
                                rfdkCertificate: rootCert,
                            },
                            internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        },
                        hostname: {
                            zone,
                            hostname: HOSTNAME,
                        },
                    };
                    // WHEN
                    expect(() => {
                        new lib_1.RenderQueue(stack, 'RenderQueue', props);
                    })
                        // THEN
                        .toThrow(/Provided rfdkCertificate does not contain a certificate chain/);
                });
            });
        });
        test('Creates default RFDK cert if no cert given', () => {
            // GIVEN
            const repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
            });
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
            };
            const rq = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            const rootCa = rq.node.findChild('RootCA');
            const rootCaGen = rootCa.node.defaultChild;
            const rfdkCert = rq.node.findChild('RenderQueuePemCert');
            const rfdkCertGen = rfdkCert.node.defaultChild;
            const acmCert = rq.node.findChild('AcmCert');
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
                Passphrase: stack.resolve(rootCa.passphrase.secretArn),
            });
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
                Passphrase: stack.resolve(rfdkCert.passphrase.secretArn),
                SigningCertificate: {
                    Cert: stack.resolve(rootCaGen.getAtt('Cert')),
                    Key: stack.resolve(rootCaGen.getAtt('Key')),
                    Passphrase: stack.resolve(rootCa.passphrase.secretArn),
                    CertChain: '',
                },
            });
            assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_AcmImportedCertificate', 1);
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_AcmImportedCertificate', {
                X509CertificatePem: {
                    Cert: stack.resolve(rfdkCertGen.getAtt('Cert')),
                    Key: stack.resolve(rfdkCertGen.getAtt('Key')),
                    Passphrase: stack.resolve(rfdkCert.passphrase.secretArn),
                    CertChain: stack.resolve(rfdkCertGen.getAtt('CertChain')),
                },
            });
            assertions_1.Template.fromStack(stack).resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 1);
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
                Certificates: [
                    {
                        CertificateArn: stack.resolve(acmCert.certificateArn),
                    },
                ],
            });
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
            });
            const zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(stack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(stack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const HOSTNAME = 'renderqueue';
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
            });
            const zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    hostname: HOSTNAME,
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let repository;
            let renderQueue;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                repository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                    vpc,
                    version,
                    // Cannot have secrets management unless external TLS is enabled on the RQ,
                    // so we disable it to allow for testing.
                    secretsManagementSettings: { enabled: false },
                });
                zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: { externalTLS: { enabled: false } },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(stack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = renderQueue.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
                assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
                    DependsOn: assertions_1.Match.arrayWith([
                        'RenderQueueLBPublicListenerBBF15D5F',
                        'RenderQueueRCSTaskA9AE70D3',
                    ]),
                });
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(stack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                renderQueue.configureClientInstance({
                    host,
                });
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
                    UserData: {
                        'Fn::Base64': {
                            'Fn::Join': [
                                '',
                                [
                                    '#!/bin/bash\n' +
                                        `mkdir -p $(dirname '/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py')\n` +
                                        'aws s3 cp \'s3://',
                                    {
                                        'Fn::Sub': asset_constants_1.RQ_CONNECTION_ASSET.Bucket,
                                    },
                                    `/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' '/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py'\n` +
                                        'if [ -f "/etc/profile.d/deadlineclient.sh" ]; then\n' +
                                        '  source "/etc/profile.d/deadlineclient.sh"\n' +
                                        'fi\n' +
                                        `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py" --render-queue "http://renderqueue.${ZONE_NAME}:8080" \n` +
                                        `rm -f "/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py"\n` +
                                        'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                        '  service deadline10launcher restart\n' +
                                        'fi',
                                ],
                            ],
                        },
                    },
                });
                // Make sure we execute the script with the correct args
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
                assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
                    DependsOn: assertions_1.Match.arrayWith([
                        'RenderQueueLBPublicListenerBBF15D5F',
                        'RenderQueueRCSTaskA9AE70D3',
                    ]),
                });
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(stack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                renderQueue.configureClientInstance({
                    host,
                });
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
                    UserData: {
                        'Fn::Base64': {
                            'Fn::Join': [
                                '',
                                [
                                    `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' ) -ea 0\n` +
                                        'Read-S3Object -BucketName \'',
                                    {
                                        'Fn::Sub': asset_constants_1.RQ_CONNECTION_ASSET.Bucket,
                                    },
                                    `' -key '${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' -file 'C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' -ErrorAction Stop\n` +
                                        '$ErrorActionPreference = "Stop"\n' +
                                        '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                        `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py" --render-queue "http://renderqueue.${ZONE_NAME}:8080"  2>&1\n` +
                                        `Remove-Item -Path "C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py"\n` +
                                        'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                        '  Restart-Service "deadline10launcherservice"\n' +
                                        '} Else {\n' +
                                        '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                        '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                        '}</powershell>',
                                ],
                            ],
                        },
                    },
                });
                // Make sure we execute the script with the correct args
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
                assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
                    DependsOn: assertions_1.Match.arrayWith([
                        'RenderQueueLBPublicListenerBBF15D5F',
                        'RenderQueueRCSTaskA9AE70D3',
                    ]),
                });
            });
        });
        describe('externalProtocol is https', () => {
            let zone;
            let renderQueue;
            const HOSTNAME = 'renderqueue';
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'arn:a:b:c:dcertarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            beforeEach(() => {
                // GIVEN
                const repository = new lib_1.Repository(dependencyStack, 'Repo', {
                    version,
                    vpc,
                });
                zone = new aws_route53_1.PrivateHostedZone(stack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    hostname: {
                        hostname: HOSTNAME,
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(stack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = renderQueue.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(stack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                renderQueue.configureClientInstance({
                    host,
                });
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
                    UserData: {
                        'Fn::Base64': {
                            'Fn::Join': [
                                '',
                                [
                                    '#!/bin/bash\n' +
                                        `mkdir -p $(dirname '/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py')\n` +
                                        'aws s3 cp \'s3://',
                                    {
                                        'Fn::Sub': asset_constants_1.RQ_CONNECTION_ASSET.Bucket,
                                    },
                                    `/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' '/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py'\n` +
                                        'if [ -f "/etc/profile.d/deadlineclient.sh" ]; then\n' +
                                        '  source "/etc/profile.d/deadlineclient.sh"\n' +
                                        'fi\n' +
                                        `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca "${CA_ARN}"\n` +
                                        `rm -f "/tmp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py"\n` +
                                        'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                        '  service deadline10launcher restart\n' +
                                        'fi',
                                ],
                            ],
                        },
                    },
                });
                // Make sure we execute the script with the correct args
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(stack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                renderQueue.configureClientInstance({
                    host,
                });
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
                    UserData: {
                        'Fn::Base64': {
                            'Fn::Join': [
                                '',
                                [
                                    `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' ) -ea 0\n` +
                                        'Read-S3Object -BucketName \'',
                                    {
                                        'Fn::Sub': asset_constants_1.RQ_CONNECTION_ASSET.Bucket,
                                    },
                                    `' -key '${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' -file 'C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py' -ErrorAction Stop\n$ErrorActionPreference = "Stop"\n` +
                                        '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                        `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca "${CA_ARN}" 2>&1\n` +
                                        `Remove-Item -Path "C:/temp/${asset_constants_1.RQ_CONNECTION_ASSET.Key}.py"\n` +
                                        'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                        '  Restart-Service "deadline10launcherservice"\n' +
                                        '} Else {\n' +
                                        '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                        '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                        '}</powershell>',
                                ],
                            ],
                        },
                    },
                });
                // Make sure we execute the script with the correct args
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            stack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                });
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        const props = {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
            vpcSubnets: {
                subnets,
            },
            vpcSubnetsAlb: {
                subnets,
            },
        };
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', props);
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assertions_1.Match.arrayWith([
                'SubnetID1',
                'SubnetID2',
            ]),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                'SubnetID1',
                'SubnetID2',
            ],
        });
    });
    test('can specify instance type', () => {
        // GIVEN
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version: renderQueueVersion,
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', props);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        });
    });
    test('no deletion protection', () => {
        // GIVEN
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        const props = {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
            deletionProtection: false,
        };
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', props);
        // THEN
        test_helper_1.resourcePropertiesCountIs(stack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ]),
            Scheme: assertions_1.Match.absent(),
            Type: assertions_1.Match.absent(),
        }, 0);
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        const props = {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', props);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'routing.http.drop_invalid_header_fields.enabled',
                    Value: 'true',
                },
            ]),
        });
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        let repository;
        beforeEach(() => {
            repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
                // Cannot have secrets management unless external TLS is enabled on the RQ,
                // so we disable it to allow for testing.
                secretsManagementSettings: { enabled: false },
            });
        });
        describe('not specified, with no TLS', () => {
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: { externalTLS: { enabled: false } },
                };
                // WHEN
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Route53::RecordSet', 0);
            });
        });
        test('not specified, with TLS', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
            };
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Route53::RecordSet', {
                Name: 'renderqueue.aws-rfdk.com.',
                Type: 'A',
                AliasTarget: assertions_1.Match.objectLike({
                    HostedZoneId: stack.resolve(renderQueue.loadBalancer.loadBalancerCanonicalHostedZoneId),
                }),
            });
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assertions_1.Match.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                });
            });
        });
        test.each([
            [false],
            [true],
        ])('specified with TLS enabled == %s', (isTlsEnabled) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                vpc,
                zoneName,
            });
            const hostname = 'testrq';
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
                trafficEncryption: {
                    externalTLS: { enabled: isTlsEnabled },
                },
            };
            // WHEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
            // THEN
            const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Route53::RecordSet', {
                Name: `${hostname}.${zoneName}.`,
                Type: 'A',
                AliasTarget: assertions_1.Match.objectLike({
                    HostedZoneId: {
                        'Fn::GetAtt': [
                            loadBalancerLogicalId,
                            'CanonicalHostedZoneID',
                        ],
                    },
                }),
            });
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let repository;
        let renderQueueProps;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const localVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            const localVersion = new lib_1.VersionQuery(isolatedStack, 'Version');
            const localImages = {
                remoteConnectionServer: rcsImage,
            };
            repository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: localVersion,
                vpc: localVpc,
            });
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
            renderQueueProps = {
                images: localImages,
                repository,
                version: localVersion,
                vpc: localVpc,
            };
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                ...renderQueueProps,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assertions_1.Match.arrayWith([
                    {
                        Key: 'access_logs.s3.enabled',
                        Value: 'true',
                    },
                    {
                        Key: 'access_logs.s3.bucket',
                        Value: {
                            Ref: 'AccessBucketE2803D76',
                        },
                    },
                ]),
            });
            assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: 's3:PutObject',
                            Condition: {
                                StringEquals: {
                                    's3:x-amz-acl': 'bucket-owner-full-control',
                                },
                            },
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: 's3:GetBucketAcl',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::GetAtt': [
                                    'AccessBucketE2803D76',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                's3:PutObject',
                                's3:PutObjectLegalHold',
                                's3:PutObjectRetention',
                                's3:PutObjectTagging',
                                's3:PutObjectVersionTagging',
                                's3:Abort*',
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':iam::127311923021:root',
                                        ],
                                    ],
                                },
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/AWSLogs/',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: 's3:PutObject',
                            Condition: {
                                StringEquals: {
                                    's3:x-amz-acl': 'bucket-owner-full-control',
                                },
                            },
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/AWSLogs/',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                    ]),
                },
            });
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                ...renderQueueProps,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assertions_1.Match.arrayWith([
                    {
                        Key: 'access_logs.s3.enabled',
                        Value: 'true',
                    },
                    {
                        Key: 'access_logs.s3.bucket',
                        Value: {
                            Ref: 'AccessBucketE2803D76',
                        },
                    },
                    {
                        Key: 'access_logs.s3.prefix',
                        Value: 'PREFIX_STRING',
                    },
                ]),
            });
            assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: 's3:PutObject',
                            Condition: {
                                StringEquals: {
                                    's3:x-amz-acl': 'bucket-owner-full-control',
                                },
                            },
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: 's3:GetBucketAcl',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::GetAtt': [
                                    'AccessBucketE2803D76',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                's3:PutObject',
                                's3:PutObjectLegalHold',
                                's3:PutObjectRetention',
                                's3:PutObjectTagging',
                                's3:PutObjectVersionTagging',
                                's3:Abort*',
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':iam::127311923021:root',
                                        ],
                                    ],
                                },
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/PREFIX_STRING/AWSLogs/',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: 's3:PutObject',
                            Condition: {
                                StringEquals: {
                                    's3:x-amz-acl': 'bucket-owner-full-control',
                                },
                            },
                            Effect: 'Allow',
                            Principal: {
                                Service: 'delivery.logs.amazonaws.com',
                            },
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'AccessBucketE2803D76',
                                                'Arn',
                                            ],
                                        },
                                        '/PREFIX_STRING/AWSLogs/',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        },
                    ]),
                },
            });
        });
    });
    describe('tagging', () => {
        let repository;
        beforeEach(() => {
            repository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                vpc,
                version,
            });
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            };
            // WHEN
            new lib_1.RenderQueue(stack, 'RenderQueue', props);
        });
        tag_helpers_1.testConstructTags({
            constructName: 'RenderQueue',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 2,
                'AWS::IAM::Role': 10,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 6,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::DynamoDB::Table': 5,
                'AWS::SecretsManager::Secret': 4,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
    describe('SEP Policies', () => {
        let repository;
        let renderQueue;
        beforeEach(() => {
            // GIVEN
            repository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                vpc,
                version,
            });
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            };
            renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', props);
        });
        test('with resource tracker', () => {
            // WHEN
            renderQueue.addSEPPolicies();
            // THEN
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Role', {
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                            ],
                        ],
                    },
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                            ],
                        ],
                    },
                ]),
            }, 1);
        });
        test('no resource tracker', () => {
            // WHEN
            renderQueue.addSEPPolicies(false);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Role', {
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                            ],
                        ],
                    },
                ]),
            });
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Role', {
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                            ],
                        ],
                    },
                ]),
            }, 0);
        });
    });
    describe('Security Groups', () => {
        let repository;
        let backendSecurityGroup;
        let frontendSecurityGroup;
        beforeEach(() => {
            repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
            });
            backendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'ASGSecurityGroup', { vpc });
            frontendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'LBSecurityGroup', { vpc });
        });
        test('adds security groups on construction', () => {
            // GIVEN
            const securityGroups = {
                backend: backendSecurityGroup,
                frontend: frontendSecurityGroup,
            };
            // WHEN
            new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                securityGroups,
            });
            // THEN
            assertSecurityGroupsWereAdded(securityGroups);
        });
        test('adds backend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                backend: backendSecurityGroup,
            });
        });
        test('adds frontend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                frontend: frontendSecurityGroup,
            });
        });
        test('security groups added post-construction are not attached to Connections object', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            const peerSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'PeerSecurityGroup', { vpc });
            // WHEN
            renderQueue.connections.allowFrom(peerSecurityGroup, aws_ec2_1.Port.tcp(22));
            // THEN
            // Existing LoadBalancer security groups shouldn't have the ingress rule added
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(frontendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }, 0);
            // Existing AutoScalingGroup security groups shouldn't have the ingress rule added
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(backendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }, 0);
        });
        function assertSecurityGroupsWereAdded(securityGroups) {
            if (securityGroups.backend !== undefined) {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(securityGroups.backend.securityGroupId)]),
                });
            }
            if (securityGroups.frontend !== undefined) {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                    SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(securityGroups.frontend.securityGroupId)]),
                });
            }
        }
    });
    test('validates VersionQuery is not in a different stack', () => {
        // GIVEN
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', {
            images,
            repository,
            version,
            vpc,
        });
        // WHEN
        function synth() {
            app.synth();
        }
        // THEN
        expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
    });
    test('Enables filesystem cache if required', () => {
        // GIVEN
        const repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        // WHEN
        new lib_1.RenderQueue(stack, 'RenderQueue', {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
            enableLocalFileCaching: true,
        });
        // THEN
        // Note: If this test breaks/fails, then it is probable that the
        //  'Does not enable filesystem cache by default' test above will also require
        //  updating/fixing.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        assertions_1.Match.arrayWith([
                            assertions_1.Match.stringLikeRegexp('.*# RenderQueue file caching enabled.*'),
                        ]),
                    ],
                },
            },
        });
    });
    describe('Secrets Management', () => {
        let repository;
        let rqSecretsManagementProps;
        beforeEach(() => {
            repository = new lib_1.Repository(dependencyStack, 'Repo', {
                version,
                vpc,
                secretsManagementSettings: {
                    enabled: true,
                },
            });
            rqSecretsManagementProps = {
                vpc,
                images,
                repository,
                version: renderQueueVersion,
                trafficEncryption: {
                    internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    externalTLS: { enabled: true },
                },
            };
        });
        test('throws if internal protocol is not HTTPS', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                trafficEncryption: {
                    internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                },
            }))
                // THEN
                .toThrowError(/The internal protocol on the Render Queue is not HTTPS./);
        });
        test('throws if external TLS is not enabled', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                trafficEncryption: {
                    externalTLS: { enabled: false },
                },
            }))
                // THEN
                .toThrowError(/External TLS on the Render Queue is not enabled./);
        });
        test('throws if repository does not have SM credentials', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                repository: {
                    ...repository,
                    secretsManagementSettings: {
                        ...repository.secretsManagementSettings,
                        credentials: undefined,
                    },
                },
            }))
                // THEN
                .toThrowError(/The Repository does not have Secrets Management credentials/);
        });
        test('throws if deadline version is too low', () => {
            // GIVEN
            const oldVersion = new lib_1.VersionQuery(new aws_cdk_lib_1.Stack(app, 'OldDeadlineVersionStack'), 'OldDeadlineVersion', { version: '10.0.0.0' });
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                version: oldVersion,
            }))
                // THEN
                /* eslint-disable-next-line dot-notation */
                .toThrowError(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        });
        test('grants read permissions to secrets management credentials', () => {
            // WHEN
            const rq = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: assertions_1.Match.objectLike({
                    Statement: assertions_1.Match.arrayWith([{
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(repository.secretsManagementSettings.credentials.node.defaultChild.ref),
                        }]),
                }),
                Roles: [stack.resolve(rq.node.tryFindChild('RCSTask').taskRole.roleName)],
            });
        });
        test('defines secrets management credentials environment variable', () => {
            // WHEN
            new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        Environment: assertions_1.Match.arrayWith([
                            {
                                Name: 'RCS_SM_CREDENTIALS_URI',
                                Value: stack.resolve(repository.secretsManagementSettings.credentials.node.defaultChild.ref),
                            },
                        ]),
                    }),
                ]),
            });
        });
        test('creates and mounts docker volume for deadline key pairs', () => {
            // WHEN
            new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        MountPoints: assertions_1.Match.arrayWith([
                            {
                                ContainerPath: '/home/ec2-user/.config/.mono/keypairs',
                                ReadOnly: false,
                                SourceVolume: 'deadline-user-keypairs',
                            },
                        ]),
                    }),
                ]),
                Volumes: assertions_1.Match.arrayWith([
                    {
                        DockerVolumeConfiguration: {
                            Autoprovision: true,
                            Driver: 'local',
                            Scope: 'shared',
                        },
                        Name: 'deadline-user-keypairs',
                    },
                ]),
            });
        });
        test('DeploymentInstance uses specified backend security group', () => {
            // GIVEN
            const backendSecurityGroupId = 'backend-sg-id';
            const backendSecurityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'BackendSG', backendSecurityGroupId);
            // WHEN
            const renderQueue = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                securityGroups: {
                    backend: backendSecurityGroup,
                },
            });
            // Force creation of the DeploymentInstance
            // eslint-disable-next-line dot-notation
            renderQueue['deploymentInstance'];
            // THEN
            const deploymentInstance = renderQueue.node.findChild('ConfigureRepository');
            expect(deploymentInstance.connections.securityGroups[0].securityGroupId).toEqual(backendSecurityGroupId);
        });
        test('DeploymentInstance uses RQ\'s log group prefix', () => {
            // GIVEN
            rqSecretsManagementProps = {
                ...rqSecretsManagementProps,
                logGroupProps: {
                    logGroupPrefix: '/customPrefix/',
                },
            };
            // WHEN
            const renderQueue = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                logGroupProps: {
                    logGroupPrefix: '/customPrefix/',
                },
            });
            // Force creation of the DeploymentInstance
            // eslint-disable-next-line dot-notation
            renderQueue['deploymentInstance'];
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
                LogGroupName: '/customPrefix/ConfigureRepository',
            });
        });
        test('DeploymentInstance uses implicitly created backend security group', () => {
            // WHEN
            const renderQueue = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // Force creation of the DeploymentInstance
            // eslint-disable-next-line dot-notation
            renderQueue['deploymentInstance'];
            // THEN
            const deploymentInstance = renderQueue.node.findChild('ConfigureRepository');
            expect(deploymentInstance.connections.securityGroups[0]).toBe(renderQueue.backendConnections.securityGroups[0]);
            expect(deploymentInstance.connections.securityGroups[0]).toBe(renderQueue.asg.connections.securityGroups[0]);
        });
        describe('client calls .configureSecretsManagementAutoRegistration()', () => {
            let callParams;
            let clientInstance;
            let identityRegistrationSettings;
            let launchConfiguration;
            let rqVpcSubnets;
            const RQ_SUBNET_IDS = ['SubnetID1', 'SubnetID2'];
            beforeEach(() => {
                // GIVEN
                const subnets = [
                    aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                        subnetId: RQ_SUBNET_IDS[0],
                        availabilityZone: 'us-west-2a',
                    }),
                    aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                        subnetId: RQ_SUBNET_IDS[1],
                        availabilityZone: 'us-west-2b',
                    }),
                ];
                rqVpcSubnets = {
                    subnets,
                };
                const rq = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                    ...rqSecretsManagementProps,
                    vpcSubnets: rqVpcSubnets,
                });
                clientInstance = new aws_ec2_1.Instance(stack, 'ClientInstance', {
                    instanceType: new aws_ec2_1.InstanceType('t3.micro'),
                    machineImage: new aws_ec2_1.AmazonLinuxImage(),
                    vpc,
                });
                callParams = {
                    dependent: clientInstance,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
                };
                launchConfiguration = (
                // @ts-ignore
                rq.deploymentInstance
                    .node.findChild('ASG')
                    .node.findChild('LaunchConfig'));
                // @ts-ignore
                identityRegistrationSettings = rq.identityRegistrationSettings;
                jest.spyOn(identityRegistrationSettings, 'addSubnetIdentityRegistrationSetting');
                // WHEN
                rq.configureSecretsManagementAutoRegistration(callParams);
            });
            test('registration is delegated to SecretsManagementIdentityRegistration', () => {
                // THEN
                expect(identityRegistrationSettings.addSubnetIdentityRegistrationSetting).toHaveBeenCalledWith(callParams);
            });
            test('deployment instance is created using specified subnets', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    LaunchConfigurationName: stack.resolve(launchConfiguration.ref),
                    VPCZoneIdentifier: assertions_1.Match.arrayWith([
                        ...RQ_SUBNET_IDS,
                    ]),
                });
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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