"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_docdb_1 = require("aws-cdk-lib/aws-docdb");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const sinon = require("sinon");
const lib_1 = require("../../core/lib");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('DocumentDB', () => {
    let stack;
    let vpc;
    let database;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            backup: {
                retention: aws_cdk_lib_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        if (!database.secret) {
            throw new Error('secret cannot be null');
        }
    });
    test('Grants access to Document DB Secret', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.grantRead(role);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'DbClusterSecretAttachment4201A1ED' },
                    }],
            },
        });
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=(');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            ToPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
        });
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
            DependsOn: [
                'DbClusterInstance155835CE5',
                'InstanceInstanceRoleE9785DE5',
            ],
        });
    });
    test('add child dependency to attributes', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
            ],
        });
    });
    test('add child dependency throws when cluster implementation changed', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        docdb.node.defaultChild = database; // Trick addChildDependency() into thinking this is a real construct.
        // THEN
        expect(() => {
            connection.addChildDependency(instance);
        }).toThrowError(/The internal implementation of the AWS CDK's DocumentDB cluster construct may have changed./);
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds warning annotation when a security group cannot be added due to unsupported IDatabaseCluster implementation', () => {
        // GIVEN
        class FakeDatabaseCluster extends aws_cdk_lib_1.Resource {
            constructor(scope, id) {
                super(scope, id);
                this.clusterIdentifier = '';
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new aws_docdb_1.Endpoint('address', 123);
                this.clusterReadEndpoint = new aws_docdb_1.Endpoint('readAddress', 123);
                this.instanceEndpoints = [];
                this.securityGroupId = '';
                this.connections = new aws_ec2_1.Connections();
                this.stack = aws_cdk_lib_1.Stack.of(scope);
                this.env = { account: this.stack.account, region: this.stack.region };
            }
            asSecretAttachmentTarget() {
                throw new Error('Method not implemented.');
            }
        }
        const fakeDatabase = new FakeDatabaseCluster(stack, 'FakeDatabase');
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: fakeDatabase, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        assertions_1.Annotations.fromStack(stack).hasWarning(`/${fakeDatabase.node.path}`, assertions_1.Match.stringLikeRegexp(`Failed to add the following security groups to ${fakeDatabase.node.id}: .*\\. ` +
            'The \\"database\\" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.'));
    });
    // This test can be removed once the following CDK PR is merged:
    // https://github.com/aws/aws-cdk/pull/13290
    test('adds warning annotation when a security group cannot be added due to implementation changes in DatabaseCluster', () => {
        // GIVEN
        if (!database.node.tryRemoveChild('Resource')) {
            throw new Error('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed. The addSecurityGroup method needs to be updated.');
        }
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        assertions_1.Annotations.fromStack(stack).hasWarning(`/${database.node.path}`, assertions_1.Match.stringLikeRegexp(`Failed to add the following security groups to ${database.node.id}: .*\\. ` +
            'The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.'));
    });
    test('Document DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(connection.databaseConstruct).toEqual(database);
    });
});
describe('DocumentDB Version Checks', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    });
    test('Compatible version', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            backup: {
                retention: aws_cdk_lib_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${database.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${database.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoError(`/${database.node.path}`, assertions_1.Match.anyValue());
    });
    test('When from attributes', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
        const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
            clusterEndpointAddress: '1.2.3.4',
            clusterIdentifier: 'foo',
            instanceEndpointAddresses: ['1.2.3.5'],
            instanceIdentifiers: ['i0'],
            port: 27001,
            readerEndpointAddress: '1.2.3.6',
            securityGroup: sg,
        });
        // WHEN
        const databaseConnection = lib_2.DatabaseConnection.forDocDB({ database, login: secret });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${database.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${database.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoError(`/${database.node.path}`, assertions_1.Match.anyValue());
        expect(databaseConnection.databaseConstruct).toBeUndefined();
    });
    test('No engineVersion given', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            backup: {
                retention: aws_cdk_lib_1.Duration.days(15),
            },
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasError(`/${database.node.path}`, 'engineVersion must be 3.6.0 to be compatible with Deadline');
    });
    test('engineVersion not 3.6.0', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            backup: {
                retention: aws_cdk_lib_1.Duration.days(15),
            },
            engineVersion: '4.0.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasError(`/${database.node.path}`, 'engineVersion must be 3.6.0 to be compatible with Deadline');
    });
});
describe('MongoDB', () => {
    let stack;
    let vpc;
    let database;
    let clientCert;
    let dnsZone;
    let serverCert;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const hostname = 'mongo';
        const zoneName = 'deadline.internal';
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        clientCert = new lib_1.X509CertificatePem(stack, 'ClientCert', {
            subject: {
                cn: 'dbuser',
                ou: 'TestClient',
            },
            signingCertificate: caCert,
        });
        database = new lib_1.MongoDbInstance(stack, 'MongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname,
                dnsZone,
                serverCertificate: serverCert,
            },
        });
    });
    test('Grants access to certificate Secrets', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        connection.grantRead(role);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'ClientCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: 'ClientCertPassphrase8A71E1E1',
                        },
                    },
                ],
            },
        });
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`'/tmp/${asset_constants_1.GET_SECRET_TO_FILE_SCRIPT_LINUX.Key}.sh' ${token} /opt/Thinkbox/certs/mongo_client.pfx`)));
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true');
        expect(userData).toContain('["--dbhost"]="mongo.deadline.internal"');
        expect(userData).toContain('["--dbport"]=27017');
        expect(userData).toContain('["--dbclientcert"]=');
        expect(userData).toContain('["--dbcertpass"]=$CERT_PASSWORD');
    });
    test('addConnectionDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addConnectionDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`'/tmp/${asset_constants_1.GET_SECRET_TO_FILE_SCRIPT_LINUX.Key}.sh' ${token} /opt/Thinkbox/certs/mongo_client.pfx`)));
        expect(userData).toContain('configure_deadline_database(){\n');
        expect(userData).toContain('\nexport -f configure_deadline_database');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nexport DB_CERT_FILE=');
        expect(userData).toContain('\nexport DB_CERT_PASSWORD=');
    });
    test('defines required container environment variables', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_URI');
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_PASSWORD_URI');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 27017,
            ToPort: 27017,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'MongoDbServerAsgInstanceSecurityGroupCE623335',
                    'GroupId',
                ],
            },
        });
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
                'MongoDbServerAsgASG47B3D94E',
            ],
        });
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds security group', () => {
        // GIVEN
        const dbSpy = sinon.spy(database, 'addSecurityGroup');
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(dbSpy.calledOnce).toBeTruthy();
    });
    test('Mongo DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toEqual(database.mongoDataVolume);
    });
    test('Mongo DB imported from attributes', () => {
        // GIVEN
        const volume = aws_ec2_1.Volume.fromVolumeAttributes(stack, 'Volume', {
            availabilityZone: 'dummy zone',
            volumeId: 'vol-05abe246af',
        });
        const mongoDB = new lib_1.MongoDbInstance(stack, 'ImportedMongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname: 'mongo',
                dnsZone,
                serverCertificate: serverCert,
                mongoDataVolume: { volume },
            },
        });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database: mongoDB, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toBeUndefined();
    });
});
//# sourceMappingURL=data:application/json;base64,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