"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = exports.InstanceUserDataProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const constructs_1 = require("constructs");
const core_1 = require("../../core");
const version_1 = require("./version");
/**
 * Implementation of {@link IInstanceUserDataProvider}.
 * Can be used as sub-class with override the desired methods
 * to add custom user data commands for WorkerInstanceFleet or WorkerInstanceConfiguration.
 */
class InstanceUserDataProvider extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * @inheritdoc
     */
    preCloudWatchAgent(_host) {
    }
    /**
     * @inheritdoc
     */
    preRenderQueueConfiguration(_host) {
    }
    /**
     * @inheritdoc
     */
    preWorkerConfiguration(_host) {
    }
    /**
     * @inheritdoc
     */
    postWorkerLaunch(_host) {
    }
}
exports.InstanceUserDataProvider = InstanceUserDataProvider;
_a = JSII_RTTI_SYMBOL_1;
InstanceUserDataProvider[_a] = { fqn: "aws-rfdk.deadline.InstanceUserDataProvider", version: "1.1.0" };
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their
 * log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * This configuration performs the following steps in order:
 * 1) Configure Cloud Watch Agent
 * 2) Configure Deadline Worker RenderQueue connection
 * 3) Configure Deadline Worker settings
 *
 * A `userDataProvider` can be specified that defines callback functions.
 * These callbacks can be used to inject user data commands at different points during the Worker instance configuration.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class WorkerInstanceConfiguration extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        props.userDataProvider?.preCloudWatchAgent(props.worker);
        if (props.cloudWatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudWatchLogSettings, props.shouldInstallCloudWatchAgent);
        }
        props.userDataProvider?.preRenderQueueConfiguration(props.worker);
        props.renderQueue?.configureClientInstance({
            host: props.worker,
            // Don't restart the Deadline Launcher service after configuring the connection to the Render Queue. We need to
            // restart it later anyways, and the Windows service for the Deadline Launcher can get locked in the "stopping"
            // state if you attempt to stop or restart it while it is still restarting. This can cause the user data execution
            // to get locked waiting for the service to finish stopping/restarting.
            restartLauncher: false,
        });
        props.userDataProvider?.preWorkerConfiguration(props.worker);
        this.listenerPort = props.workerSettings?.listenerPort ?? WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT;
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
        props.userDataProvider?.postWorkerLaunch(props.worker);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch
     * service. The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *   for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure. This can be an instance, auto scaling group, launch template, etc.
     * @param id Identifier to disambiguate the resources that are created.
     * @param shouldInstallAgent Boolean for if the worker's User Data should attempt to install the CloudWatch agent
     * @param logGroupProps Configuration for the log group in CloudWatch.
     */
    configureCloudWatchLogStream(worker, id, logGroupProps, shouldInstallAgent) {
        const logGroup = core_1.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_1.CloudWatchConfigBuilder(aws_cdk_lib_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_1.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
            shouldInstallAgent,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance. From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure. This can be an instance, auto scaling group, launch template, etc.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     */
    configureWorkerSettings(worker, id, settings) {
        const configureWorkerScriptAsset = core_1.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        const configureWorkerPortAsset = new aws_s3_assets_1.Asset(this, `${id}WorkerListenerScript`, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'worker-listening-port.py'),
        });
        const configWorkerPortLocalPath = worker.userData.addS3DownloadCommand({
            bucket: configureWorkerPortAsset.bucket,
            bucketKey: configureWorkerPortAsset.s3ObjectKey,
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = settings?.groups?.map(val => val.toLowerCase()).join(',') ?? '';
        const pools = settings?.pools?.map(val => val.toLowerCase()).join(',') ?? '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${settings?.region ?? ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                this.listenerPort.toString(),
                configWorkerPortLocalPath,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
_b = JSII_RTTI_SYMBOL_1;
WorkerInstanceConfiguration[_b] = { fqn: "aws-rfdk.deadline.WorkerInstanceConfiguration", version: "1.1.0" };
/**
 * The default port to use for a worker to listen on for remote commands.
 */
WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT = 56032;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2VyLWNvbmZpZ3VyYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ3b3JrZXItY29uZmlndXJhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7R0FHRztBQUVILDZCQUE2QjtBQUU3Qiw2Q0FFcUI7QUFDckIsaURBRTZCO0FBQzdCLDZEQUFrRDtBQUNsRCwyQ0FBdUM7QUFDdkMscUNBTW9CO0FBT3BCLHVDQUVtQjtBQTZCbkI7Ozs7R0FJRztBQUNILE1BQWEsd0JBQXlCLFNBQVEsc0JBQVM7SUFDckQsWUFBWSxLQUFnQixFQUFFLEVBQVU7UUFDdEMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNuQixDQUFDO0lBRUQ7O09BRUc7SUFDSCxrQkFBa0IsQ0FBQyxLQUFZO0lBQy9CLENBQUM7SUFFRDs7T0FFRztJQUNILDJCQUEyQixDQUFDLEtBQVk7SUFDeEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsc0JBQXNCLENBQUMsS0FBWTtJQUNuQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxnQkFBZ0IsQ0FBQyxLQUFZO0lBQzdCLENBQUM7O0FBM0JILDREQTRCQzs7O0FBc0ZEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FxQkc7QUFDSCxNQUFhLDJCQUE0QixTQUFRLHNCQUFTO0lBV3hELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUM7UUFDL0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixLQUFLLENBQUMsZ0JBQWdCLEVBQUUsa0JBQWtCLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pELElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO1lBQy9CLElBQUksQ0FBQyw0QkFBNEIsQ0FDL0IsS0FBSyxDQUFDLE1BQU0sRUFDWixFQUFFLEVBQ0YsS0FBSyxDQUFDLHFCQUFxQixFQUMzQixLQUFLLENBQUMsNEJBQTRCLENBQ25DLENBQUM7U0FDSDtRQUNELEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSwyQkFBMkIsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDbEUsS0FBSyxDQUFDLFdBQVcsRUFBRSx1QkFBdUIsQ0FBQztZQUN6QyxJQUFJLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDbEIsK0dBQStHO1lBQy9HLCtHQUErRztZQUMvRyxrSEFBa0g7WUFDbEgsdUVBQXVFO1lBQ3ZFLGVBQWUsRUFBRSxLQUFLO1NBQ3ZCLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFN0QsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsY0FBYyxFQUFFLFlBQVksSUFBSSwyQkFBMkIsQ0FBQyxxQkFBcUIsQ0FBQztRQUM1RyxJQUFJLENBQUMsdUJBQXVCLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXJFLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNPLDRCQUE0QixDQUNwQyxNQUFhLEVBQ2IsRUFBVSxFQUNWLGFBQW1DLEVBQ25DLGtCQUE0QjtRQUU1QixNQUFNLFFBQVEsR0FBRyxzQkFBZSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLGlCQUFpQixFQUFFLEVBQUUsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVoRyxRQUFRLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTVCLE1BQU0sOEJBQThCLEdBQUcsSUFBSSw4QkFBdUIsQ0FBQyxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXpGLElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyw2QkFBbUIsQ0FBQyxPQUFPLEVBQUU7WUFDakQsOEJBQThCLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLFlBQVksRUFDckUsbUJBQW1CLEVBQ25CLDBFQUEwRSxDQUFDLENBQUM7WUFDOUUsOEJBQThCLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLFlBQVksRUFDckUsWUFBWSxFQUNaLGlFQUFpRSxDQUFDLENBQUM7WUFDckUsOEJBQThCLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLFlBQVksRUFDckUsY0FBYyxFQUNkLG9FQUFvRSxDQUFDLENBQUM7U0FDekU7YUFBTTtZQUNMLDhCQUE4QixDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQ3JFLG1CQUFtQixFQUNuQixnQ0FBZ0MsQ0FBQyxDQUFDO1lBQ3BDLDhCQUE4QixDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQ3JFLFlBQVksRUFDWixpREFBaUQsQ0FBQyxDQUFDO1lBQ3JELDhCQUE4QixDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQ3JFLGNBQWMsRUFDZCxvREFBb0QsQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsSUFBSSxzQkFBZSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFO1lBQzNDLGdCQUFnQixFQUFFLDhCQUE4QixDQUFDLCtCQUErQixFQUFFO1lBQ2xGLElBQUksRUFBRSxNQUFNO1lBQ1osa0JBQWtCO1NBQ25CLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNPLHVCQUF1QixDQUFDLE1BQWEsRUFBRSxFQUFVLEVBQUUsUUFBeUI7UUFDcEYsTUFBTSwwQkFBMEIsR0FBRyxrQkFBVyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO1lBQzNGLE1BQU0sRUFBRSxNQUFNLENBQUMsTUFBTTtZQUNyQixRQUFRLEVBQUUsaUJBQWlCO1lBQzNCLE9BQU8sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUNoQixTQUFTLEVBQ1QsSUFBSSxFQUNKLFVBQVUsQ0FDWDtTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxxQkFBSyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsc0JBQXNCLEVBQUU7WUFDNUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLDBCQUEwQixDQUFDO1NBQ2xGLENBQUMsQ0FBQztRQUVILE1BQU0seUJBQXlCLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztZQUNyRSxNQUFNLEVBQUUsd0JBQXdCLENBQUMsTUFBTTtZQUN2QyxTQUFTLEVBQUUsd0JBQXdCLENBQUMsV0FBVztTQUNoRCxDQUFDLENBQUM7UUFFSCwwRkFBMEY7UUFDMUYsTUFBTSxNQUFNLEdBQUcsUUFBUSxFQUFFLE1BQU0sRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQy9FLE1BQU0sS0FBSyxHQUFHLFFBQVEsRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUU3RSwwQkFBMEIsQ0FBQyxTQUFTLENBQUM7WUFDbkMsSUFBSSxFQUFFLE1BQU07WUFDWixJQUFJLEVBQUU7Z0JBQ0osSUFBSSxNQUFNLEdBQUc7Z0JBQ2IsSUFBSSxLQUFLLEdBQUc7Z0JBQ1osSUFBSSxRQUFRLEVBQUUsTUFBTSxJQUFJLEVBQUUsR0FBRztnQkFDN0IsSUFBSSxpQkFBTyxDQUFDLGtDQUFrQyxDQUFDLFFBQVEsRUFBRSxHQUFHO2dCQUM1RCxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtnQkFDNUIseUJBQXlCO2FBQzFCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUF4SUgsa0VBeUlDOzs7QUF4SUM7O0dBRUc7QUFDcUIsaURBQXFCLEdBQUcsS0FBSyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcblxuaW1wb3J0IHtcbiAgRHVyYXRpb24sXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7XG4gIE9wZXJhdGluZ1N5c3RlbVR5cGUsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0IHsgQXNzZXQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMtYXNzZXRzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHtcbiAgQ2xvdWRXYXRjaEFnZW50LFxuICBDbG91ZFdhdGNoQ29uZmlnQnVpbGRlcixcbiAgTG9nR3JvdXBGYWN0b3J5LFxuICBMb2dHcm91cEZhY3RvcnlQcm9wcyxcbiAgU2NyaXB0QXNzZXQsXG59IGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHtcbiAgSUhvc3QsXG59IGZyb20gJy4vaG9zdC1yZWYnO1xuaW1wb3J0IHtcbiAgSVJlbmRlclF1ZXVlLFxufSBmcm9tICcuL3JlbmRlci1xdWV1ZSc7XG5pbXBvcnQge1xuICBWZXJzaW9uLFxufSBmcm9tICcuL3ZlcnNpb24nO1xuXG4vKipcbiAqIFByb3ZpZGVyIGZvciBhZGRpbmcgdXNlciBkYXRhIHNjcmlwdHNcbiAqIE1ldGhvZHMgb2YgdGhpcyBpbnRlcmZhY2Ugd2lsbCBiZSBpbnZva2VkIGluIFdvcmtlckluc3RhbmNlQ29uZmlndXJhdGlvblxuICogb24gZGlmZmVyZW50IHN0YWdlcyBvZiB3b3JrZXIgY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIElJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXIge1xuICAvKipcbiAgICogTWV0aG9kIHRoYXQgaXMgaW52b2tlZCBiZWZvcmUgY29uZmlndXJpbmcgdGhlIENsb3VkIFdhdGNoIEFnZW50LlxuICAgKi9cbiAgcHJlQ2xvdWRXYXRjaEFnZW50KGhvc3Q6IElIb3N0KTogdm9pZDtcblxuICAvKipcbiAgICogTWV0aG9kIHRoYXQgaXMgaW52b2tlZCBiZWZvcmUgdGhlIHJlbmRlciBxdWV1ZSBjb25maWd1cmF0aW9uLlxuICAgKi9cbiAgcHJlUmVuZGVyUXVldWVDb25maWd1cmF0aW9uKGhvc3Q6IElIb3N0KTogdm9pZDtcblxuICAvKipcbiAgICogTWV0aG9kIHRoYXQgaXMgaW52b2tlZCBhZnRlciBjb25maWd1cmluZyB0aGUgY29ubmVjdGlvbiB0byB0aGUgcmVuZGVyIHF1ZXVlIGFuZCBiZWZvcmUgY29uZmlndXJpbmcgdGhlIERlYWRsaW5lIFdvcmtlci5cbiAgICovXG4gIHByZVdvcmtlckNvbmZpZ3VyYXRpb24oaG9zdDogSUhvc3QpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBNZXRob2QgdGhhdCBpcyBpbnZva2VkIGFmdGVyIGFsbCBjb25maWd1cmF0aW9uIGlzIGRvbmUgYW5kIHdvcmtlciBzdGFydGVkLlxuICAgKi9cbiAgcG9zdFdvcmtlckxhdW5jaChob3N0OiBJSG9zdCk6IHZvaWQ7XG59XG5cbi8qKlxuICogSW1wbGVtZW50YXRpb24gb2Yge0BsaW5rIElJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXJ9LlxuICogQ2FuIGJlIHVzZWQgYXMgc3ViLWNsYXNzIHdpdGggb3ZlcnJpZGUgdGhlIGRlc2lyZWQgbWV0aG9kc1xuICogdG8gYWRkIGN1c3RvbSB1c2VyIGRhdGEgY29tbWFuZHMgZm9yIFdvcmtlckluc3RhbmNlRmxlZXQgb3IgV29ya2VySW5zdGFuY2VDb25maWd1cmF0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgSW5zdGFuY2VVc2VyRGF0YVByb3ZpZGVyIGV4dGVuZHMgQ29uc3RydWN0ICBpbXBsZW1lbnRzIElJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXJ7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwcmVDbG91ZFdhdGNoQWdlbnQoX2hvc3Q6IElIb3N0KTogdm9pZCB7XG4gIH1cblxuICAvKipcbiAgICogQGluaGVyaXRkb2NcbiAgICovXG4gIHByZVJlbmRlclF1ZXVlQ29uZmlndXJhdGlvbihfaG9zdDogSUhvc3QpOiB2b2lkIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHJlV29ya2VyQ29uZmlndXJhdGlvbihfaG9zdDogSUhvc3QpOiB2b2lkIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcG9zdFdvcmtlckxhdW5jaChfaG9zdDogSUhvc3QpOiB2b2lkIHtcbiAgfVxufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIHNldHRpbmdzIGZvciBEZWFkbGluZSBXb3JrZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgV29ya2VyU2V0dGluZ3Mge1xuICAvKipcbiAgICogRGVhZGxpbmUgZ3JvdXBzIHRoZXNlIHdvcmtlcnMgbmVlZHMgdG8gYmUgYXNzaWduZWQgdG8uIFRoZSBncm91cCBpc1xuICAgKiBjcmVhdGVkIGlmIGl0IGRvZXMgbm90IGFscmVhZHkgZXhpc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gV29ya2VyIGlzIG5vdCBhc3NpZ25lZCB0byBhbnkgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IGdyb3Vwcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBEZWFkbGluZSBwb29scyB0aGVzZSB3b3JrZXJzIG5lZWRzIHRvIGJlIGFzc2lnbmVkIHRvLiBUaGUgcG9vbCBpcyBjcmVhdGVkXG4gICAqIGlmIGl0IGRvZXMgbm90IGFscmVhZHkgZXhpc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gV29ya2VyIGlzIG5vdCBhc3NpZ25lZCB0byBhbnkgcG9vbC5cbiAgICovXG4gIHJlYWRvbmx5IHBvb2xzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIERlYWRsaW5lIHJlZ2lvbiB0aGVzZSB3b3JrZXJzIG5lZWRzIHRvIGJlIGFzc2lnbmVkIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFdvcmtlciBpcyBub3QgYXNzaWduZWQgdG8gYW55IHJlZ2lvblxuICAgKi9cbiAgcmVhZG9ubHkgcmVnaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCB0byBjb25maWd1cmUgdGhlIHdvcmtlciB0byBsaXN0ZW4gb24gZm9yIHJlbW90ZSBjb21tYW5kcyBzdWNoIGFzXG4gICAqIHJlcXVlc3RzIGZvciBpdHMgbG9nIHN0cmVhbS4gSWYgbW9yZSB0aGFuIG9uZSB3b3JrZXIgaXMgcHJlc2VudCBvbiBhIHNpbmdsZVxuICAgKiBob3N0LCBjb25uc2VjdXRpdmUgcG9ydHMgd2lsbCBiZSBvcGVuZWQsIHN0YXJ0aW5nIHdpdGggdGhlIHN1cHBsaWVkIHBvcnQsXG4gICAqIHVwIHRvIHRoZSBtYXhpbXVtIG51bWJlciBvZiB3b3JrZXJzIGRlZmluZWQgYnkgdGhlIFdvcmtlckluc3RhbmNlRmxlZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IDU2MDMyXG4gICAqL1xuICByZWFkb25seSBsaXN0ZW5lclBvcnQ/OiBudW1iZXI7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBXb3JrZXJJbnN0YW5jZUNvbmZpZ3VyYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3JrZXJJbnN0YW5jZUNvbmZpZ3VyYXRpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgRGVhZGxpbmUgV29ya2VyIHRoYXQgc2hvdWxkIGJlIGNvbmZpZ3VyZWQuXG4gICAqL1xuICByZWFkb25seSB3b3JrZXI6IElIb3N0O1xuXG4gIC8qKlxuICAgKiBUaGUgUmVuZGVyUXVldWUgdGhhdCB0aGUgd29ya2VyIHNob3VsZCBiZSBjb25maWd1cmVkIHRvIGNvbm5lY3QgdG9cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIFdvcmtlciBpcyBub3QgY29uZmlndXJlZCB0byBjb25uZWN0IHRvIGEgUmVuZGVyUXVldWVcbiAgICovXG4gIHJlYWRvbmx5IHJlbmRlclF1ZXVlPzogSVJlbmRlclF1ZXVlO1xuXG4gIC8qKlxuICAgKiBUaGUgY29uZmlndXJhdGlvbiBmb3Igc3RyZWFtaW5nIHRoZSBEZWFkbGluZSBXb3JrZXIgbG9ncyB0byBBV1MgQ2xvdWRXYXRjaC5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIFdvcmtlciBsb2dzIHdpbGwgbm90IGJlIHN0cmVhbWVkIHRvIENsb3VkV2F0Y2guXG4gICAqL1xuICByZWFkb25seSBjbG91ZFdhdGNoTG9nU2V0dGluZ3M/OiBMb2dHcm91cEZhY3RvcnlQcm9wcztcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhlIENsb3VkV2F0Y2ggYWdlbnQgc2hvdWxkIGJlIGF1dG9tYXRpY2FsbHkgaW5zdGFsbGVkIG9udG8gYWxsIHdvcmtlciBpbnN0YW5jZXMuXG4gICAqIFRoaXMgaW5zdGFsbGF0aW9uIHdpbGwgYmUgYSBiZXN0IGVmZm9ydCwgYnV0IHdpbGwgbm90IGZhaWwgdGhlIGRlcGxveW1lbnQgaWYgaXQgaXNuJ3QgY29tcGxldGVkXG4gICAqIHN1Y2Nlc3NmdWxseS4gSWRlYWxseSB0aGUgQ2xvdWRXYXRjaCBhZ2VudCBzaG91bGQgYmUgaW5zdGFsbGVkIG9uIHRoZSBBTUkgdG8gYXZvaWQgaXNzdWVzLiBJZlxuICAgKiB0aGUgaW5zdGFsbGF0aW9uIGZhaWxzLCBsb2dzIHdpbGwgbm90IGJlIHN0cmVhbWVkIG9mZiBvZiB0aGUgd29ya2VycyBpbnRvIENsb3VkV2F0Y2guXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHNob3VsZEluc3RhbGxDbG91ZFdhdGNoQWdlbnQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgc2V0dGluZ3MgdG8gYXBwbHkgdG8gdGhlIERlYWRsaW5lIFdvcmtlci5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIFdvcmtlciBpcyBhc3NpZ25lZCB0aGUgZGVmYXVsdCBzZXR0aW5ncyBhcyBvdXRsaW5lZCBpbiB0aGUgV29ya2VyU2V0dGluZ3MgaW50ZXJmYWNlLlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2VyU2V0dGluZ3M/OiBXb3JrZXJTZXR0aW5ncztcblxuICAvKipcbiAgICogQW4gb3B0aW9uYWwgcHJvdmlkZXIgb2YgdXNlciBkYXRhIGNvbW1hbmRzIHRvIGJlIGluamVjdGVkIGF0IHZhcmlvdXMgcG9pbnRzIGR1cmluZyB0aGUgV29ya2VyIGNvbmZpZ3VyYXRpb24gbGlmZWN5Y2xlLlxuICAgKiBZb3UgY2FuIHByb3ZpZGUgYSBzdWJjbGFzcyBvZiBJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXIgd2l0aCB0aGUgbWV0aG9kcyBvdmVycmlkZGVuIGFzIGRlc2lyZWQuXG4gICAqL1xuICByZWFkb25seSB1c2VyRGF0YVByb3ZpZGVyPzogSUluc3RhbmNlVXNlckRhdGFQcm92aWRlcjtcbn1cblxuLyoqXG4gKiBUaGlzIGNvbnN0cnVjdCBjYW4gYmUgdXNlZCB0byBjb25maWd1cmUgRGVhZGxpbmUgV29ya2VycyBvbiBhbiBpbnN0YW5jZSB0byBjb25uZWN0IHRvIGEgUmVuZGVyUXVldWUsIHN0cmVhbSB0aGVpclxuICogbG9nIGZpbGVzIHRvIENsb3VkV2F0Y2gsIGFuZCBjb25maWd1cmUgdmFyaW91cyBzZXR0aW5ncyBvZiB0aGUgRGVhZGxpbmUgV29ya2VyLlxuICpcbiAqIFRoZSBjb25maWd1cmF0aW9uIGhhcHBlbnMgb24gaW5zdGFuY2Ugc3RhcnQtdXAgdXNpbmcgdXNlciBkYXRhIHNjcmlwdGluZy5cbiAqXG4gKiBUaGlzIGNvbmZpZ3VyYXRpb24gcGVyZm9ybXMgdGhlIGZvbGxvd2luZyBzdGVwcyBpbiBvcmRlcjpcbiAqIDEpIENvbmZpZ3VyZSBDbG91ZCBXYXRjaCBBZ2VudFxuICogMikgQ29uZmlndXJlIERlYWRsaW5lIFdvcmtlciBSZW5kZXJRdWV1ZSBjb25uZWN0aW9uXG4gKiAzKSBDb25maWd1cmUgRGVhZGxpbmUgV29ya2VyIHNldHRpbmdzXG4gKlxuICogQSBgdXNlckRhdGFQcm92aWRlcmAgY2FuIGJlIHNwZWNpZmllZCB0aGF0IGRlZmluZXMgY2FsbGJhY2sgZnVuY3Rpb25zLlxuICogVGhlc2UgY2FsbGJhY2tzIGNhbiBiZSB1c2VkIHRvIGluamVjdCB1c2VyIGRhdGEgY29tbWFuZHMgYXQgZGlmZmVyZW50IHBvaW50cyBkdXJpbmcgdGhlIFdvcmtlciBpbnN0YW5jZSBjb25maWd1cmF0aW9uLlxuICpcbiAqIFNlY3VyaXR5IENvbnNpZGVyYXRpb25zXG4gKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqIC0gVGhlIGluc3RhbmNlcyBjb25maWd1cmVkIGJ5IHRoaXMgY29uc3RydWN0IHdpbGwgZG93bmxvYWQgYW5kIHJ1biBzY3JpcHRzIGZyb20geW91ciBDREsgYm9vdHN0cmFwIGJ1Y2tldCB3aGVuIHRoYXQgaW5zdGFuY2VcbiAqICAgaXMgbGF1bmNoZWQuIFlvdSBtdXN0IGxpbWl0IHdyaXRlIGFjY2VzcyB0byB5b3VyIENESyBib290c3RyYXAgYnVja2V0IHRvIHByZXZlbnQgYW4gYXR0YWNrZXIgZnJvbSBtb2RpZnlpbmcgdGhlIGFjdGlvbnNcbiAqICAgcGVyZm9ybWVkIGJ5IHRoZXNlIHNjcmlwdHMuIFdlIHN0cm9uZ2x5IHJlY29tbWVuZCB0aGF0IHlvdSBlaXRoZXIgZW5hYmxlIEFtYXpvbiBTMyBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmcgb24geW91ciBDREtcbiAqICAgYm9vdHN0cmFwIGJ1Y2tldCwgb3IgZW5hYmxlIEFXUyBDbG91ZFRyYWlsIG9uIHlvdXIgYWNjb3VudCB0byBhc3Npc3QgaW4gcG9zdC1pbmNpZGVudCBhbmFseXNpcyBvZiBjb21wcm9taXNlZCBwcm9kdWN0aW9uXG4gKiAgIGVudmlyb25tZW50cy5cbiAqL1xuZXhwb3J0IGNsYXNzIFdvcmtlckluc3RhbmNlQ29uZmlndXJhdGlvbiBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBwb3J0IHRvIHVzZSBmb3IgYSB3b3JrZXIgdG8gbGlzdGVuIG9uIGZvciByZW1vdGUgY29tbWFuZHMuXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBERUZBVUxUX0xJU1RFTkVSX1BPUlQgPSA1NjAzMjtcblxuICAvKipcbiAgICogQGluaGVyaXRkb2NcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsaXN0ZW5lclBvcnQ6IG51bWJlcjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogV29ya2VySW5zdGFuY2VDb25maWd1cmF0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIHByb3BzLnVzZXJEYXRhUHJvdmlkZXI/LnByZUNsb3VkV2F0Y2hBZ2VudChwcm9wcy53b3JrZXIpO1xuICAgIGlmIChwcm9wcy5jbG91ZFdhdGNoTG9nU2V0dGluZ3MpIHtcbiAgICAgIHRoaXMuY29uZmlndXJlQ2xvdWRXYXRjaExvZ1N0cmVhbShcbiAgICAgICAgcHJvcHMud29ya2VyLFxuICAgICAgICBpZCxcbiAgICAgICAgcHJvcHMuY2xvdWRXYXRjaExvZ1NldHRpbmdzLFxuICAgICAgICBwcm9wcy5zaG91bGRJbnN0YWxsQ2xvdWRXYXRjaEFnZW50LFxuICAgICAgKTtcbiAgICB9XG4gICAgcHJvcHMudXNlckRhdGFQcm92aWRlcj8ucHJlUmVuZGVyUXVldWVDb25maWd1cmF0aW9uKHByb3BzLndvcmtlcik7XG4gICAgcHJvcHMucmVuZGVyUXVldWU/LmNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHtcbiAgICAgIGhvc3Q6IHByb3BzLndvcmtlcixcbiAgICAgIC8vIERvbid0IHJlc3RhcnQgdGhlIERlYWRsaW5lIExhdW5jaGVyIHNlcnZpY2UgYWZ0ZXIgY29uZmlndXJpbmcgdGhlIGNvbm5lY3Rpb24gdG8gdGhlIFJlbmRlciBRdWV1ZS4gV2UgbmVlZCB0b1xuICAgICAgLy8gcmVzdGFydCBpdCBsYXRlciBhbnl3YXlzLCBhbmQgdGhlIFdpbmRvd3Mgc2VydmljZSBmb3IgdGhlIERlYWRsaW5lIExhdW5jaGVyIGNhbiBnZXQgbG9ja2VkIGluIHRoZSBcInN0b3BwaW5nXCJcbiAgICAgIC8vIHN0YXRlIGlmIHlvdSBhdHRlbXB0IHRvIHN0b3Agb3IgcmVzdGFydCBpdCB3aGlsZSBpdCBpcyBzdGlsbCByZXN0YXJ0aW5nLiBUaGlzIGNhbiBjYXVzZSB0aGUgdXNlciBkYXRhIGV4ZWN1dGlvblxuICAgICAgLy8gdG8gZ2V0IGxvY2tlZCB3YWl0aW5nIGZvciB0aGUgc2VydmljZSB0byBmaW5pc2ggc3RvcHBpbmcvcmVzdGFydGluZy5cbiAgICAgIHJlc3RhcnRMYXVuY2hlcjogZmFsc2UsXG4gICAgfSk7XG4gICAgcHJvcHMudXNlckRhdGFQcm92aWRlcj8ucHJlV29ya2VyQ29uZmlndXJhdGlvbihwcm9wcy53b3JrZXIpO1xuXG4gICAgdGhpcy5saXN0ZW5lclBvcnQgPSBwcm9wcy53b3JrZXJTZXR0aW5ncz8ubGlzdGVuZXJQb3J0ID8/IFdvcmtlckluc3RhbmNlQ29uZmlndXJhdGlvbi5ERUZBVUxUX0xJU1RFTkVSX1BPUlQ7XG4gICAgdGhpcy5jb25maWd1cmVXb3JrZXJTZXR0aW5ncyhwcm9wcy53b3JrZXIsIGlkLCBwcm9wcy53b3JrZXJTZXR0aW5ncyk7XG5cbiAgICBwcm9wcy51c2VyRGF0YVByb3ZpZGVyPy5wb3N0V29ya2VyTGF1bmNoKHByb3BzLndvcmtlcik7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRob2QgY2FuIGJlIHVzZWQgdG8gY29uZmlndXJlIGEgRGVhZGxpbmUgV29ya2VyIGluc3RhbmNlIHRvIHN0cmVhbSBpdHMgbG9ncyB0byB0aGUgQVdTIENsb3VkV2F0Y2hcbiAgICogc2VydmljZS4gVGhlIGxvZ3MgdGhhdCB0aGlzIGNvbmZpZ3VyZXMgdG8gc3RyZWFtIGFyZTpcbiAgICogLSBFQzIgSW5zdGFuY2UgVXNlckRhdGEgZXhlY3V0aW9uOyB0aGlzIGlzIHRoZSBzdGFydHVwIHNjcmlwdGluZyB0aGF0IGlzIHJ1biB3aGVuIHRoZSBpbnN0YW5jZSBsYXVuY2hlc1xuICAgKiAgIGZvciB0aGUgZmlyc3QgdGltZS5cbiAgICogLSBEZWFkbGluZSBXb3JrZXIgbG9ncy5cbiAgICogLSBEZWFkbGluZSBMYXVuY2hlciBsb2dzLlxuICAgKlxuICAgKiBAcGFyYW0gd29ya2VyIFRoZSB3b3JrZXIgdG8gY29uZmlndXJlLiBUaGlzIGNhbiBiZSBhbiBpbnN0YW5jZSwgYXV0byBzY2FsaW5nIGdyb3VwLCBsYXVuY2ggdGVtcGxhdGUsIGV0Yy5cbiAgICogQHBhcmFtIGlkIElkZW50aWZpZXIgdG8gZGlzYW1iaWd1YXRlIHRoZSByZXNvdXJjZXMgdGhhdCBhcmUgY3JlYXRlZC5cbiAgICogQHBhcmFtIHNob3VsZEluc3RhbGxBZ2VudCBCb29sZWFuIGZvciBpZiB0aGUgd29ya2VyJ3MgVXNlciBEYXRhIHNob3VsZCBhdHRlbXB0IHRvIGluc3RhbGwgdGhlIENsb3VkV2F0Y2ggYWdlbnRcbiAgICogQHBhcmFtIGxvZ0dyb3VwUHJvcHMgQ29uZmlndXJhdGlvbiBmb3IgdGhlIGxvZyBncm91cCBpbiBDbG91ZFdhdGNoLlxuICAgKi9cbiAgcHJvdGVjdGVkIGNvbmZpZ3VyZUNsb3VkV2F0Y2hMb2dTdHJlYW0oXG4gICAgd29ya2VyOiBJSG9zdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIGxvZ0dyb3VwUHJvcHM6IExvZ0dyb3VwRmFjdG9yeVByb3BzLFxuICAgIHNob3VsZEluc3RhbGxBZ2VudD86IGJvb2xlYW4sXG4gICk6IHZvaWQge1xuICAgIGNvbnN0IGxvZ0dyb3VwID0gTG9nR3JvdXBGYWN0b3J5LmNyZWF0ZU9yRmV0Y2godGhpcywgYCR7aWR9TG9nR3JvdXBXcmFwcGVyYCwgaWQsIGxvZ0dyb3VwUHJvcHMpO1xuXG4gICAgbG9nR3JvdXAuZ3JhbnRXcml0ZSh3b3JrZXIpO1xuXG4gICAgY29uc3QgY2xvdWRXYXRjaENvbmZpZ3VyYXRpb25CdWlsZGVyID0gbmV3IENsb3VkV2F0Y2hDb25maWdCdWlsZGVyKER1cmF0aW9uLnNlY29uZHMoMTUpKTtcblxuICAgIGlmICh3b3JrZXIub3NUeXBlID09PSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1MpIHtcbiAgICAgIGNsb3VkV2F0Y2hDb25maWd1cmF0aW9uQnVpbGRlci5hZGRMb2dzQ29sbGVjdExpc3QobG9nR3JvdXAubG9nR3JvdXBOYW1lLFxuICAgICAgICAnVXNlcmRhdGFFeGVjdXRpb24nLFxuICAgICAgICAnQzpcXFxcUHJvZ3JhbURhdGFcXFxcQW1hem9uXFxcXEVDMi1XaW5kb3dzXFxcXExhdW5jaFxcXFxMb2dcXFxcVXNlcmRhdGFFeGVjdXRpb24ubG9nJyk7XG4gICAgICBjbG91ZFdhdGNoQ29uZmlndXJhdGlvbkJ1aWxkZXIuYWRkTG9nc0NvbGxlY3RMaXN0KGxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgJ1dvcmtlckxvZ3MnLFxuICAgICAgICAnQzpcXFxcUHJvZ3JhbURhdGFcXFxcVGhpbmtib3hcXFxcRGVhZGxpbmUxMFxcXFxsb2dzXFxcXGRlYWRsaW5lc2xhdmUqLmxvZycpO1xuICAgICAgY2xvdWRXYXRjaENvbmZpZ3VyYXRpb25CdWlsZGVyLmFkZExvZ3NDb2xsZWN0TGlzdChsb2dHcm91cC5sb2dHcm91cE5hbWUsXG4gICAgICAgICdMYXVuY2hlckxvZ3MnLFxuICAgICAgICAnQzpcXFxcUHJvZ3JhbURhdGFcXFxcVGhpbmtib3hcXFxcRGVhZGxpbmUxMFxcXFxsb2dzXFxcXGRlYWRsaW5lbGF1bmNoZXIqLmxvZycpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjbG91ZFdhdGNoQ29uZmlndXJhdGlvbkJ1aWxkZXIuYWRkTG9nc0NvbGxlY3RMaXN0KGxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgJ2Nsb3VkLWluaXQtb3V0cHV0JyxcbiAgICAgICAgJy92YXIvbG9nL2Nsb3VkLWluaXQtb3V0cHV0LmxvZycpO1xuICAgICAgY2xvdWRXYXRjaENvbmZpZ3VyYXRpb25CdWlsZGVyLmFkZExvZ3NDb2xsZWN0TGlzdChsb2dHcm91cC5sb2dHcm91cE5hbWUsXG4gICAgICAgICdXb3JrZXJMb2dzJyxcbiAgICAgICAgJy92YXIvbG9nL1RoaW5rYm94L0RlYWRsaW5lMTAvZGVhZGxpbmVzbGF2ZSoubG9nJyk7XG4gICAgICBjbG91ZFdhdGNoQ29uZmlndXJhdGlvbkJ1aWxkZXIuYWRkTG9nc0NvbGxlY3RMaXN0KGxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgJ0xhdW5jaGVyTG9ncycsXG4gICAgICAgICcvdmFyL2xvZy9UaGlua2JveC9EZWFkbGluZTEwL2RlYWRsaW5lbGF1bmNoZXIqLmxvZycpO1xuICAgIH1cblxuICAgIG5ldyBDbG91ZFdhdGNoQWdlbnQodGhpcywgYCR7aWR9TG9nc0NvbmZpZ2AsIHtcbiAgICAgIGNsb3VkV2F0Y2hDb25maWc6IGNsb3VkV2F0Y2hDb25maWd1cmF0aW9uQnVpbGRlci5nZW5lcmF0ZUNsb3VkV2F0Y2hDb25maWd1cmF0aW9uKCksXG4gICAgICBob3N0OiB3b3JrZXIsXG4gICAgICBzaG91bGRJbnN0YWxsQWdlbnQsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRob2QgY2FuIGJlIHVzZWQgdG8gc2V0IHVwIHRoZSBEZWFkbGluZSBXb3JrZXIgYXBwbGljYXRpb24gb24gYW4gRUMyIGluc3RhbmNlLiBGcm9tIGEgcHJhY3RpY2FsXG4gICAqIHBlcnNwZWN0aXZlLCB0aGlzIGlzIGV4ZWN1dGluZyB0aGUgc2NyaXB0IGZvdW5kIGluIGF3cy1yZmRrL2xpYi9kZWFkbGluZS9zY3JpcHRzL1tiYXNoLHBvd2Vyc2hlbGxdL2NvbmZpZ3VyZVdvcmtlci5bc2gscHMxXVxuICAgKiB0byBjb25maWd1cmUgdGhlIERlYWRsaW5lIFdvcmtlciBhcHBsaWNhdGlvbi5cbiAgICpcbiAgICogQHBhcmFtIHdvcmtlciBUaGUgd29ya2VyIHRvIGNvbmZpZ3VyZS4gVGhpcyBjYW4gYmUgYW4gaW5zdGFuY2UsIGF1dG8gc2NhbGluZyBncm91cCwgbGF1bmNoIHRlbXBsYXRlLCBldGMuXG4gICAqIEBwYXJhbSBpZCBJZGVudGlmaWVyIHRvIGRpc2FtYmlndWF0ZSB0aGUgcmVzb3VyY2VzIHRoYXQgYXJlIGNyZWF0ZWQuXG4gICAqIEBwYXJhbSBzZXR0aW5ncyBUaGUgRGVhZGxpbmUgV29ya2VyIHNldHRpbmdzIHRvIGFwcGx5LlxuICAgKi9cbiAgcHJvdGVjdGVkIGNvbmZpZ3VyZVdvcmtlclNldHRpbmdzKHdvcmtlcjogSUhvc3QsIGlkOiBzdHJpbmcsIHNldHRpbmdzPzogV29ya2VyU2V0dGluZ3MpOiB2b2lkIHtcbiAgICBjb25zdCBjb25maWd1cmVXb3JrZXJTY3JpcHRBc3NldCA9IFNjcmlwdEFzc2V0LmZyb21QYXRoQ29udmVudGlvbih0aGlzLCBgJHtpZH1Db25maWdTY3JpcHRgLCB7XG4gICAgICBvc1R5cGU6IHdvcmtlci5vc1R5cGUsXG4gICAgICBiYXNlTmFtZTogJ2NvbmZpZ3VyZVdvcmtlcicsXG4gICAgICByb290RGlyOiBwYXRoLmpvaW4oXG4gICAgICAgIF9fZGlybmFtZSxcbiAgICAgICAgJy4uJyxcbiAgICAgICAgJ3NjcmlwdHMvJyxcbiAgICAgICksXG4gICAgfSk7XG4gICAgY29uc3QgY29uZmlndXJlV29ya2VyUG9ydEFzc2V0ID0gbmV3IEFzc2V0KHRoaXMsIGAke2lkfVdvcmtlckxpc3RlbmVyU2NyaXB0YCwge1xuICAgICAgcGF0aDogcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ3NjcmlwdHMnLCAncHl0aG9uJywgJ3dvcmtlci1saXN0ZW5pbmctcG9ydC5weScpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY29uZmlnV29ya2VyUG9ydExvY2FsUGF0aCA9IHdvcmtlci51c2VyRGF0YS5hZGRTM0Rvd25sb2FkQ29tbWFuZCh7XG4gICAgICBidWNrZXQ6IGNvbmZpZ3VyZVdvcmtlclBvcnRBc3NldC5idWNrZXQsXG4gICAgICBidWNrZXRLZXk6IGNvbmZpZ3VyZVdvcmtlclBvcnRBc3NldC5zM09iamVjdEtleSxcbiAgICB9KTtcblxuICAgIC8vIENvbnZlcnRpbmcgdG8gbG93ZXIgY2FzZSwgYXMgZ3JvdXBzIGFuZCBwb29scyBhcmUgYWxsIHN0b3JlZCBpbiBsb3dlciBjYXNlIGluIGRlYWRsaW5lLlxuICAgIGNvbnN0IGdyb3VwcyA9IHNldHRpbmdzPy5ncm91cHM/Lm1hcCh2YWwgPT4gdmFsLnRvTG93ZXJDYXNlKCkpLmpvaW4oJywnKSA/PyAnJztcbiAgICBjb25zdCBwb29scyA9IHNldHRpbmdzPy5wb29scz8ubWFwKHZhbCA9PiB2YWwudG9Mb3dlckNhc2UoKSkuam9pbignLCcpID8/ICcnO1xuXG4gICAgY29uZmlndXJlV29ya2VyU2NyaXB0QXNzZXQuZXhlY3V0ZU9uKHtcbiAgICAgIGhvc3Q6IHdvcmtlcixcbiAgICAgIGFyZ3M6IFtcbiAgICAgICAgYCcke2dyb3Vwc30nYCxcbiAgICAgICAgYCcke3Bvb2xzfSdgLFxuICAgICAgICBgJyR7c2V0dGluZ3M/LnJlZ2lvbiA/PyAnJ30nYCxcbiAgICAgICAgYCcke1ZlcnNpb24uTUlOSU1VTV9TVVBQT1JURURfREVBRExJTkVfVkVSU0lPTi50b1N0cmluZygpfSdgLFxuICAgICAgICB0aGlzLmxpc3RlbmVyUG9ydC50b1N0cmluZygpLFxuICAgICAgICBjb25maWdXb3JrZXJQb3J0TG9jYWxQYXRoLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxufVxuIl19