"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends constructs_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.expression = props?.version;
        const match = (props?.version ?? '').match(VersionQuery.EXPRESSION_REGEX);
        if (match === null) {
            throw new Error(`Invalid version expression "${props.version}`);
        }
        this.pinnedVersionComponents = (match
            // First capture group is the entire matched string, so slice it off
            .slice(1)
            // Capture groups that are not matched return as undefined, so we filter them out
            .filter(component => component)
            // Parse strings to numbers
            .map(component => Number(component)));
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'version-provider.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props?.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props?.version),
        };
        this.deadlineResource = new aws_cdk_lib_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = this.versionComponent({
            expressionIndex: 0,
            customResourceAttribute: 'MajorVersion',
        });
        this.minorVersion = this.versionComponent({
            expressionIndex: 1,
            customResourceAttribute: 'MinorVersion',
        });
        this.releaseVersion = this.versionComponent({
            expressionIndex: 2,
            customResourceAttribute: 'ReleaseVersion',
        });
        const installerBucket = aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', this.deadlineResource.getAttString('S3Bucket'));
        this.linuxInstallers = {
            patchVersion: aws_cdk_lib_1.Token.asNumber(this.deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: this.deadlineResource.getAttString('LinuxRepositoryInstaller'),
                s3Bucket: installerBucket,
            },
            client: {
                objectKey: this.deadlineResource.getAttString('LinuxClientInstaller'),
                s3Bucket: installerBucket,
            },
        };
    }
    versionComponent(args) {
        const { expressionIndex, customResourceAttribute } = args;
        return this.pinnedVersionComponents.length > expressionIndex
            ? this.pinnedVersionComponents[expressionIndex]
            : aws_cdk_lib_1.Token.asNumber(this.deadlineResource.getAtt(customResourceAttribute));
    }
    linuxFullVersionString() {
        const major = aws_cdk_lib_1.Token.isUnresolved(this.majorVersion) ? aws_cdk_lib_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = aws_cdk_lib_1.Token.isUnresolved(this.minorVersion) ? aws_cdk_lib_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = aws_cdk_lib_1.Token.isUnresolved(this.releaseVersion) ? aws_cdk_lib_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = aws_cdk_lib_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? aws_cdk_lib_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    isLessThan(other) {
        if (other.patchVersion !== 0) {
            throw new Error('Cannot compare a VersionQuery to a fully-qualified version with a non-zero patch number');
        }
        // We compare each component from highest order to lowest
        const componentGetters = [
            v => v.majorVersion,
            v => v.minorVersion,
            v => v.releaseVersion,
        ];
        for (const componentGetter of componentGetters) {
            const thisComponent = componentGetter(this);
            const otherComponent = componentGetter(other);
            if (aws_cdk_lib_1.Token.isUnresolved(thisComponent)) {
                // Unresolved components are unpinned. These will resolve to the latest and are not less than any provided
                // version
                return false;
            }
            else {
                const componentDiff = thisComponent - otherComponent;
                if (componentDiff !== 0) {
                    // If the components are different, return whether this component is smaller than the other component
                    return componentDiff < 0;
                }
            }
        }
        // If we've exited the loop naturally, it means all version components are pinned and equal. This means the version
        // is not less than the other, they are the same
        return false;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version?.split('.').map(x => parseInt(x));
        if (!components || components?.length != 4) {
            return false;
        }
        return true;
    }
    get versionString() {
        return this.expression ?? '(latest)';
    }
}
exports.VersionQuery = VersionQuery;
_a = JSII_RTTI_SYMBOL_1;
VersionQuery[_a] = { fqn: "aws-rfdk.deadline.VersionQuery", version: "1.1.0" };
/**
 * Regular expression for valid version query expressions
 */
VersionQuery.EXPRESSION_REGEX = /^(?:(\d+)(?:\.(\d+)(?:\.(\d+)(?:\.(\d+))?)?)?)?$/;
//# sourceMappingURL=data:application/json;base64,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