"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsCustomerAgreementAndIpLicenseAcceptance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const _1 = require(".");
/**
 * The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.
 *
 * By downloading or using the Deadline software, you agree to the AWS Customer Agreement (https://aws.amazon.com/agreement/)
 * and AWS Intellectual Property License (https://aws.amazon.com/legal/aws-ip-license-terms/). You acknowledge that Deadline
 * is AWS Content as defined in those Agreements.
 *
 * This enum is used to signify acceptance or rejection of these terms.
 */
var AwsCustomerAgreementAndIpLicenseAcceptance;
(function (AwsCustomerAgreementAndIpLicenseAcceptance) {
    /**
     * The user signifies their explicit rejection of the terms.
     */
    AwsCustomerAgreementAndIpLicenseAcceptance[AwsCustomerAgreementAndIpLicenseAcceptance["USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE"] = 0] = "USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE";
    /**
     * The user signifies their explicit acceptance of the terms.
     */
    AwsCustomerAgreementAndIpLicenseAcceptance[AwsCustomerAgreementAndIpLicenseAcceptance["USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE"] = 1] = "USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE";
})(AwsCustomerAgreementAndIpLicenseAcceptance = exports.AwsCustomerAgreementAndIpLicenseAcceptance || (exports.AwsCustomerAgreementAndIpLicenseAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *    distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *    requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *   recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *   CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsCustomerAgreementAndIpLicenseAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *   version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *   version,
 *   // Change this to AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to accept the terms
 *   // of the AWS Customer Agreement and AWS Intellectual Property License.
 *   userAwsCustomerAgreementAndIpLicenseAcceptance: AwsCustomerAgreementAndIpLicenseAcceptance.USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   images: images.forRenderQueue(),
 *   // ...
 * });
 * ```
 */
class ThinkboxDockerImages extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsCustomerAgreementAndIpLicenseAcceptance = props.userAwsCustomerAgreementAndIpLicenseAcceptance;
        this.version = props?.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'ecr-provider.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new aws_cdk_lib_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
        const thisConstruct = this;
        this.node.addValidation({
            validate() {
                const errors = [];
                // Users must accept the AWS Customer Agreement and AWS Intellectual Property License to use the container images
                if (thisConstruct.userAwsCustomerAgreementAndIpLicenseAcceptance !==
                    AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE) {
                    errors.push(ThinkboxDockerImages.AWS_CONTENT_NOTICE);
                }
                // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
                // a resource in another stack is referencing it.
                if (thisConstruct.version instanceof _1.VersionQuery) {
                    const versionStack = aws_cdk_lib_1.Stack.of(thisConstruct.version);
                    const thisStack = aws_cdk_lib_1.Stack.of(thisConstruct);
                    if (versionStack != thisStack) {
                        errors.push('A VersionQuery can not be supplied from a different stack');
                    }
                }
                return errors;
            },
        });
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct
     */
    forRenderQueue() {
        return {
            remoteConnectionServer: this.remoteConnectionServer,
        };
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct
     */
    forUsageBasedLicensing() {
        return {
            licenseForwarder: this.licenseForwarder,
        };
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return aws_cdk_lib_1.Token.isUnresolved(num) ? aws_cdk_lib_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
_a = JSII_RTTI_SYMBOL_1;
ThinkboxDockerImages[_a] = { fqn: "aws-rfdk.deadline.ThinkboxDockerImages", version: "1.1.0" };
/**
 * The Deadline licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Content Agreement and AWS Intellectual Property License.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * without approval.
 */
ThinkboxDockerImages.AWS_CONTENT_NOTICE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

By downloading or using the Deadline software, you agree to the AWS Customer Agreement (https://aws.amazon.com/agreement/)
and AWS Intellectual Property License (https://aws.amazon.com/legal/aws-ip-license-terms/). You acknowledge that Deadline
is AWS Content as defined in those Agreements.

Please set the userAwsCustomerAgreementAndIpLicenseAcceptance property to
USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to signify your acceptance of these terms.
`;
//# sourceMappingURL=data:application/json;base64,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