"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const _1 = require("./");
/**
 * Class for interacting with the Deadline stage directory
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 */
class Stage {
    /**
     * Constructs a Stage instance
     *
     * @param dirPath The path to the Deadline staging directory
     * @param manifest The parsed manifest file
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path
     * @param manifestPath The path to the manifest JSON file
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context
     * and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        /* istanbul ignore else */
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        // Do minimum supported deadline version check
        const stagedVersion = _1.Version.parse(version);
        if (stagedVersion.isLessThan(_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            throw new TypeError(`Staged Deadline Version (${version}) is less than the minimum supported version (${_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()})`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version
     *
     * @param scope The parent scope
     * @param id The construct ID
     */
    getVersion(scope, id) {
        const releaseVersion = this.getReleaseVersion(this.manifest.version);
        return new _1.VersionQuery(scope, id, { version: releaseVersion });
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage
     * @param scope The scope for the {@link DockerImageAsset}
     * @param id The construct ID of the {@link DockerImageAsset}
     * @param recipeName The name of the recipe
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
    /**
     * This removes the patch version from a full version string. No validation is done as that is handled
     * in the constructor with the version check.
     */
    getReleaseVersion(fullVersion) {
        const versionComponents = fullVersion.split('.');
        return `${versionComponents[0]}.${versionComponents[1]}.${versionComponents[2]}`;
    }
    get clientInstallerPath() {
        const INSTALLER_FILENAME_RE = /^DeadlineClient-(?<version>.+)-linux-x64-installer\.run$/;
        const listing = fs.readdirSync(path.join(this.dirPath, 'bin')).filter(filename => INSTALLER_FILENAME_RE.test(filename));
        /* istanbul ignore else */
        if (listing.length === 1) {
            const filename = listing[0];
            const match = INSTALLER_FILENAME_RE.exec(filename);
            const version = match.groups.version;
            const recipes = Array.from(Object.values(this.manifest.recipes));
            const aRecipeHasMatchingDlVersion = recipes.some((recipe) => {
                return recipe.buildArgs?.DL_VERSION === version;
            });
            const installerPath = path.join(this.dirPath, 'bin', listing[0]);
            if (!aRecipeHasMatchingDlVersion) {
                throw new Error(`No stage recipes refer to the Deadline Client installer found (${installerPath})`);
            }
            return installerPath;
        }
        else if (listing.length === 0) {
            throw new Error(`No matching Client installer found in "${this.dirPath}"`);
        }
        else {
            throw new Error(`Multiple Client installers found: ${listing.join(',')}`);
        }
    }
}
exports.Stage = Stage;
_a = JSII_RTTI_SYMBOL_1;
Stage[_a] = { fqn: "aws-rfdk.deadline.Stage", version: "1.1.0" };
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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