/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Stack } from 'aws-cdk-lib';
import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { ConnectableApplicationEndpoint } from '../../core';
import { IHost } from './host-ref';
import { ECSConnectOptions, InstanceConnectOptions } from './render-queue-ref';
/**
 * The basic options that all connections require
 */
export interface BaseConnectionOptions {
    /**
     * The endpoint of the Renderqueue we are connecting to.
     */
    readonly endpoint: ConnectableApplicationEndpoint;
}
/**
 * Options used for creating an HTTPS Connection object
 */
export interface HTTPSConnectionOptions extends BaseConnectionOptions {
    /**
     * The CACert that signed the Cert the Render Queue uses.
     */
    readonly caCert: ISecret;
}
interface ConnectionScriptArguments {
    /**
     * The address of the Render Queue
     */
    readonly address: string;
    /**
     * The ARN to the AWS Secrets Manager secret containing the X509 CA Certificate in PEM format.
     */
    readonly tlsCaArn?: string;
    /**
     * Whether to restart the Deadline launcher after configuring the Render Queue connection.
     */
    readonly restartLauncher?: boolean;
}
/**
 * Helper class for connecting Clients to a render queue
 */
export declare abstract class RenderQueueConnection {
    /**
     * Create a RQ Connection for http traffic
     */
    static forHttp(options: BaseConnectionOptions): RenderQueueConnection;
    /**
     * Create a RQ Connection for https traffic
     */
    static forHttps(options: HTTPSConnectionOptions): RenderQueueConnection;
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue
     * @returns An environment mapping that is used to configure the Docker Images
     */
    abstract configureClientECS(params: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue
     */
    abstract configureClientInstance(params: InstanceConnectOptions): void;
    /**
     * Fetch the instance configuration python script
     * @param stack The stack that the asset should be created in
     */
    protected connectionAssetSingleton(stack: Stack): Asset;
    /**
     * Executes connection asset singleton wth a given list of arguments
     */
    protected ExecuteConnectionAsset(host: IHost, args: ConnectionScriptArguments): void;
}
export {};
