/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Duration, RemovalPolicy } from 'aws-cdk-lib';
import { ISecurityGroup, IVpc, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { MountPoint, TaskDefinition } from 'aws-cdk-lib/aws-ecs';
import { FileSystem as EfsFileSystem } from 'aws-cdk-lib/aws-efs';
import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct, IConstruct } from 'constructs';
import { IMountableLinuxFilesystem, LogGroupFactoryProps } from '../../core';
import { DatabaseConnection } from './database-connection';
import { IHost } from './host-ref';
import { IVersion } from './version-ref';
/**
 * Configuration interface for specifying ECS container instances to permit connecting hosted ECS tasks to the repository
 */
export interface ECSContainerInstanceProps {
    /**
     * The set of hosts that will be hosting the containers.
     *
     * This can be AutoScalingGroups that make up the capacity of an Amazon ECS cluster, or individual instances.
     */
    readonly hosts: IHost[];
    /**
     * The path where the repository file-system is mounted on the container hosts.
     *
     * @default "/mnt/repo"
     */
    readonly filesystemMountPoint?: string;
}
/**
 * Configuration interface to directly connect an ECS task to the repository.
 */
export interface ECSTaskProps {
    /**
     * The task definition to connect to the repository.
     *
     * [disable-awslint:ref-via-interface]
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * The path where the repository file-system is mounted within the container.
     *
     * @default "/opt/Thinkbox/DeadlineRepository{MAJOR_VER}"
     */
    readonly filesystemMountPoint?: string;
}
/**
 * The properties used to configure Deadline running in an Amazon EC2 ECS task to directly connect to the repository.
 */
export interface ECSDirectConnectProps {
    /**
     * Configuration of ECS host instances to permit connecting hosted ECS tasks to the repository
     */
    readonly containerInstances: ECSContainerInstanceProps;
    /**
     * Configuration to directly connect an ECS task to the repository.
     */
    readonly containers: ECSTaskProps;
}
/**
 * Interface that can be used to configure a {@link @aws-cdk/aws-ecs#ContainerDefinition} definition to directly connect
 * to the repository.
 */
export interface IContainerDirectRepositoryConnection {
    /**
     * Environment variables that configure a direct connection to the repository.
     */
    readonly containerEnvironment: {
        [name: string]: string;
    };
    /**
     * A {@link MountPoint} that can be used to create a read/write mount the repository file-system from the task's
     * container instance into a container. This can be used with the `addMountPoint` method of the
     * {@link @aws-cdk/aws-ecs#ContainerDefinition} instance.
     */
    readonly readWriteMountPoint: MountPoint;
    /**
     * A {@link MountPoint} that can be used to create a read/write mount the repository file-system from the task's
     * container instance into a container. This can be used with the `addMountPoint` method of the
     * {@link @aws-cdk/aws-ecs#ContainerDefinition} instance.
     */
    readonly readOnlyMountPoint: MountPoint;
}
/**
 *  The Properties used to configure Deadline, that is running in an Amazon EC2 instance, a direct connection with a repository.
 */
export interface InstanceDirectConnectProps {
    /**
     * The Instance/UserData which will directly connect to the Repository
     */
    readonly host: IHost;
    /**
     * The location where the Repositories file system will be mounted on the instance.
     */
    readonly mountPoint: string;
}
/**
 * Interface for Deadline Repository.
 */
export interface IRepository extends IConstruct {
    /**
     * The path to the Deadline Repository directory.
     *
     * This is expressed as a relative path from the root of the Deadline Repository file-system.
     */
    readonly rootPrefix: string;
    /**
     * The version of Deadline for Linux that is installed on this Repository.
     */
    readonly version: IVersion;
    /**
     * Deadline Secrets Management settings.
     */
    readonly secretsManagementSettings: SecretsManagementProps;
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to
     * this repository.
     *
     * This includes:
     *   - Ingress to database & filesystem Security Groups, as required.
     *   - IAM Permissions for database & filesystem, as required.
     *   - Mounts the Repository File System via UserData
     *
     * @param props The props used to configure the Deadline client.
     * @returns A mapping of environment variable names and their values to set in the container
     */
    configureClientECS(props: ECSDirectConnectProps): IContainerDirectRepositoryConnection;
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     * This includes:
     *   - Ingress to database & filesystem Security Groups, as required.
     *   - IAM Permissions for database & filesystem, as required.
     *   - Mounts the Repository File System via UserData
     *   - Configures Deadline to direct-connect to the Repository.
     *
     * @param props The props used to configure the Deadline client.
     */
    configureClientInstance(props: InstanceDirectConnectProps): void;
}
/**
 * Properties for backups of resources that are created by the Repository.
 */
export interface RepositoryBackupOptions {
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the retention period to
     * use on the database. If the Repository is not creating a DocumentDB database, because one was given,
     * then this property is ignored.
     * Please visit https://aws.amazon.com/documentdb/pricing/ to learn more about DocumentDB backup storage pricing.
     *
     * @default Duration.days(15)
     */
    readonly databaseRetention?: Duration;
}
export interface RepositoryRemovalPolicies {
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the retention policy to
     * use on the database. If the Repository is not creating a DocumentDB database, because one was given,
     * then this property is ignored.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly database?: RemovalPolicy;
    /**
     * If this Repository is creating its own Amazon Elastic File System (EFS), then this specifies the retention policy to
     * use on the filesystem. If the Repository is not creating an EFS, because one was given, then this property is ignored.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly filesystem?: RemovalPolicy;
}
/**
 * Options for the security groups of the Repository.
 */
export interface RepositorySecurityGroupsOptions {
    /**
     * The security group for the filesystem of the Repository. This is ignored if the Repository is not creating
     * its own Amazon Elastic File System (EFS) because one was given.
     */
    readonly fileSystem?: ISecurityGroup;
    /**
     * The security group for the database of the Repository. This is ignored if the Repository is not creating
     * its own DocumentDB database because one was given.
     */
    readonly database?: ISecurityGroup;
    /**
     * The security group for the AutoScalingGroup of the instance that runs the Deadline Repository installer.
     */
    readonly installer?: ISecurityGroup;
}
/**
 * Settings used by Deadline Secrets Management, a feature introduced in Deadline 10.1.10 for securely managing storage
 * and access of Secrets for your render farm.
 * More details at:
 * https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/secrets-management/deadline-secrets-management.html
 * Using Secrets Management requires TLS to be enabled between the RenderQueue and its clients. If this feature is enabled, the
 * `externalTLS` on the `RenderQueueTrafficEncryptionProps` interface on the RenderQueue cannot be disabled.
 */
export interface SecretsManagementProps {
    /**
     * Whether or not to enable the Secrets Management feature.
     */
    readonly enabled: boolean;
    /**
     * A Secret containing the username and password to use for the admin role.
     * The contents of this secret must be a JSON document with the keys "username" and "password". ex:
     *     {
     *         "username": <admin user name>,
     *         "password": <admin user password>,
     *     }
     * Password should be at least 8 characters long and contain at least one lowercase letter, one uppercase letter, one symbol and one number.
     * In the case when the password does not meet the requirements, the repository construct will fail to deploy.
     * It is highly recommended that you leave this parameter undefined to enable the automatic generation of a strong password.
     *
     * @default: A random username and password will be generated in a Secret with ID `SMAdminUser` and will need to be retrieved from AWS Secrets Manager if it is needed
     */
    readonly credentials?: ISecret;
    /**
     * If Secret with admin credentials is not defined in property `credentials`, then this specifies the retention policy to
     * use on the Secret with generated credentials. If Secret with credentials is provided, then this property is ignored.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly credentialsRemovalPolicy?: RemovalPolicy;
}
/**
 * Properties for the Deadline repository
 */
export interface RepositoryProps {
    /**
     * VPC to launch the Repository In
     */
    readonly vpc: IVpc;
    /**
     * Version property to specify the version of deadline repository to be installed.
     * This, in future, would be an optional property. If not passed, it should fetch
     * the latest version of deadline. The current implementation of Version construct
     * only supports importing it with static values, hence keeping it mandatory for now.
     */
    readonly version: IVersion;
    /**
     * Properties for setting up the Deadline Repository's LogGroup in CloudWatch
     * @default - LogGroup will be created with all properties' default values to the LogGroup: /renderfarm/<construct id>
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * The length of time to wait for the repository installation before considering it as failure.
     *
     * The maximum value is 43200 (12 hours).
     *
     * @default Duration.minutes(15)
     */
    readonly repositoryInstallationTimeout?: Duration;
    /**
     * Specify the file system where the deadline repository needs to be initialized.
     *
     * If not providing a filesystem, then we will provision an Amazon EFS filesystem for you.
     * This filesystem will contain files for the Deadline Repository filesystem. It will also
     * contain 40GB of additional padding files (see RFDK's PadEfsStorage for details) to increase
     * the baseline throughput of the filesystem; these files will be added to the /RFDK_PaddingFiles directory
     * in the filesystem.
     *
     * @default An Encrypted EFS File System and Access Point will be created.
     */
    readonly fileSystem?: IMountableLinuxFilesystem;
    /**
     * The prefix for the Deadline Repository installation path on the mounted file system.
     *
     * @default: "/DeadlineRepository/"
     */
    readonly repositoryInstallationPrefix?: string;
    /**
     * Specify the database where the deadline schema needs to be initialized.
     * Note that Deadline supports only databases that are compatible with MongoDB 3.6.
     *
     * @default A Document DB Cluster will be created with a single db.r5.large instance.
     */
    readonly database?: DatabaseConnection;
    /**
     * Define the removal policies for the resources that this Repository creates. These define what happens
     * to the resoureces when the stack that defines them is destroyed.
     *
     * @default RemovalPolicy.RETAIN for all resources
     */
    readonly removalPolicy?: RepositoryRemovalPolicies;
    /**
     * If this Repository is creating its own DocumentDB database, then this specifies if audit logging will be enabled
     *
     * Audit logs are a security best-practice. They record connection, data definition language (DDL), user management,
     * and authorization events within the database, and are useful for post-incident auditing. That is, they can help you
     * figure out what an unauthorized user, who gained access to your database, has done with that access.
     *
     * @default true
     */
    readonly databaseAuditLogging?: boolean;
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the number of
     * compute instances to be created.
     *
     * @default 1
     */
    readonly documentDbInstanceCount?: number;
    /**
     * All resources that are created by this Repository will be deployed to these Subnets. This includes the
     * Auto Scaling Group that is created for running the Repository Installer. If this Repository is creating
     * an Amazon DocumentDB database and/or Amazon Elastic File System (EFS), then this specifies the subnets
     * to which they are deployed.
     *
     * @default: Private subnets in the VPC
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Define the backup options for the resources that this Repository creates.
     *
     * @default Duration.days(15) for the database
     */
    readonly backupOptions?: RepositoryBackupOptions;
    /**
     * Options to add additional security groups to the Repository.
     */
    readonly securityGroupsOptions?: RepositorySecurityGroupsOptions;
    /**
     * The Deadline Repository settings file to import.
     * @see https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/repository-settings-importer-exporter.html
     *
     * @default Repository settings are not imported.
     */
    readonly repositorySettings?: Asset;
    /**
     * Define the settings used by Deadline Secrets Management, a feature introduced in Deadline 10.1.10 for securely managing storage
     * and access of Secrets for your render farm.
     * More details at:
     * https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/secrets-management/deadline-secrets-management.html
     * @default: Secrets Management will be enabled and a username and password will be automatically generated if none are supplied.
     */
    readonly secretsManagementSettings?: SecretsManagementProps;
}
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *   system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *   Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *   Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *   require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *   submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *   that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *   to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *   access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *   Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *   Repository construct instead.
 */
export declare class Repository extends Construct implements IRepository {
    /**
     * Default file system mount path for repository
     */
    private static DEFAULT_FILE_SYSTEM_MOUNT_PATH;
    /**
     * Default installation prefix for deadline repository.
     */
    private static DEFAULT_REPO_PREFIX;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static DEFAULT_LOG_GROUP_PREFIX;
    /**
     * How often Cloudwatch logs will be flushed.
     */
    private static CLOUDWATCH_LOG_FLUSH_INTERVAL;
    /**
     * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
     * containers.
     */
    private static ECS_VOLUME_NAME;
    /**
     * The default number of DocDB instances if one isn't provided in the props.
     */
    private static DEFAULT_NUM_DOCDB_INSTANCES;
    /**
     * Default retention period for DocumentDB automated backups if one isn't provided in the props.
     */
    private static DEFAULT_DATABASE_RETENTION_PERIOD;
    /**
     * The Repository owner is 1000:1000 (ec2-user on AL2).
     */
    private static REPOSITORY_OWNER;
    /**
     * Default username for auto generated admin credentials in Secret Manager.
     */
    private static DEFAULT_SECRETS_MANAGEMENT_USERNAME;
    /**
     * @inheritdoc
     */
    readonly rootPrefix: string;
    /**
     * @inheritdoc
     */
    readonly version: IVersion;
    /**
     * Connection object for the database for this repository.
     */
    readonly databaseConnection: DatabaseConnection;
    /**
     * The Linux-mountable filesystem that will store the Deadline repository filesystem contents.
     */
    readonly fileSystem: IMountableLinuxFilesystem;
    /**
     * The underlying Amazon Elastic File System (EFS) used by the Repository.
     * This is only defined if this Repository created its own filesystem, otherwise it will be `undefined`.
     */
    readonly efs?: EfsFileSystem;
    /**
     * The autoscaling group for this repository's installer-running instance.
     */
    private readonly installerGroup;
    /**
     * @inheritdoc
     */
    readonly secretsManagementSettings: SecretsManagementProps;
    constructor(scope: Construct, id: string, props: RepositoryProps);
    /**
     * @inheritdoc
     */
    configureClientECS(props: ECSDirectConnectProps): IContainerDirectRepositoryConnection;
    /**
     * @inheritdoc
     */
    configureClientInstance(props: InstanceDirectConnectProps): void;
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    private setupDirectConnect;
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    private configureCloudWatchLogStream;
    private configureSelfTermination;
    private configureRepositoryInstallerScript;
}
