"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const render_queue_1 = require("./render-queue");
const secrets_management_ref_1 = require("./secrets-management-ref");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    /**
     * The Render Queue, all jobs and Workers will trigger the events for this plugin.
     */
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    /**
     * No events are triggered for the plugin.
     */
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    /**
     * Standard logging level.
     */
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    /**
     * Detailed logging about the inner workings of the Spot Event Plugin.
     */
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    /**
     * All Verbose logs plus additional information on AWS API calls that are used.
     */
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    /**
     * No logging enabled.
     */
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    /**
     * Only start 1 Spot instance for the pre job task and ignore any other tasks for that job until the pre job task is completed.
     */
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    /**
     * Do not take the pre job task into account when calculating target capacity.
     */
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    /**
     * Treat the pre job task like a regular job queued task.
     */
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status
 * if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo1";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo2";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo3";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo4";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo5";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo6";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo7";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo8";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo9";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from 'aws-cdk-lib/aws-ec2';
 * import { AwsCustomerAgreementAndIpLicenseAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *   version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *   version,
 *   // Change this to AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to accept the terms
 *   // of the AWS Customer Agreement and AWS Intellectual Property License.
 *   userAwsCustomerAgreementAndIpLicenseAcceptance: AwsCustomerAgreementAndIpLicenseAcceptance.USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   vpc,
 *   images: images.forRenderQueue(),
 *   repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *   vpc,
 *   renderQueue,
 *   deadlineGroups: ['group_name'],
 *   instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *   workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *   naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *   vpc,
 *   renderQueue: renderQueue,
 *   spotFleets: [fleet],
 *   configuration: {
 *     enableResourceTracker: true,
 *   },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *  - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *  - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *  - A policy to pass a fleet and instance role
 *  - A policy to create tags for spot fleet requests
 *
 * The Spot Fleet Requests that this construct configures Deadline to create will always use the latest version of the
 * corresponding EC2 Launch Template that was created for them.
 *
 * ![architecture diagram](/diagrams/deadline/ConfigureSpotEventPlugin.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 * - EC2 Launch Templates for each Spot Event Plugin fleet.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *   and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *   execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *   or execute this Lambda.
 */
class ConfigureSpotEventPlugin extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = constructs_1.Construct.isConstruct(props.renderQueue) ? aws_cdk_lib_1.Stack.of(props.renderQueue).region : aws_cdk_lib_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: aws_cdk_lib_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        props.renderQueue.certChain?.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: props.configuration?.awsInstanceStatus ?? SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: props.configuration?.deleteEC2SpotInterruptedWorkers ?? false,
            DeleteTerminatedSlaves: props.configuration?.deleteSEPTerminatedWorkers ?? false,
            IdleShutdown: props.configuration?.idleShutdown?.toMinutes({ integral: true }) ?? 10,
            Logging: props.configuration?.loggingLevel ?? SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: props.configuration?.preJobTaskMode ?? SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: props.configuration?.region ?? region,
            ResourceTracker: props.configuration?.enableResourceTracker ?? true,
            StaggerInstances: props.configuration?.maximumInstancesStartedPerCycle ?? 50,
            State: props.configuration?.state ?? SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: props.configuration?.strictHardCap ?? false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const deadlineGroups = Array.from(new Set(props.spotFleets?.map(fleet => fleet.deadlineGroups).reduce((p, c) => p.concat(c), [])));
        const deadlinePools = Array.from(new Set(props.spotFleets?.map(fleet => fleet.deadlinePools).reduce((p, c) => p?.concat(c ?? []), [])));
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: props.renderQueue.certChain?.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
            deadlineGroups,
            deadlinePools,
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        if (props.spotFleets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.spotFleets.forEach(spotFleet => {
                if (spotFleet.defaultSubnets) {
                    aws_cdk_lib_1.Annotations.of(spotFleet).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
                }
                props.renderQueue.configureSecretsManagementAutoRegistration({
                    dependent: resource,
                    role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                    registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                    vpc: props.vpc,
                    vpcSubnets: spotFleet.subnets,
                });
            });
        }
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return aws_cdk_lib_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each configuration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchTemplateConfigs: fleet._launchTemplateConfigs,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: fleet.validUntil?.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group.toLowerCase()]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "1.1.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,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