"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Generate cert', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    // Expect the custom resource for cert generation
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Cannot have a CertificateValidFor property if not given one. Adding one
    // would cause existing certificates to be re-generated on re-deploy, and thus
    // risk breaking customer's setups.
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', assertions_1.Match.not({
        CertificateValidFor: assertions_1.Match.anyValue(),
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
    // Expect the DynamoDB table used for custom resource tracking
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DynamoDB::Table', 1);
    // Expect a Secret used to store the cert passphrase
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::SecretsManager::Secret', 1);
    // Expect a policy that can interact with DynamoDB and SecretsManager
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^CertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-F53F5427': 'f53f5427b2e9eb4739661fcc0b249b6e',
                        },
                    },
                }),
            ]),
        },
    });
    // Expect no KMS key for encryption
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 0);
    // Expect Lambda for doing the cert generation to use the generate() handler and openssl layer
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
        Layers: assertions_1.Match.arrayWith([
            assertions_1.Match.stringLikeRegexp('^arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:.*'),
        ]),
        Environment: {
            Variables: {
                DATABASE: {
                    Ref: assertions_1.Match.stringLikeRegexp('^CertTable.*'),
                },
            },
        },
    });
    // Expect Table to have point in time recovery set to true
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DynamoDB::Table', {
        PointInTimeRecoverySpecification: {
            PointInTimeRecoveryEnabled: true,
        },
    });
    // Should not be any errors.
    assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoError(`/${cert.node.path}`, assertions_1.Match.anyValue());
});
test('Generate cert, all options set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = {
        cn: 'testCN',
        o: 'testO',
        ou: 'testOu',
    };
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    const signingCertificate = new x509_certificate_1.X509CertificatePem(stack, 'SigningCert', { subject });
    new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        encryptionKey,
        signingCertificate,
        validFor: 3000,
    });
    // Expect the custom resource for cert generation
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'testO',
            OU: 'testOu',
        },
        SigningCertificate: {
            Cert: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: assertions_1.Match.stringLikeRegexp('^SigningCertPassphrase.*'),
            },
            CertChain: '',
        },
        CertificateValidFor: '3000',
    });
    // Expect the resource for converting to PKCS #12 not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
    // Expect a policy that can interact with DynamoDB and SecretsManager for the signing cert's custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^CertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-B2B09A60': 'b2b09a6086e87fe14005f4e0b800e4f0',
                        },
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: 'SigningCertPassphrase42F0BC4F',
                    },
                }),
            ]),
        },
    });
    // Expect a policy that can interact with DynamoDB and SecretsManager for the cert's custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^SigningCertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-BA0FA489': 'ba0fa4898b2088c5b25f15075f605300',
                        },
                    },
                }),
            ]),
        }),
    });
    // Expect Lambda for doing the cert generation to use the generate() handler
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
    });
});
test('Grant cert read', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantCertRead(grantable);
    // Expect the custom resource to be created
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Expect the grantCertRead() to add this policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
            ],
        },
    });
    // Expect the grantCertRead() not to add this full read policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', assertions_1.Match.not({
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
});
test('Grant full read', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantFullRead(grantable);
    // Expect the custom resource to be created
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Expect the grantFullRead() to add this policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', assertions_1.Match.not({
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
});
test('Validating expiry', () => {
    // GIVEN
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: 0,
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasError(`/${cert.node.path}`, 'Certificates must be valid for at least one day.');
});
test('Validating expiry with token', () => {
    // GIVEN
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: aws_cdk_lib_1.Lazy.number({
            produce() {
                return 0;
            },
        }),
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoError(`/${cert.node.path}`, assertions_1.Match.anyValue());
});
test('Convert to PKCS #12', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    const pkcs12Cert = new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', { sourceCertificate: cert });
    const pkcs12CertPassphraseID = stack.getLogicalId(pkcs12Cert.passphrase.node.defaultChild);
    // Expect the PKCS #12 custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    });
    // Expect the source certificate (custom resource)
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509Generator', 1);
    // Expect the PKCS #12 to have a password secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    });
    // Expect the PKCS #12 resource to have a policy with access to the X.509 resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: pkcs12CertPassphraseID,
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-71090F78': '71090f7809ce64f7c970cb645d4d473c',
                        },
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        }),
    });
    // Expect no KMS key
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 0);
    // Expect the Lambda for converting the PEM to PKCS 12
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    });
});
test('Convert to PKCS #12, use KMS', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = { cn: 'testCN' };
    const sourceCertificate = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(sourceCertificate.passphrase.node.defaultChild);
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', {
        sourceCertificate,
        encryptionKey,
    });
    // Expect the PKCS #12 custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    });
    // Expect the source certificate (custom resource)
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509Generator', 1);
    // Expect the PKCS #12 to have a password secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    });
    // Expect a KMS key for encryption
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 1);
    // Expect the Lambda for converting the PEM to PKCS #12
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    });
});
//# sourceMappingURL=data:application/json;base64,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