"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
let stack;
let vpc;
const instanceType = aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
const machineImage = new aws_ec2_1.AmazonLinuxImage();
beforeEach(() => {
    stack = new aws_cdk_lib_1.Stack();
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
});
test('Grant SSM permissions to Instance', () => {
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType,
        machineImage,
    });
    lib_1.SessionManagerHelper.grantPermissionsTo(instance);
    const instanceRole = stack.getLogicalId(instance.role.node.defaultChild);
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'ssmmessages:CreateControlChannel',
                        'ssmmessages:CreateDataChannel',
                        'ssmmessages:OpenControlChannel',
                        'ssmmessages:OpenDataChannel',
                        'ssm:UpdateInstanceInformation',
                    ],
                    Effect: 'Allow',
                    Resource: '*',
                },
            ],
        },
        Roles: [{ Ref: instanceRole }],
    });
});
test('Grant SSM permissions to ASG', () => {
    const asg = new aws_autoscaling_1.AutoScalingGroup(stack, 'ASG', {
        vpc,
        instanceType,
        machineImage,
    });
    lib_1.SessionManagerHelper.grantPermissionsTo(asg);
    const asgRole = stack.getLogicalId(asg.role.node.defaultChild);
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'ssmmessages:CreateControlChannel',
                        'ssmmessages:CreateDataChannel',
                        'ssmmessages:OpenControlChannel',
                        'ssmmessages:OpenDataChannel',
                        'ssm:UpdateInstanceInformation',
                    ],
                    Effect: 'Allow',
                    Resource: '*',
                },
            ],
        },
        Roles: [{ Ref: asgRole }],
    });
});
//# sourceMappingURL=data:application/json;base64,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