"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const pad_efs_storage_1 = require("../lib/pad-efs-storage");
describe('Test PadEfsStorage', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let accessPoint;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
        accessPoint = new aws_efs_1.AccessPoint(stack, 'AccessPoint', {
            fileSystem: efsFS,
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            path: '/SpaceFillingFiles',
            posixUser: {
                uid: '1000',
                gid: '1000',
            },
        });
    });
    test('Create with defaults', () => {
        // WHEN
        const pad = new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: aws_cdk_lib_1.Size.gibibytes(20),
        });
        const sg = pad.node.findChild('LambdaSecurityGroup');
        const diskUsage = pad.node.findChild('DiskUsage');
        const padFilesystem = pad.node.findChild('PadFilesystem');
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::Lambda::Function', {
            Properties: {
                FileSystemConfigs: [
                    {
                        Arn: stack.resolve(accessPoint.accessPointArn),
                        LocalMountPath: '/mnt/efs',
                    },
                ],
                Handler: 'pad-efs-storage.getDiskUsage',
                Runtime: 'nodejs16.x',
                Timeout: 300,
                VpcConfig: {
                    SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
                    SubnetIds: [
                        {
                            Ref: 'VpcPrivateSubnet1Subnet536B997A',
                        },
                        {
                            Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                        },
                    ],
                },
            },
            DependsOn: assertions_1.Match.arrayWith([stack.getLogicalId(accessPoint.node.defaultChild)]),
        });
        assertions_1.Template.fromStack(stack).hasResource('AWS::Lambda::Function', {
            Properties: {
                FileSystemConfigs: [
                    {
                        Arn: stack.resolve(accessPoint.accessPointArn),
                        LocalMountPath: '/mnt/efs',
                    },
                ],
                Handler: 'pad-efs-storage.padFilesystem',
                Runtime: 'nodejs16.x',
                Timeout: 900,
                VpcConfig: {
                    SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
                    SubnetIds: [
                        {
                            Ref: 'VpcPrivateSubnet1Subnet536B997A',
                        },
                        {
                            Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                        },
                    ],
                },
            },
            DependsOn: assertions_1.Match.arrayWith([stack.getLogicalId(accessPoint.node.defaultChild)]),
        });
        const lambdaRetryCatch = {
            Retry: [
                {
                    ErrorEquals: [
                        'Lambda.ServiceException',
                        'Lambda.AWSLambdaException',
                        'Lambda.SdkClientException',
                    ],
                    IntervalSeconds: 2,
                    MaxAttempts: 6,
                    BackoffRate: 2,
                },
            ],
            Catch: [
                {
                    ErrorEquals: ['States.ALL'],
                    Next: 'Fail',
                },
            ],
        };
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::StepFunctions::StateMachine', {
            DefinitionString: stack.resolve(JSON.stringify({
                StartAt: 'QueryDiskUsage',
                States: {
                    QueryDiskUsage: {
                        Next: 'BranchOnDiskUsage',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Determine the number of GB currently stored in the EFS access point',
                        TimeoutSeconds: 300,
                        ResultPath: '$.diskUsage',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${diskUsage.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    GrowTask: {
                        Next: 'QueryDiskUsage',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Add up to 20 numbered 1GB files to the EFS access point',
                        TimeoutSeconds: 900,
                        ResultPath: '$.null',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${padFilesystem.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    BranchOnDiskUsage: {
                        Type: 'Choice',
                        Choices: [
                            {
                                Variable: '$.diskUsage.Payload',
                                NumericLessThanPath: '$.desiredPadding',
                                Next: 'GrowTask',
                            },
                            {
                                Variable: '$.diskUsage.Payload',
                                NumericGreaterThanPath: '$.desiredPadding',
                                Next: 'ShrinkTask',
                            },
                        ],
                        Default: 'Succeed',
                    },
                    Succeed: {
                        Type: 'Succeed',
                    },
                    ShrinkTask: {
                        Next: 'Succeed',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Remove 1GB numbered files from the EFS access point to shrink the padding',
                        TimeoutSeconds: 900,
                        ResultPath: '$.null',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${padFilesystem.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    Fail: {
                        Type: 'Fail',
                    },
                },
            })),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::AWS', {
            Create: {
                'Fn::Join': [
                    '',
                    [
                        '{"action":"startExecution","service":"StepFunctions","apiVersion":"2016-11-23","region":"',
                        {
                            Ref: 'AWS::Region',
                        },
                        '","physicalResourceId":{"responsePath":"executionArn"},"parameters":{"stateMachineArn":"',
                        {
                            Ref: 'PadEfsStateMachineDA538E87',
                        },
                        '","input":"{\\"desiredPadding\\":20}"}}',
                    ],
                ],
            },
            Update: {
                'Fn::Join': [
                    '',
                    [
                        '{"action":"startExecution","service":"StepFunctions","apiVersion":"2016-11-23","region":"',
                        {
                            Ref: 'AWS::Region',
                        },
                        '","physicalResourceId":{"responsePath":"executionArn"},"parameters":{"stateMachineArn":"',
                        {
                            Ref: 'PadEfsStateMachineDA538E87',
                        },
                        '","input":"{\\"desiredPadding\\":20}"}}',
                    ],
                ],
            },
        });
    });
    test('Set desiredPadding', () => {
        // WHEN
        const desiredPadding = 200;
        new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: aws_cdk_lib_1.Size.gibibytes(desiredPadding),
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::AWS', {
            Create: {
                'Fn::Join': [
                    '',
                    assertions_1.Match.arrayWith([`","input":"{\\"desiredPadding\\":${desiredPadding}}"}}`]),
                ],
            },
            Update: {
                'Fn::Join': [
                    '',
                    assertions_1.Match.arrayWith([`","input":"{\\"desiredPadding\\":${desiredPadding}}"}}`]),
                ],
            },
        });
    });
    test('Throws on bad desiredPadding', () => {
        // GIVEN
        const pad = new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: aws_cdk_lib_1.Size.mebibytes(100),
        });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasError(`/${pad.node.path}`, 'Failed to round desiredSize to an integer number of GiB. The size must be in GiB.');
    });
    test('Provide SecurityGroup', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'TestSG', {
            vpc,
        });
        // WHEN
        new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: aws_cdk_lib_1.Size.gibibytes(20),
            securityGroup: sg,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'pad-efs-storage.getDiskUsage',
            VpcConfig: {
                SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'pad-efs-storage.padFilesystem',
            VpcConfig: {
                SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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