"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableBlockVolume', () => {
    let stack;
    let vpc;
    let ebsVol;
    let instance;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        ebsVol = new aws_ec2_1.Volume(stack, 'EBS', {
            availabilityZone: vpc.availabilityZones[0],
            size: aws_cdk_lib_1.Size.gibibytes(5),
        });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance role has the correct permissions to get & run the script
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assertions_1.Match.arrayWith([
                    {
                        Effect: 'Allow',
                        Action: 'ec2:DescribeVolumes',
                        Resource: '*',
                    },
                    {
                        Effect: 'Allow',
                        Action: 'ec2:AttachVolume',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':volume/',
                                        {
                                            Ref: 'EBSB2DACE72',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantAttach-274F1FE7E3': '274f1fe7e3ac0d55b2991dd00242f5ea',
                            },
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Bucket,
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Bucket,
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ]),
            },
        });
        // Make sure we download the mountEFS script asset bundle
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip' '/tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip'`;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`unzip /tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip`)));
        // Make sure we run the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/fs',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs r')));
    });
    test('non-default filesystem type', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
            volumeFormat: lib_1.BlockVolumeFormat.EXT4,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} ext4 /mnt/fs r')));
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs rw,option1,option2')));
    });
    test('fails if non-construct target', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        class FakeTarget {
            constructor() {
                this.connections = instance.connections;
                this.osType = instance.osType;
                this.userData = instance.userData;
                this.grantPrincipal = instance.grantPrincipal;
                this.node = instance.node;
            }
        }
        const fakeTarget = new FakeTarget();
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(fakeTarget, {
                location: '/mnt/fs',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError(/Target instance must be a construct./);
    });
    test('asset is singleton', () => {
        // GIVEN
        const mount1 = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        const mount2 = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip'`;
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = userData.match(regex) ?? [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    test('.usesUserPosixPermissions() is true', () => {
        // GIVEN
        const mount1 = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        const usesUserPosixPermissions = mount1.usesUserPosixPermissions();
        // THEN
        expect(usesUserPosixPermissions).toEqual(true);
    });
});
//# sourceMappingURL=data:application/json;base64,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