"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
let stack;
let secretCert;
let certPassphraseID;
let importedAcmCertificate;
describe('ImportedAcmCertificate', () => {
    beforeEach(() => {
        // GIVEN
        stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
        secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
            subject: { cn: 'Server' },
        });
        certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
        // WHEN
        importedAcmCertificate = new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
            cert: secretCert.cert,
            certChain: secretCert.certChain,
            key: secretCert.key,
            passphrase: secretCert.passphrase,
        });
    });
    test('creates Custom::RFDK_AcmImportedCertificate', () => {
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_AcmImportedCertificate', {
            X509CertificatePem: {
                Cert: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Cert',
                    ],
                },
                Key: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Key',
                    ],
                },
                Passphrase: {
                    Ref: certPassphraseID,
                },
                CertChain: '',
            },
            Tags: [
                {
                    Key: 'AcmCertImport-F4E2ABF9',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
                {
                    Key: 'Name',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
            ],
        });
    });
    describe('creates AWS::DynamoDB::Table for database', () => {
        test('with PhysicalID partition key', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DynamoDB::Table', {
                AttributeDefinitions: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'PhysicalId',
                        AttributeType: 'S',
                    },
                ]),
                KeySchema: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'PhysicalId',
                        KeyType: 'HASH',
                    },
                ]),
            });
        });
        test('with CustomResource sort key', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DynamoDB::Table', {
                AttributeDefinitions: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'CustomResource',
                        AttributeType: 'S',
                    },
                ]),
                KeySchema: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'CustomResource',
                        KeyType: 'RANGE',
                    },
                ]),
            });
        });
    });
    test('creates AWS::IAM::Policy', () => {
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    assertions_1.Match.objectLike({
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:ConditionCheckItem',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                            'dynamodb:DescribeTable',
                        ],
                    }),
                    assertions_1.Match.objectLike({
                        Action: 'dynamodb:DescribeTable',
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Cert',
                            ],
                        },
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Key',
                            ],
                        },
                    }),
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: certPassphraseID,
                        },
                    },
                    assertions_1.Match.objectLike({
                        Action: [
                            'acm:AddTagsToCertificate',
                            'acm:ImportCertificate',
                        ],
                        Condition: {
                            StringEquals: {
                                'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                            },
                        },
                        Resource: '*',
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'acm:DeleteCertificate',
                            'acm:DescribeCertificate',
                            'acm:GetCertificate',
                        ],
                        Resource: '*',
                    }),
                ],
            },
        });
    });
    describe('custom resource lambda function', () => {
        test('uses correct handler', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Handler: 'x509-certificate.importCert',
            });
        });
        test('uses RFDK lambda layer', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Layers: assertions_1.Match.arrayWith([
                    assertions_1.Match.stringLikeRegexp('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:.*'),
                ]),
            });
        });
        test('sets DATABASE environment variable', () => {
            // GIVEN
            const table = importedAcmCertificate.node.findChild('Table');
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Environment: {
                    Variables: assertions_1.Match.objectLike({
                        DATABASE: stack.resolve(table.tableName),
                    }),
                },
            });
        });
    });
    describe('applyRemovalPolicy', () => {
        test('default RemovalPolicy is Delete', () => {
            assertions_1.Template.fromStack(stack).hasResource('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Delete',
                UpdateReplacePolicy: 'Delete',
            });
        });
        test('Different policy can be applied', () => {
            importedAcmCertificate.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
            assertions_1.Template.fromStack(stack).hasResource('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            });
        });
    });
    describe('metricDaysToExpiry', () => {
        let metricExpiry;
        beforeEach(() => {
            // GIVEN
            metricExpiry = importedAcmCertificate.metricDaysToExpiry();
        });
        test('uses certificate ARN', () => {
            // THEN
            expect(metricExpiry.dimensions?.CertificateArn).toEqual(importedAcmCertificate.certificateArn);
        });
        test('uses correct metric', () => {
            // THEN
            expect(metricExpiry.metricName).toEqual('DaysToExpiry');
            expect(metricExpiry.namespace).toEqual('AWS/CertificateManager');
            expect(metricExpiry.statistic).toEqual(aws_cloudwatch_1.Stats.MINIMUM);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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