"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    // Make sure we have a known osType. The error message is pretty obtuse if we don't:
    //  The "path" argument must be of type string. Received undefined
    if (ScriptPathPrefix[osType] === undefined || ScriptExtension[osType] == undefined) {
        throw Error(`Unknown osType: ${osType}`);
    }
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * - An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure
     * convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *   bash/
     *     script-one.sh
     *     script-two.sh
     *   powershell
     *     script-one.ps1
     *     script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}
     * @param id The construct id for the created {@link ScriptAsset}
     * @param scriptParams The parameters that are used to compute the conventional path to the script file
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
_a = JSII_RTTI_SYMBOL_1;
ScriptAsset[_a] = { fqn: "aws-rfdk.ScriptAsset", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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