"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableFsxLustre = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon FSx for Lustre File System onto
 * an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class MountableFsxLustre {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const mountName = this.props.fileset ? path.posix.join(this.props.filesystem.mountName, this.props.fileset) : this.props.filesystem.mountName;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, 'bash ./installLustreClient.sh', `bash ./mountFsxLustre.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountName} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions() {
        return true;
    }
    /**
     * Fetch the Asset singleton for the FSx for Lustre mounting scripts, or generate it if needed.
     */
    mountAssetSingleton(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const uuid = '0db888da-5901-4948-aaa5-e71c541c8060';
        const uniqueId = 'MountableFsxLustreAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountFsxLustre.sh', '!installLustreClient.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableFsxLustre = MountableFsxLustre;
_a = JSII_RTTI_SYMBOL_1;
MountableFsxLustre[_a] = { fqn: "aws-rfdk.MountableFsxLustre", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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