"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPermissionsHelper = void 0;
const mountable_filesystem_1 = require("./mountable-filesystem");
/**
 * This is an internal-only class that can be used to convert enum values from
 * {@link MountPermissions} into strings, or whatever, that are needed by OS-specific
 * mount utilities.
 */
class MountPermissionsHelper {
    /**
     * Convert the given permission into the appropriate mount option for Linux's mount command.
     *
     * @param permission The permission to convert. Defaults to {@link MountPermissions.READWRITE} if not defined.
     */
    static toLinuxMountOption(permission) {
        permission = permission ?? mountable_filesystem_1.MountPermissions.READWRITE;
        switch (permission) {
            case mountable_filesystem_1.MountPermissions.READONLY:
                return 'ro';
            case mountable_filesystem_1.MountPermissions.READWRITE:
                return 'rw';
        }
        throw new Error(`Unhandled MountPermission: ${permission}`);
    }
    /**
     * Convert the given permission into the appropriate list of IAM actions allowed on the EFS FileSystem required for
     * the mount.
     *
     * @param permission The permission to convert. Defaults to {@link MountPermissions.READWRITE} if not defined.
     */
    static toEfsIAMActions(permission) {
        permission = permission ?? mountable_filesystem_1.MountPermissions.READWRITE;
        const iamActions = [
            'elasticfilesystem:ClientMount',
        ];
        if (permission === mountable_filesystem_1.MountPermissions.READWRITE) {
            iamActions.push('elasticfilesystem:ClientWrite');
        }
        return iamActions;
    }
}
exports.MountPermissionsHelper = MountPermissionsHelper;
//# sourceMappingURL=data:application/json;base64,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