"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and
 * executing commands against it. To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda function that is created by this resource has access to both the MongoDB administrator credentials,
 *   and the MongoDB application port. An attacker that can find a way to modify and execute this lambda could use it to
 *   modify or read any data in the database. You should not grant any additional actors/principals the ability to modify
 *   or execute this Lambda.
 */
class MongoDbPostInstallSetup extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        props.users.x509AuthUsers?.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
            description: `Used by a MongoDbPostInstallSetup ${aws_cdk_lib_1.Names.uniqueId(this)} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: aws_cdk_lib_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        props.users.passwordAuthUsers?.forEach(secret => secret.grantRead(lamdbaFunc));
        props.users.x509AuthUsers?.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: props.users.passwordAuthUsers?.map(secret => secret.secretArn),
            X509AuthUsers: props.users.x509AuthUsers?.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
_a = JSII_RTTI_SYMBOL_1;
MongoDbPostInstallSetup[_a] = { fqn: "aws-rfdk.MongoDbPostInstallSetup", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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