"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const _1 = require("./");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB. The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A-Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret in AWS SecretsManager that contains the administrator credentials for MongoDB.
 * - An encrypted Amazon Elastic Block Store (EBS) Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * Security Considerations
 * ------------------------
 * - The administrator credentials for MongoDB are stored in a Secret within AWS SecretsManager. You must strictly limit
 *   access to this secret to only entities that require it.
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The EBS Volume that is created by, or provided to, this construct is used to store the contents of your MongoDB data. To
 *   protect the sensitive data in your database, you should not grant access to this EBS Volume to any principal or instance
 *   other than the instance created by this construct. Furthermore, we recommend that you ensure that the volume that is
 *   used for this purpose is encrypted at rest.
 * - This construct uses this package's {@link StaticPrivateIpServer}, {@link MongoDbInstaller}, {@link CloudWatchAgent},
 *   {@link ExportingLogGroup}, and {@link MountableBlockVolume}. Security considerations that are outlined by the documentation
 *   for those constructs should also be taken into account.
 */
class MongoDbInstance extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: props.instanceType ?? new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: aws_cdk_lib_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = props.mongoDb.adminUser ?? new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${aws_cdk_lib_1.Names.uniqueId(this)}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = props.mongoDb.mongoDataVolume?.volume ?? new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...props.mongoDb.mongoDataVolume?.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        securityGroups?.forEach(securityGroup => this.server.autoscalingGroup.addSecurityGroup(securityGroup));
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *     - The cloud-init log
     *     - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        const prefix = logGroupProps?.logGroupPrefix ?? MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and
     * to to start mongod running.
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR}" != "" && sudo umount "${MONGO_SETUP_DIR}"');
    }
}
exports.MongoDbInstance = MongoDbInstance;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstance[_a] = { fqn: "aws-rfdk.MongoDbInstance", version: "1.1.0" };
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = aws_cdk_lib_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = aws_cdk_lib_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = aws_cdk_lib_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,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