"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogGroupFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const exporting_log_group_1 = require("./exporting-log-group");
/**
 * This factory will return an ILogGroup based on the configuration provided to it. The LogGroup will either be
 * wrapped in a LogRetention from the aws-lambda package that has the ability to look up and reuse an existing LogGroup
 * or an ExportingLogGroup that uses a LogRetention and adds additional functionality to export the logs to S3.
 */
class LogGroupFactory {
    /**
     * Either create a new LogGroup given the LogGroup name, or return the existing LogGroup.
     */
    static createOrFetch(scope, logWrapperId, logGroupName, props) {
        const fullLogGroupName = props?.logGroupPrefix ? `${props.logGroupPrefix}${logGroupName}` : logGroupName;
        const retention = props?.retention ? props.retention : LogGroupFactory.DEFAULT_LOG_RETENTION;
        // Define log retention retry options to reduce the risk of the rate exceed error
        // as the default create log group TPS is only 5. Make sure to set the timeout of log retention function
        // to be greater than total retry time. That's because if the function that is used for a custom resource
        // doesn't exit properly, it'd end up in retries and may take cloud formation an hour to realize that
        // the custom resource failed.
        const logGroup = props?.bucketName
            ? new exporting_log_group_1.ExportingLogGroup(scope, logWrapperId, {
                bucketName: props.bucketName,
                logGroupName: fullLogGroupName,
                retention: props.retention,
            }).logGroup
            : aws_logs_1.LogGroup.fromLogGroupArn(scope, `${logGroupName}LogGroup`, new aws_logs_1.LogRetention(scope, logWrapperId, {
                logGroupName: fullLogGroupName,
                retention,
                logRetentionRetryOptions: {
                    base: aws_cdk_lib_1.Duration.millis(200),
                    maxRetries: 7,
                },
            }).logGroupArn);
        // referenced from cdk code: https://github.com/aws/aws-cdk/blob/v2.33.0/packages/@aws-cdk/aws-logs/lib/log-retention.ts#L116
        const logRetentionFunctionConstructId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const logRetentionFunction = aws_cdk_lib_1.Stack.of(scope).node.findChild(logRetentionFunctionConstructId);
        const cfnFunction = logRetentionFunction.node.defaultChild;
        cfnFunction.addPropertyOverride('Timeout', 30);
        return logGroup;
    }
}
exports.LogGroupFactory = LogGroupFactory;
_a = JSII_RTTI_SYMBOL_1;
LogGroupFactory[_a] = { fqn: "aws-rfdk.LogGroupFactory", version: "1.1.0" };
/**
 * Default retention period to hold logs in CloudWatch for.
 */
LogGroupFactory.DEFAULT_LOG_RETENTION = aws_logs_1.RetentionDays.THREE_DAYS;
//# sourceMappingURL=data:application/json;base64,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