"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AWSLimitExhaustedError = exports.LoadBalancerFactory = void 0;
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const health_monitor_1 = require("./health-monitor");
/**
 * This class is responsible for managing the statistics for all the
 * load balancers created in this construct. It is also responsible to search
 * for the finding the first Load balancer/Listener which can accomodate the
 * worker-fleet based on its size.
 *
 * A typical load balancer hierarchy looks like following:
 *  |__ Load Balancer 1
 *  |         |____________Listener 1
 *  |         |                 |_______Target Group 1 ------- Target/Fleet
 *  |         |                 |_______Target Group 2 ------- Target/Fleet
 *  |         |
 *  |         |____________Listener 2
 *  |                           |_______Target Group 1 ------- Target/Fleet
 *  |                           |_______Target Group 2 ------- Target/Fleet
 *  |
 *  |__ Load Balancer 2
 *            |____________Listener 1
 *            |                 |_______Target Group 1 ------- Target/Fleet
 *            |                 |_______Target Group 2 ------- Target/Fleet
 *            |
 *            |____________Listener 2
 *                              |_______Target Group 1 ------- Target/Fleet
 *                              |_______Target Group 2 ------- Target/Fleet
 *
 *  Components:
 *  1. LoadBalancerFactory: This is the root node of the tree. It contains the
 *     map of load balancer to its managers. It is responsible for creating a
 *     new load balancer if required. It delegates the registerFleet calls to
 *     downstream and returns parent load balancer, listener and target group
 *     of the registered fleet if the registration was successful
 *
 *  2. LoadBalancerManager: This class manages a single load balancer. It
 *     contains a map of all the listeners->manager. It also contains the component
 *     counts like listener, target group and target count. It delegates the
 *     registration call to downstream listeners and updates the stats when
 *     the registration is successful. It returns the parent listener and
 *     target group on successful registration.
 *
 *  3. ListenerManager: This class managers a single Listener. It contains a map
 *     of all of its target groups to its associated fleet. It also contains the
 *     component counts. It returns the target group on registration.
 */
class LoadBalancerFactory {
    constructor(healthMonitorScope, vpc) {
        this.loadBalancerMap = new Map();
        this.healthMonitorScope = healthMonitorScope;
        this.vpc = vpc;
    }
    static getAccountLimit(limitName, defaultValue, elbAccountLimits) {
        if (!elbAccountLimits) {
            return defaultValue;
        }
        const foundLimit = elbAccountLimits.find(limit => limit.name === limitName);
        if (!foundLimit) {
            return defaultValue;
        }
        return foundLimit.max;
    }
    /**
     * This method scans all the load balancers and its listeners and registers the fleet
     * to the load balancer and/or listener which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the load balancer, listener
     * and target group to which the fleet was registered.
     *
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(fleet, healthCheckConfig, healthMonitorProps) {
        let loadBalancerParent = null;
        let listenerParent = null;
        let targetGroupParent = null;
        // iterate through each load balancer and try registering to each one.
        for (const [loadBalancer, loadBalancerMeta] of this.loadBalancerMap.entries()) {
            try {
                const { listener, targetGroup } = loadBalancerMeta.registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps);
                loadBalancerParent = loadBalancer;
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        // Check if fleet was not registered.
        if (!loadBalancerParent) {
            // If this section is reached, no load balancer was found which could
            // accommodate fleet, create a new one and register
            loadBalancerParent = this.createLoadBalancer(this.healthMonitorScope, this.loadBalancerMap.size, healthMonitorProps);
            const loadBalancerManager = new LoadBalancerManager();
            // Add it to the map
            this.loadBalancerMap.set(loadBalancerParent, loadBalancerManager);
            // try registering the fleet to the new load balancer
            try {
                const { listener, targetGroup } = loadBalancerManager.registerWorkerFleet(loadBalancerParent, fleet, healthCheckConfig, healthMonitorProps);
                listenerParent = listener;
                targetGroupParent = targetGroup;
            }
            catch (e) {
                throw e;
            }
        }
        /* istanbul ignore next */
        if (!loadBalancerParent || !listenerParent || !targetGroupParent) {
            /* istanbul ignore next */
            throw new Error('Fleet registered successfully but a parent was found null');
        }
        return {
            loadBalancer: loadBalancerParent,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new load balancer within the given scope.
     *
     * @param scope
     * @param loadBalancerindex
     */
    createLoadBalancer(scope, loadBalancerindex, healthMonitorProps) {
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(scope, `ALB_${loadBalancerindex}`, {
            vpc: this.vpc,
            internetFacing: false,
            vpcSubnets: healthMonitorProps.vpcSubnets,
            deletionProtection: healthMonitorProps.deletionProtection ?? true,
            securityGroup: healthMonitorProps.securityGroup,
        });
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        return loadBalancer;
    }
}
exports.LoadBalancerFactory = LoadBalancerFactory;
LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER = 50;
LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER = 1000;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER = 5;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER = 100;
/**
 * This class manages the properties of a single load balancer and its statistics.
 * It is also responsible to scan through all the listeners registered under it
 * and register the given fleet.
 */
class LoadBalancerManager {
    constructor() {
        this.listenerMap = new Map();
        this.loadBalancerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accomodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the listener
     * and target group to which the fleet was registered.
     *
     * @param loadBalancer
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps) {
        // this initializes with 0 and keeps the track of all components
        // newly added down the hierarchy.
        const statsDelta = new LoadBalancerComponentStats();
        // Do all the load balancer level service limit checks first
        // check for target limit in load balancer
        const targetPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('targets-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetCount + fleet.targetCapacity) > targetPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "targets-per-application-load-balancer" reached. Limit: ' +
                targetPerLoadBalancerLimit);
        }
        // check for target group limit in load balancer
        const targetGroupsPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetGroupCount + 1) > targetGroupsPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-application-load-balancer" reached. Limit: ' +
                targetGroupsPerLoadBalancerLimit);
        }
        let listenerParent = null;
        let targetGroupParent = null;
        // try registering to each listener.
        for (const [listener, listenerMeta] of this.listenerMap.entries()) {
            try {
                const { componentsAdded, targetGroup } = listenerMeta.registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps);
                statsDelta.add(componentsAdded);
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        /* istanbul ignore next */
        if (!listenerParent) {
            // If this section is reached, no listener was found which could accommodate fleet
            // create new listener and register
            const listenersPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('listeners-per-application-load-balancer', LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
            if ((this.loadBalancerComponentCount.listenerCount + 1) > listenersPerLoadBalancerLimit) {
                throw new AWSLimitExhaustedError('AWS service limit "listeners-per-application-load-balancer" reached. Limit: ' +
                    listenersPerLoadBalancerLimit);
            }
            listenerParent = this.createListener(fleet.targetScope, loadBalancer);
            const listenerManager = new ListenerManager();
            this.listenerMap.set(listenerParent, listenerManager);
            statsDelta.add(new LoadBalancerComponentStats(1, 0, 0));
            try {
                const { componentsAdded, targetGroup } = listenerManager.registerWorkerFleet(loadBalancer, listenerParent, fleet, healthCheckConfig, healthMonitorProps);
                targetGroupParent = targetGroup;
                statsDelta.add(componentsAdded);
            }
            catch (e) {
                throw e;
            }
        }
        // update the current load balancer's stats
        this.loadBalancerComponentCount.add(statsDelta);
        return {
            componentsAdded: statsDelta,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new listener in the fleet's scope and
     * registers it to the given load balancer.
     *
     * @param scope
     * @param loadBalancer
     */
    createListener(scope, loadBalancer) {
        return new aws_elasticloadbalancingv2_1.ApplicationListener(scope, 'Listener', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT + this.listenerMap.size,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            loadBalancer,
            open: false,
        });
    }
}
/**
 * This class manages the properties of a single listener and all the components
 * under its hierarchy.
 * It is also responsible to create a new target group and register the given fleet.
 */
class ListenerManager {
    constructor() {
        this.targetMap = new Map();
        this.listenerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the target group
     * to which the fleet was registered.
     *
     * @param loadBalancer
     * @param listener
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps) {
        const componentsAdded = new LoadBalancerComponentStats();
        // Do all listener level service limit checks
        // check for target limit in listener
        const targetGroupPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-action-on-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.listenerComponentCount.targetGroupCount + 1) > targetGroupPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-action-on-application-load-balancer" reached. Limit: ' +
                targetGroupPerLoadBalancerLimit);
        }
        // latest version of CDK does not support 'forwardConfig' in listener rule yet. This means
        // we cannot add multiple target groups to a single listener. Adding this check till this
        // feature is supported.
        if (this.listenerComponentCount.targetGroupCount > 0) {
            throw new AWSLimitExhaustedError('Unable to add more than 1 Target Group to Listener.');
        }
        // Create a new target group
        const targetGroup = this.createTargetGroup(fleet.targetScope, loadBalancer, listener, fleet, healthCheckConfig);
        this.targetMap.set(targetGroup, fleet);
        // update the listener stats
        componentsAdded.targetGroupCount++;
        componentsAdded.targetCount += fleet.targetCapacity;
        // update the current listener's stats
        this.listenerComponentCount.add(componentsAdded);
        return {
            componentsAdded,
            targetGroup,
        };
    }
    /**
     * Following method creates a new new target group in the fleet's scope and
     * registers it to the given listener.
     *
     * @param scope
     * @param loadBalancer
     * @param listener
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    createTargetGroup(scope, loadBalancer, listener, monitorableFleet, healthCheckConfig) {
        const targetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(scope, 'TargetGroup', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            targets: [monitorableFleet.targetToMonitor],
            healthCheck: {
                port: healthCheckConfig.port ? healthCheckConfig.port.toString() : health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT.toString(),
                interval: healthCheckConfig.interval || health_monitor_1.HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL,
                healthyThresholdCount: healthCheckConfig.instanceHealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD,
                unhealthyThresholdCount: healthCheckConfig.instanceUnhealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD,
                protocol: aws_elasticloadbalancingv2_1.Protocol.HTTP,
            },
            vpc: loadBalancer.vpc,
        });
        listener.addTargetGroups('TargetGroup', {
            targetGroups: [targetGroup],
        });
        return targetGroup;
    }
}
/**
 * This class contains the statistics of all the nested load balancer
 * components like listener count, target group count and target count.
 * This statistics object will be associated with each load balancer
 * and listener for tracking the count of components.
 */
class LoadBalancerComponentStats {
    constructor(listenerCount = 0, targetGroupCount = 0, targetCount = 0) {
        this.listenerCount = listenerCount;
        this.targetGroupCount = targetGroupCount;
        this.targetCount = targetCount;
    }
    add(operand) {
        this.listenerCount += operand.listenerCount;
        this.targetGroupCount += operand.targetGroupCount;
        this.targetCount += operand.targetCount;
    }
}
class AWSLimitExhaustedError extends Error {
    constructor(message) {
        super(message);
    }
}
exports.AWSLimitExhaustedError = AWSLimitExhaustedError;
//# sourceMappingURL=data:application/json;base64,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