/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { IVpc } from 'aws-cdk-lib/aws-ec2';
import { ApplicationListener, ApplicationLoadBalancer, ApplicationTargetGroup } from 'aws-cdk-lib/aws-elasticloadbalancingv2';
import { Construct } from 'constructs';
import { HealthCheckConfig, IMonitorableFleet, Limit, HealthMonitorProps } from './health-monitor';
/**
 * This class is responsible for managing the statistics for all the
 * load balancers created in this construct. It is also responsible to search
 * for the finding the first Load balancer/Listener which can accomodate the
 * worker-fleet based on its size.
 *
 * A typical load balancer hierarchy looks like following:
 *  |__ Load Balancer 1
 *  |         |____________Listener 1
 *  |         |                 |_______Target Group 1 ------- Target/Fleet
 *  |         |                 |_______Target Group 2 ------- Target/Fleet
 *  |         |
 *  |         |____________Listener 2
 *  |                           |_______Target Group 1 ------- Target/Fleet
 *  |                           |_______Target Group 2 ------- Target/Fleet
 *  |
 *  |__ Load Balancer 2
 *            |____________Listener 1
 *            |                 |_______Target Group 1 ------- Target/Fleet
 *            |                 |_______Target Group 2 ------- Target/Fleet
 *            |
 *            |____________Listener 2
 *                              |_______Target Group 1 ------- Target/Fleet
 *                              |_______Target Group 2 ------- Target/Fleet
 *
 *  Components:
 *  1. LoadBalancerFactory: This is the root node of the tree. It contains the
 *     map of load balancer to its managers. It is responsible for creating a
 *     new load balancer if required. It delegates the registerFleet calls to
 *     downstream and returns parent load balancer, listener and target group
 *     of the registered fleet if the registration was successful
 *
 *  2. LoadBalancerManager: This class manages a single load balancer. It
 *     contains a map of all the listeners->manager. It also contains the component
 *     counts like listener, target group and target count. It delegates the
 *     registration call to downstream listeners and updates the stats when
 *     the registration is successful. It returns the parent listener and
 *     target group on successful registration.
 *
 *  3. ListenerManager: This class managers a single Listener. It contains a map
 *     of all of its target groups to its associated fleet. It also contains the
 *     component counts. It returns the target group on registration.
 */
export declare class LoadBalancerFactory {
    static readonly DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER = 50;
    static readonly DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER = 1000;
    static readonly DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER = 5;
    static readonly DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER = 100;
    static getAccountLimit(limitName: string, defaultValue: number, elbAccountLimits?: Limit[]): number;
    private readonly vpc;
    private readonly healthMonitorScope;
    private loadBalancerMap;
    constructor(healthMonitorScope: Construct, vpc: IVpc);
    /**
     * This method scans all the load balancers and its listeners and registers the fleet
     * to the load balancer and/or listener which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the load balancer, listener
     * and target group to which the fleet was registered.
     *
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(fleet: IMonitorableFleet, healthCheckConfig: HealthCheckConfig, healthMonitorProps: HealthMonitorProps): {
        loadBalancer: ApplicationLoadBalancer;
        listener: ApplicationListener;
        targetGroup: ApplicationTargetGroup;
    };
    /**
     * Following method creates a new load balancer within the given scope.
     *
     * @param scope
     * @param loadBalancerindex
     */
    private createLoadBalancer;
}
export declare class AWSLimitExhaustedError extends Error {
    constructor(message: string);
}
