"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectableApplicationEndpoint = exports.ApplicationEndpoint = exports.Endpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
/**
 * Connection endpoint
 *
 * Consists of a combination of hostname, port, and transport protocol.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint
     */
    constructor(props) {
        const { address, port } = props;
        const protocol = props.protocol ?? aws_ec2_1.Protocol.TCP;
        if (!aws_cdk_lib_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.portNumber = port;
        this.protocol = protocol;
        this.port = new aws_ec2_1.Port({
            protocol: this.protocol,
            fromPort: port,
            toPort: port,
            stringRepresentation: this.renderPort(port),
        });
        this.socketAddress = `${address}:${this.portAsString()}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (aws_cdk_lib_1.Token.isUnresolved(this.portNumber)) {
            return aws_cdk_lib_1.Token.asString(this.portNumber);
        }
        else {
            return this.portNumber.toString();
        }
    }
    renderPort(port) {
        return aws_cdk_lib_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "aws-rfdk.Endpoint", version: "1.1.0" };
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
/**
 * An endpoint serving http or https for an application.
 */
class ApplicationEndpoint extends Endpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        super({
            address: props.address,
            port: props.port,
            protocol: aws_ec2_1.Protocol.TCP,
        });
        this.applicationProtocol = props.protocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
    }
}
exports.ApplicationEndpoint = ApplicationEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ApplicationEndpoint[_b] = { fqn: "aws-rfdk.ApplicationEndpoint", version: "1.1.0" };
/**
 * An endpoint serving http or https for an application.
 */
class ConnectableApplicationEndpoint extends ApplicationEndpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        super(props);
        this.connections = props.connections;
    }
}
exports.ConnectableApplicationEndpoint = ConnectableApplicationEndpoint;
_c = JSII_RTTI_SYMBOL_1;
ConnectableApplicationEndpoint[_c] = { fqn: "aws-rfdk.ConnectableApplicationEndpoint", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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