"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubminuteStateMachine = exports.IteratorLambda = exports.LambdaSubminute = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_nodejs_1 = require("@aws-cdk/aws-lambda-nodejs");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class LambdaSubminute extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(parent, name, props) {
        var _d, _e, _f;
        super(parent, name);
        const iterator = new IteratorLambda(this, 'IteratorLambda', { targetFunction: props.targetFunction });
        this.iteratorFunction = iterator.function;
        const subminuteStateMachine = new SubminuteStateMachine(this, 'SubminuteStateMachine', {
            stateMachineName: 'lambda-subminute-statemachine',
            targetFunction: props.targetFunction,
            iteratorFunction: this.iteratorFunction,
            intervalTime: (_d = props.intervalTime) !== null && _d !== void 0 ? _d : 10,
            frequency: (_e = props.frequency) !== null && _e !== void 0 ? _e : 6,
        });
        this.stateMachineArn = subminuteStateMachine.stateMachine.stateMachineArn;
        const startRule = new events.Rule(this, 'StartSubminuteStateMachine', {
            schedule: events.Schedule.expression((_f = props.cronjobExpression) !== null && _f !== void 0 ? _f : 'cron(50/1 15-17 ? * * *)'),
            ruleName: 'subminute-statemachine-lambda-rule',
            description: `A rule to run the subminute state machine, i.e. ${subminuteStateMachine.stateMachine.stateMachineName}`,
        });
        startRule.addTarget(new targets.SfnStateMachine(subminuteStateMachine.stateMachine, {
            input: events.RuleTargetInput.fromObject({
                iterator: {
                    index: 0,
                    count: 6,
                },
            }),
        }));
    }
}
exports.LambdaSubminute = LambdaSubminute;
_a = JSII_RTTI_SYMBOL_1;
LambdaSubminute[_a] = { fqn: "cdk-lambda-subminute.LambdaSubminute", version: "0.1.19" };
/**
 * @stability stable
 */
class IteratorLambda extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, name, props) {
        super(scope, name);
        const iteratorLambdaRole = new iam.Role(this, 'IteratorLambdaRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('lambda.amazonaws.com')),
            description: 'An execution role for a Lambda function to invoke a target Lambda Function per time unit less than one minute',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSXRayDaemonWriteAccess'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'),
            ],
            roleName: 'Lambda-Iterator-Role',
        });
        cdk.DockerVolumeConsistency.CONSISTENT;
        iteratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            sid: 'TargetLambdaPermission',
            effect: iam.Effect.ALLOW,
            actions: ['lambda:InvokeFunction'],
            resources: [props.targetFunction.functionArn],
        }));
        this.function = new aws_lambda_nodejs_1.NodejsFunction(this, 'Iterator', {
            functionName: 'lambda-subminute-iterator',
            description: 'A function for breaking the limit of 1 minute with the CloudWatch Rules.',
            logRetention: aws_logs_1.RetentionDays.THREE_MONTHS,
            runtime: aws_lambda_1.Runtime.NODEJS_14_X,
            entry: fs.existsSync(path.join(__dirname, 'resources/iterator/iterator_agent.ts')) ? path.join(__dirname, 'resources/iterator/iterator_agent.ts') : path.join(__dirname, 'resources/iterator/iterator_agent.js'),
            handler: 'lambdaHandler',
            environment: {
                TARGET_FN_NAME: props.targetFunction.functionName,
            },
            memorySize: 128,
            role: iteratorLambdaRole,
            timeout: cdk.Duration.seconds(58),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
    }
}
exports.IteratorLambda = IteratorLambda;
_b = JSII_RTTI_SYMBOL_1;
IteratorLambda[_b] = { fqn: "cdk-lambda-subminute.IteratorLambda", version: "0.1.19" };
/**
 * @stability stable
 */
class SubminuteStateMachine extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Creates a state machine for breaking the limit of 1 minute with the CloudWatch Rules.
         *
         * @param iteratorFunction The iterator Lambda function for the target Labmda funciton.
         * @param intervalTime Seconds for an interval, the product of `frequency` and `intervalTime` should be approximagely 1 minute.
         * @param frequency How many times you intent to execute in a minute.
         * @returns THe job definition for the state machine.
         */
        this.createJobDefinition = (iteratorFunction, intervalTime, frequency) => {
            const configureCount = new sfn.Pass(this, 'ConfigureCount', {
                result: sfn.Result.fromObject({
                    index: 0,
                    count: frequency,
                }),
                resultPath: '$.iterator',
            });
            const iterator = new tasks.LambdaInvoke(this, 'Iterator', {
                lambdaFunction: iteratorFunction,
                resultPath: '$.iterator',
                resultSelector: {
                    'index.$': '$.Payload.index',
                    'count.$': '$.Payload.count',
                    'continue.$': '$.Payload.continue',
                },
            });
            const wait = new sfn.Wait(this, 'Wait for the target Lambda function finished', {
                time: sfn.WaitTime.duration(cdk.Duration.seconds(intervalTime)),
            });
            wait.next(iterator);
            const done = new sfn.Pass(this, 'Done');
            const isCountReached = new sfn.Choice(this, 'IsCountReached');
            isCountReached.when(sfn.Condition.booleanEquals('$.iterator.continue', true), wait);
            isCountReached.otherwise(done);
            const jobDefinition = configureCount.next(iterator).next(isCountReached);
            return jobDefinition;
        };
        /**
         * Creates a role and corresponding policies for the subminute state machine.
         *
         * @param targetFunctionArn the ARN of the Lambda function that executes your intention.
         * @param iteratorFunctionArn the ARN of the iterator Lambda function for the target Lambda function.
         * @returns the role as the documentation indicates.
         */
        this._createWorkFlowRole = (targetFunctionArn, iteratorFunctionArn) => {
            const workFlowExecutionRole = new iam.Role(this, 'StepFunctionExecutionRole', {
                assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
                description: 'Execute a workflow related to executing a Lambda function per time unit less than 1 minute.',
            });
            workFlowExecutionRole.addToPolicy(new iam.PolicyStatement({
                sid: 'LambdaInvokePermissions',
                effect: iam.Effect.ALLOW,
                actions: ['lambda:InvokeFunction'],
                resources: [
                    targetFunctionArn,
                    iteratorFunctionArn,
                ],
            }));
            return workFlowExecutionRole;
        };
        const stateMachineRole = this._createWorkFlowRole(props.targetFunction.functionArn, props.iteratorFunction.functionArn);
        const jobDefinition = this.createJobDefinition(props.iteratorFunction, props.intervalTime, props.frequency);
        const stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            stateMachineName: props.stateMachineName,
            definition: jobDefinition,
            role: stateMachineRole,
        });
        this.stateMachine = stateMachine;
    }
}
exports.SubminuteStateMachine = SubminuteStateMachine;
_c = JSII_RTTI_SYMBOL_1;
SubminuteStateMachine[_c] = { fqn: "cdk-lambda-subminute.SubminuteStateMachine", version: "0.1.19" };
//# sourceMappingURL=data:application/json;base64,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