"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const index_1 = require("./index");
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const mockApp = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(mockApp, 'testing-stack', { env });
/**
 * Example create an Secret for testing.
 */
const secret = new aws_secretsmanager_1.CfnSecret(stack, 'Mysecret', {
    secretString: JSON.stringify({
        KEY1: 'VALUE1',
        KEY2: 'VALUE2',
        KEY3: 'VALUE3',
    }),
});
const layer = new index_1.SecretManagerWrapperLayer(stack, 'SecretManagerWrapperLayer');
const lambda = new aws_lambda_1.Function(stack, 'fn', {
    runtime: aws_lambda_1.Runtime.PYTHON_3_9,
    code: aws_lambda_1.Code.fromInline(`
import os
def hander(events, contexts):
    env = {}
    env['KEY1'] = os.environ.get('KEY1', 'Not Found')
    env['KEY2'] = os.environ.get('KEY2', 'Not Found')
    env['KEY3'] = os.environ.get('KEY3', 'Not Found')
    return env
    `),
    handler: 'index.hander',
    layers: [layer],
    timeout: aws_cdk_lib_1.Duration.minutes(1),
    /**
     * you need to define this 4 environment various.
     */
    environment: {
        AWS_LAMBDA_EXEC_WRAPPER: '/opt/get-secrets-layer',
        SECRET_REGION: stack.region,
        SECRET_ARN: secret.ref,
        API_TIMEOUT: '5000',
    },
});
/**
 * Add Permission for lambda get secret value from secret manager.
 */
lambda.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
    effect: aws_iam_1.Effect.ALLOW,
    actions: ['secretsmanager:GetSecretValue'],
    // Also you can use find from context.
    resources: [secret.ref],
}));
/**
 * For Testing.
 */
const FnUrl = lambda.addFunctionUrl({
    authType: aws_lambda_1.FunctionUrlAuthType.NONE,
});
new aws_cdk_lib_1.CfnOutput(stack, 'FnUrl', {
    value: FnUrl.url,
});
//# sourceMappingURL=data:application/json;base64,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