from jinja2 import Markup
from slugify import slugify
from markdown2 import markdown

from pathlib import Path
import more_itertools
import re
import typing


def _parse_content(content: str, matcher: str):
    """
    split content into attributes and content text

    Parameters:
        content:
            The content to be parsed
        matcher: 
            A compiled regular expression that splits the content.
    """

    matchmaker = re.compile(matcher, flags=re.M)
    split_content = content.split("\n\n", maxsplit=1)
    attr_section = split_content[0]

    if len(split_content) == 2:
        base_content = split_content[1]

    else:
        base_content = ""

    parsed_attrs = re.split(matchmaker, attr_section)
    content = base_content.strip()

    attrs = list(filter(lambda x: x.strip(), parsed_attrs))
    return attrs, content


class Page:
    """Base component used to make web pages.

    All components that represent content in HTML, XML, or user-facing JSON generated by
    Render Engine should be a Page object.

    Pages can be rendered directly from a template or generated from a file.

    Page objects can be used to extend existing page objects.

    .. note::
        Not all attributes are defined by default (those that are marked *optional*) but will be checked for in other areas of the code.

    Attributes:

        title (str): **[Optional]** title of the :class:`Page <.Page>` object
        engine (str): **[Optional]** 
            The engine the :class:`Site <.site.Site>` should use to generate markup. By default this is `Jinja2 <https://palletsprojects.com/p/jinja/>`_.
        template (str): **[Optional]** template filename for the :attr:`engine <.site.Site.engine>` to look for.
            If defined, it must be a valid file.

        content_path (str): **[Optional]** The filepath to load content from.

            The `content_path` will be checked for additional attributes and :attr:`base_content <Page.base_content>`.

            Raises:
                FileNotFoundError: 
                    If a `content_path` was supplied path that points to a path that does not exist.


            For more information about content paths, markdown and content_path rendering see ``:ref:<page_from_content_path.rst>``
        base_content (str):  **[Optional]** Pre-rendered markdown or HTML to be converted to Markup.

            Uses the `Markdown2 <https://pypi.org/project/markdown2/>`_ generator.

            If ``content_path`` is provided the content section of the file will be stored as the `base_content`
        list_attrs (typing.Optional[typing.List[str]]): list of attributes whose string values should be converted
            to a comma-delimited list

            This is primarily for converting a comma-separated list of values provided in your ``content_path``

            Example::

                >>> class myPage(Page):
                    list_attrs = ["foo"]

                # calling myPage will convert the string value of `foo` to a list of values

                >>> myPage(foo='my, values, here')
                >>> myPage.foo
                ["my", "values", "here"]

        slug (str): **[Optional]** The rendered pages filename

            A slug passed in will be `slugified <https://github.com/un33k/python-slugify>`_
            to ensure a valid path is given. ``eg. slug="my cool slug" == "my-cool-slug"``

            If no slug is provided the class name will be used as the slug.::

                class MyPage(Page):
                    pass

                # the path for this page would be https://example.com/mypage

    """

    routes: typing.List[str] = [""]
    """the directory in the :attr:`output_path <.site.Site.output_path>` that the :class:`Page <.page.Page>` should be created at.

            Example a route of "pages" for a :py:class `render_engine.Page`: object with the :py:attr `Page.slug`: "foo" would generate *https://example.com/**pages**/foo*.

            An empty string will apply the route at the root `https://example.com/foo`
    """

    matcher: str = r"(^\w+: \b.+$)"  # expression to find attrs/vals
    """Regular expression string to match key/value pairs at the beginning of a content_path. 

    By default ``content_path`` markdown is set to detect markdown metadata format (key: value) with each key/value pair on it's own line.

    *In most cases this should not be changed*
    """

    markdown_extras: typing.List[str] = ["fenced-code-blocks", "footnotes"]
    """Plugins to be included when generating HTML from your ``base_content``. Defaults to ``["fenced-code-blocks", "footnotes"]``

    For more information on available extras or creating your own, see the `Markdown2 <https://pypi.org/project/markdown2/>`_ documentation 
    """
    always_refresh: bool = False # Ignore cache and always regenerate

    def __init__(
        self,
    ):

        if hasattr(self, 'content_path'):
            content = Path(self.content_path).read_text()

            valid_attrs, self.base_content = _parse_content(
                content,
                matcher=self.matcher,
            )

            for attr in valid_attrs:
                name, value = attr.split(": ", maxsplit=1)

                # comma delimit attributes.
                if name.lower() in getattr(self, 'list_attrs', []):
                    value = [attrval.lower() for attrval in value.split(", ")]

                else:
                    value = value.strip()

                setattr(self, name.lower(), value)

        if not hasattr(self, 'title'):
            self.title = self.__class__.__name__

        if not hasattr(self, 'slug'):
            self.slug = self.title or self.__class__.__name__

        self.slug = slugify(self.slug)

    @property
    def url(self) -> str:
        """The first route and the slug of the page."""
        return f"{self.routes[0]}/{self.slug}"

    @classmethod
    def from_content_path(cls, filepath, **kwargs):
        class NewPage(cls):
            content_path = filepath

            def __init__(self, markdown_extras: typing.List[str] = []):
                for extra in markdown_extras:
                    if extra not in self.markdown_extras:
                        self.markdown_extras.append(extra)

                super().__init__()

        newpage = NewPage(**kwargs)

        return newpage

    @property
    def html(self) -> str:
        """Text from self.content converted to html"""

        if content:=getattr(self, 'base_content', None):
            return markdown(content, extras=self.markdown_extras)

        else:
            return ""

    @property
    def markup(self) -> str:
        """html = rendered HTML (not marked up). Is `None` if `content == None`"""
        if self.html:
            return Markup(self.html)

        else:
            return ""

    @property
    def content(self) -> str:
        return self.markup
