"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const ecs = require("aws-cdk-lib/aws-ecs");
test('New service/new table, public API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const tableName = "custom-table-name";
    const familyName = "family-name";
    const construct = new lib_1.FargateToDynamoDB(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        dynamoTableProps: {
            tableName,
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
        },
        tablePermissions: 'ReadWrite'
    });
    expect(construct.vpc !== null);
    expect(construct.service !== null);
    expect(construct.container !== null);
    expect(construct.dynamoTable !== null);
    expect(construct.dynamoTableInterface !== null);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        TableName: tableName
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testconstructDynamoTable67BDAFC5",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".dynamodb"
                ]
            ]
        },
    });
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::DynamoDB::Table', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/new table, private API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const tableName = 'table-name';
    new lib_1.FargateToDynamoDB(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        dynamoTableProps: {
            tableName,
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
        },
        tablePermissions: 'Read',
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        TableName: tableName,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testconstructDynamoTable67BDAFC5",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".dynamodb"
                ]
            ]
        },
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::DynamoDB::Table', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/existing table, private API, existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const tableName = 'custom-table-name';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const existingTable = new dynamodb.Table(stack, 'MyTable', {
        tableName,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
    });
    const construct = new lib_1.FargateToDynamoDB(stack, 'test-construct', {
        publicApi,
        existingVpc,
        existingTableInterface: existingTable,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        tablePermissions: 'ALL'
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        TableName: tableName
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:*",
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "MyTable794EDED1",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".dynamodb"
                ]
            ]
        },
    });
    expect(construct.dynamoTable == null);
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toCountResources('AWS::DynamoDB::Table', 1);
});
test('Existing service/new table, public API, existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const serviceName = 'custom-name';
    const customName = 'CUSTOM_NAME';
    const customArn = 'CUSTOM_ARN';
    const tableName = 'table-name';
    const existingVpc = defaults.getTestVpc(stack);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    const construct = new lib_1.FargateToDynamoDB(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
        tableArnEnvironmentVariableName: customArn,
        tableEnvironmentVariableName: customName,
        dynamoTableProps: {
            tableName,
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customArn,
                        Value: {
                            "Fn::GetAtt": [
                                "testconstructDynamoTable67BDAFC5",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: customName,
                        Value: {
                            Ref: "testconstructDynamoTable67BDAFC5"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        TableName: tableName
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testconstructDynamoTable67BDAFC5",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".dynamodb"
                ]
            ]
        },
    });
    expect(construct.dynamoTable == null);
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toCountResources('AWS::DynamoDB::Table', 1);
});
test('Existing service/existing table, private API, existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const serviceName = 'custom-name';
    const tableName = 'custom-table-name';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    const existingTable = new dynamodb.Table(stack, 'MyTablet', {
        tableName,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    });
    const construct = new lib_1.FargateToDynamoDB(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
        tablePermissions: 'Write',
        existingTableInterface: existingTable
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: "DYNAMODB_TABLE_ARN",
                        Value: {
                            "Fn::GetAtt": [
                                "MyTabletD7ADAF4F",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: "DYNAMODB_TABLE_NAME",
                        Value: {
                            Ref: "MyTabletD7ADAF4F"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        TableName: tableName
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "MyTabletD7ADAF4F",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".dynamodb"
                ]
            ]
        },
    });
    expect(construct.dynamoTable == null);
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toCountResources('AWS::DynamoDB::Table', 1);
});
test('test error invalid table permission', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const serviceName = 'custom-name';
    const tableName = 'custom-table-name';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    const existingTable = new dynamodb.Table(stack, 'MyTablet', {
        tableName,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    });
    const app = () => {
        new lib_1.FargateToDynamoDB(stack, 'test-construct', {
            publicApi,
            existingFargateServiceObject: testService,
            existingContainerDefinitionObject: testContainer,
            existingVpc,
            tablePermissions: 'reed',
            existingTableInterface: existingTable
        });
    };
    expect(app).toThrowError('Invalid tablePermission submitted - REED');
});
//# sourceMappingURL=data:application/json;base64,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