# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fasta_checksum_utils']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=22.1.0,<23.0.0', 'pysam>=0.20.0,<0.21.0']

entry_points = \
{'console_scripts': ['fasta-checksum-utils = fasta_checksum_utils.entry:entry']}

setup_kwargs = {
    'name': 'fasta-checksum-utils',
    'version': '0.1.0',
    'description': 'Library and command-line utility for checksumming FASTA files and individual contigs.',
    'long_description': '# fasta-checksum-utils\n\nAsynchronous library and command-line utility for checksumming FASTA files and individual contigs.\n\n\n## Installation\n\nTo install `fasta-checksum-utils`, run the following `pip` command:\n\n```bash\npip install fasta-checksum-utils\n```\n\n\n## CLI Usage\n\nTo generate a text report of checksums in the FASTA document, run the following command:\n\n```bash\nfasta-checksum-utils ./my-fasta.fa[.gz]\n```\n\nThis will print output in the following tab-delimited format:\n\n```\nfile  [file size in bytes]    MD5 [file MD5 hash]           TRUNC512  [file TRUNC512 hash]\nchr1  [chr1 sequence length]  MD5 [chr1 sequence MD5 hash]  TRUNC512  [chr1 sequence TRUNC512 hash]\nchr2  [chr2 sequence length]  MD5 [chr2 sequence MD5 hash]  TRUNC512  [chr2 sequence TRUNC512 hash]\n...\n```\n\nThe following example is the output generated by specifying the SARS-CoV-2 genome FASTA from NCBI:\n\n```\nfile        30429  MD5  863ee5dba1da0ca3f87783782284d489  TRUNC512  3036e94352072c8cd4b5d2e855a72af3d4ca010f6fac1353\nNC_045512.2 29903  MD5  105c82802b67521950854a851fc6eefd  TRUNC512  4b2195260fd845e771bec8e9a8d754832caac7b9547eefc3\n```\n\nIf the `--out-format bento-json` arguments are passed, the tool will instead output the report in a JSON\nformat, designed to be compatible with the requirements of the \n[Bento Reference Service](https://github.com/bento-platform/bento_reference_service). The following example\nis the output generated by specifying the SARS-CoV-2 genome:\n\n```json\n{\n  "md5": "863ee5dba1da0ca3f87783782284d489",\n  "trunc512": "3036e94352072c8cd4b5d2e855a72af3d4ca010f6fac1353",\n  "fasta_size": 30429,\n  "contigs": [\n    {\n      "name": "NC_045512.2",\n      "md5": "105c82802b67521950854a851fc6eefd",\n      "trunc512": "4b2195260fd845e771bec8e9a8d754832caac7b9547eefc3",\n      "length": 29903\n    }\n  ]\n}\n```\n\n\n## Library Usage\n\nBelow are some examples of how `fasta-checksum-utils` can be used as an asynchronous Python library:\n\n```python\nimport asyncio\nimport fasta_checksum_utils as fc\nimport pysam\nfrom pathlib import Path\n\n\nasync def demo():\n    covid_genome: Path = Path("./sars_cov_2.fa")\n    \n    # calculate an MD5 checksum for a whole file\n    file_checksum: str = await fc.algorithms.AlgorithmMD5.checksum_file(covid_genome)\n    print(file_checksum)\n    # prints "863ee5dba1da0ca3f87783782284d489"\n    \n    all_algorithms = (fc.algorithms.AlgorithmMD5, fc.algorithms.AlgorithmTRUNC512)\n    \n    # calculate multiple checksums for a whole file\n    all_checksums: tuple[str, ...] = await fc.checksum_file(file=covid_genome, algorithms=all_algorithms)\n    print(all_checksums)\n    # prints tuple: ("863ee5dba1da0ca3f87783782284d489", "3036e94352072c8cd4b5d2e855a72af3d4ca010f6fac1353")\n    \n    # calculate an MD5 and TRUNC512 checksum for a specific contig in a PySAM FASTA file:\n    fh = pysam.FastaFile(str(covid_genome))\n    try:\n        contig_checksums: tuple[str, ...] = await fc.checksum_contig(\n            fh=fh, \n            contig_name="NC_045512.2", \n            algorithms=all_algorithms,\n        )\n        print(contig_checksums)\n        # prints tuple: ("105c82802b67521950854a851fc6eefd", "4b2195260fd845e771bec8e9a8d754832caac7b9547eefc3")\n    finally:\n        fh.close()  # always close the file handle\n\n\nasyncio.run(demo())\n```\n',
    'author': 'David Lougheed',
    'author_email': 'david.lougheed@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
