"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _e;
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
        this.combineImportAndExportRoles = props.combineImportAndExportRoles;
    }
    bindToCluster(scope, options) {
        var _e;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            // versions which combine the import and export Roles (right now, this is only 8.0)
            // require a different parameter name (identical for both import and export)
            const s3ImportParam = this.combineImportAndExportRoles
                ? 'aws_default_s3_role'
                : 'aurora_load_from_s3_role';
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter(s3ImportParam, options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            const s3ExportParam = this.combineImportAndExportRoles
                ? 'aws_default_s3_role'
                : 'aurora_select_into_s3_role';
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter(s3ExportParam, options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
_a = JSII_RTTI_SYMBOL_1;
AuroraEngineVersion[_a] = { fqn: "@aws-cdk/aws-rds.AuroraEngineVersion", version: "1.157.0" };
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7', combineImportAndExportRoles) {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
        this._combineImportAndExportRoles = combineImportAndExportRoles;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
    static builtIn_8_0(minorVersion) {
        // 8.0 of the MySQL engine needs to combine the import and export Roles
        return new AuroraMysqlEngineVersion(`8.0.mysql_aurora.${minorVersion}`, '8.0', true);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
_b = JSII_RTTI_SYMBOL_1;
AuroraMysqlEngineVersion[_b] = { fqn: "@aws-cdk/aws-rds.AuroraMysqlEngineVersion", version: "1.157.0" };
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/** Version "5.7.mysql_aurora.2.08.2". */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/** Version "5.7.mysql_aurora.2.09.0". */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
/** Version "5.7.mysql_aurora.2.09.1". */
AuroraMysqlEngineVersion.VER_2_09_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.1');
/** Version "5.7.mysql_aurora.2.09.2". */
AuroraMysqlEngineVersion.VER_2_09_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.2');
/** Version "5.7.mysql_aurora.2.09.3". */
AuroraMysqlEngineVersion.VER_2_09_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.3');
/** Version "5.7.mysql_aurora.2.10.0". */
AuroraMysqlEngineVersion.VER_2_10_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.0');
/** Version "5.7.mysql_aurora.2.10.1". */
AuroraMysqlEngineVersion.VER_2_10_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.1');
/** Version "5.7.mysql_aurora.2.10.2". */
AuroraMysqlEngineVersion.VER_2_10_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.2');
/** Version "8.0.mysql_aurora.3.01.0". */
AuroraMysqlEngineVersion.VER_3_01_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.01.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
            combineImportAndExportRoles: version === null || version === void 0 ? void 0 : version._combineImportAndExportRoles,
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraPostgresEngineFeatures(auroraPostgresFeatures);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
_c = JSII_RTTI_SYMBOL_1;
AuroraPostgresEngineVersion[_c] = { fqn: "@aws-cdk/aws-rds.AuroraPostgresEngineVersion", version: "1.157.0" };
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
AuroraPostgresEngineVersion.VER_9_6_18 = AuroraPostgresEngineVersion.of('9.6.18', '9.6');
/** Version "9.6.19". */
AuroraPostgresEngineVersion.VER_9_6_19 = AuroraPostgresEngineVersion.of('9.6.19', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/** Version "10.13". */
AuroraPostgresEngineVersion.VER_10_13 = AuroraPostgresEngineVersion.of('10.13', '10', { s3Import: true, s3Export: true });
/** Version "10.14". */
AuroraPostgresEngineVersion.VER_10_14 = AuroraPostgresEngineVersion.of('10.14', '10', { s3Import: true, s3Export: true });
/** Version "10.16". */
AuroraPostgresEngineVersion.VER_10_16 = AuroraPostgresEngineVersion.of('10.16', '10', { s3Import: true, s3Export: true });
/** Version "10.18". */
AuroraPostgresEngineVersion.VER_10_18 = AuroraPostgresEngineVersion.of('10.18', '10', { s3Import: true, s3Export: true });
/** Version "10.19". */
AuroraPostgresEngineVersion.VER_10_19 = AuroraPostgresEngineVersion.of('10.19', '10', { s3Import: true, s3Export: true });
/** Version "10.20". */
AuroraPostgresEngineVersion.VER_10_20 = AuroraPostgresEngineVersion.of('10.20', '10', { s3Import: true, s3Export: true });
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/** Version "11.8". */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
/** Version "11.9". */
AuroraPostgresEngineVersion.VER_11_9 = AuroraPostgresEngineVersion.of('11.9', '11', { s3Import: true, s3Export: true });
/** Version "11.11". */
AuroraPostgresEngineVersion.VER_11_11 = AuroraPostgresEngineVersion.of('11.11', '11', { s3Import: true, s3Export: true });
/** Version "11.13". */
AuroraPostgresEngineVersion.VER_11_13 = AuroraPostgresEngineVersion.of('11.13', '11', { s3Import: true, s3Export: true });
/** Version "11.14". */
AuroraPostgresEngineVersion.VER_11_14 = AuroraPostgresEngineVersion.of('11.14', '11', { s3Import: true, s3Export: true });
/** Version "11.15". */
AuroraPostgresEngineVersion.VER_11_15 = AuroraPostgresEngineVersion.of('11.15', '11', { s3Import: true, s3Export: true });
/** Version "12.4". */
AuroraPostgresEngineVersion.VER_12_4 = AuroraPostgresEngineVersion.of('12.4', '12', { s3Import: true, s3Export: true });
/** Version "12.6". */
AuroraPostgresEngineVersion.VER_12_6 = AuroraPostgresEngineVersion.of('12.6', '12', { s3Import: true, s3Export: true });
/** Version "12.8". */
AuroraPostgresEngineVersion.VER_12_8 = AuroraPostgresEngineVersion.of('12.8', '12', { s3Import: true, s3Export: true });
/** Version "12.8". */
AuroraPostgresEngineVersion.VER_12_9 = AuroraPostgresEngineVersion.of('12.9', '12', { s3Import: true, s3Export: true });
/** Version "12.10". */
AuroraPostgresEngineVersion.VER_12_10 = AuroraPostgresEngineVersion.of('12.10', '12', { s3Import: true, s3Export: true });
/** Version "13.3". */
AuroraPostgresEngineVersion.VER_13_3 = AuroraPostgresEngineVersion.of('13.3', '13', { s3Import: true, s3Export: true });
/** Version "13.4". */
AuroraPostgresEngineVersion.VER_13_4 = AuroraPostgresEngineVersion.of('13.4', '13', { s3Import: true, s3Export: true });
/** Version "13.5". */
AuroraPostgresEngineVersion.VER_13_5 = AuroraPostgresEngineVersion.of('13.5', '13', { s3Import: true, s3Export: true });
/** Version "13.6". */
AuroraPostgresEngineVersion.VER_13_6 = AuroraPostgresEngineVersion.of('13.6', '13', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _e, _f, _g;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_e = this.engineVersion) === null || _e === void 0 ? void 0 : _e.fullVersion) {
            if (options.s3ImportRole && !((_f = config.features) === null || _f === void 0 ? void 0 : _f.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_g = config.features) === null || _g === void 0 ? void 0 : _g.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.aurora);
            }
            throw error;
        }
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraMysqlClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.auroraMysql);
            }
            throw error;
        }
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraPostgresClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.auroraPostgres);
            }
            throw error;
        }
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
_d = JSII_RTTI_SYMBOL_1;
DatabaseClusterEngine[_d] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterEngine", version: "1.157.0" };
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci1lbmdpbmUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbHVzdGVyLWVuZ2luZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFDQSw4REFBOEQ7QUFHOUQsdURBQW9FO0FBc0hwRSxNQUFlLGlCQUFpQjtJQVc5QixZQUFZLEtBQTZCO1FBQ3ZDLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLDZCQUE2QixHQUFHLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQztRQUN6RSxJQUFJLENBQUMsNEJBQTRCLEdBQUcsS0FBSyxDQUFDLDRCQUE0QixDQUFDO1FBQ3ZFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDekMsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7S0FDckg7SUFFTSxhQUFhLENBQUMsS0FBZ0IsRUFBRSxPQUFpQzs7UUFDdEUsTUFBTSxjQUFjLFNBQUcsT0FBTyxDQUFDLGNBQWMsbUNBQUksSUFBSSxDQUFDLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ25GLE9BQU87WUFDTCxjQUFjO1lBQ2QsSUFBSSxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQ3RCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtTQUN4QixDQUFDO0tBQ0g7Q0FRRjtBQVNELE1BQWUsc0JBQXVCLFNBQVEsaUJBQWlCO0lBSzdELFlBQVksS0FBa0M7UUFDNUMsS0FBSyxDQUFDO1lBQ0osR0FBRyxLQUFLO1lBQ1IsNkJBQTZCLEVBQUUsY0FBYyxDQUFDLHlCQUF5QixDQUFDLDBCQUEwQjtZQUNsRyw0QkFBNEIsRUFBRSxjQUFjLENBQUMseUJBQXlCLENBQUMseUJBQXlCO1lBQ2hHLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxFQUFFLFlBQVksRUFBRSxLQUFLLENBQUMsbUJBQW1CLEVBQUU7U0FDdkcsQ0FBQyxDQUFDO1FBVlcsaUJBQVksR0FBRyxPQUFPLENBQUM7UUFDdkIsc0JBQWlCLEdBQWEsQ0FBQyxPQUFPLEVBQUUsU0FBUyxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQVV2RixJQUFJLENBQUMsMkJBQTJCLEdBQUcsS0FBSyxDQUFDLDJCQUEyQixDQUFDO0tBQ3RFO0lBRU0sYUFBYSxDQUFDLEtBQWdCLEVBQUUsT0FBaUM7O1FBQ3RFLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ25ELE1BQU0sY0FBYyxTQUFHLE9BQU8sQ0FBQyxjQUFjLG1DQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksSUFBSSxPQUFPLENBQUMsWUFBWTtZQUM1RixDQUFDLENBQUMsSUFBSSxnQ0FBYyxDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRTtnQkFDbkQsTUFBTSxFQUFFLElBQUk7YUFDYixDQUFDO1lBQ0YsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUMzQixJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUU7WUFDeEIsbUZBQW1GO1lBQ25GLDRFQUE0RTtZQUM1RSxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsMkJBQTJCO2dCQUNwRCxDQUFDLENBQUMscUJBQXFCO2dCQUN2QixDQUFDLENBQUMsMEJBQTBCLENBQUM7WUFDL0IsY0FBYyxhQUFkLGNBQWMsdUJBQWQsY0FBYyxDQUFFLFlBQVksQ0FBQyxhQUFhLEVBQUUsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUU7U0FDM0U7UUFDRCxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUU7WUFDeEIsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLDJCQUEyQjtnQkFDcEQsQ0FBQyxDQUFDLHFCQUFxQjtnQkFDdkIsQ0FBQyxDQUFDLDRCQUE0QixDQUFDO1lBQ2pDLGNBQWMsYUFBZCxjQUFjLHVCQUFkLGNBQWMsQ0FBRSxZQUFZLENBQUMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1NBQzNFO1FBRUQsT0FBTztZQUNMLEdBQUcsTUFBTTtZQUNULGNBQWM7U0FDZixDQUFDO0tBQ0g7Q0FDRjtBQUVEOzs7R0FHRztBQUNILE1BQWEsbUJBQW1CO0lBbUQ5QixZQUFvQixpQkFBeUIsRUFBRSxxQkFBNkIsS0FBSztRQUMvRSxJQUFJLENBQUMsaUJBQWlCLEdBQUcsaUJBQWlCLENBQUM7UUFDM0MsSUFBSSxDQUFDLGtCQUFrQixHQUFHLGtCQUFrQixDQUFDO0tBQzlDO0lBeEJEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsRUFBRSxDQUFDLGlCQUF5QixFQUFFLGtCQUEyQjtRQUNyRSxPQUFPLElBQUksbUJBQW1CLENBQUMsaUJBQWlCLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztLQUN2RTtJQUVPLE1BQU0sQ0FBQyxXQUFXLENBQUMsWUFBb0IsRUFBRSxvQkFBNkIsSUFBSTtRQUNoRixPQUFPLElBQUksbUJBQW1CLENBQUMsT0FBTyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsWUFBWSxFQUFFLENBQUMsQ0FBQztLQUNsRzs7QUE1Q0gsa0RBdURDOzs7QUF0REMseUJBQXlCO0FBQ0YsMkJBQU8sR0FBRyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQy9FLHlDQUF5QztBQUNsQiw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUM7QUFDbEIsOEJBQVUsR0FBRyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDOUUseUNBQXlDO0FBQ2xCLDhCQUFVLEdBQUcsbUJBQW1CLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQzlFLHlDQUF5QztBQUNsQiw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUM7QUFDbEIsOEJBQVUsR0FBRyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDOUUseUNBQXlDO0FBQ2xCLDhCQUFVLEdBQUcsbUJBQW1CLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQzlFLHlDQUF5QztBQUNsQiw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUM7QUFDbEIsOEJBQVUsR0FBRyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDOUUseUNBQXlDO0FBQ2xCLDhCQUFVLEdBQUcsbUJBQW1CLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQzlFLHlDQUF5QztBQUNsQiw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUM7QUFDbEIsOEJBQVUsR0FBRyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDOUUsMkNBQTJDO0FBQ3BCLGdDQUFZLEdBQUcsbUJBQW1CLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0FBQ2xGLHlDQUF5QztBQUNsQiw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQXNDaEYsTUFBTSxtQkFBb0IsU0FBUSxzQkFBc0I7SUFDdEQsWUFBWSxPQUE2QjtRQUN2QyxLQUFLLENBQUM7WUFDSixVQUFVLEVBQUUsUUFBUTtZQUNwQixhQUFhLEVBQUUsT0FBTztnQkFDcEIsQ0FBQyxDQUFDO29CQUNBLFdBQVcsRUFBRSxPQUFPLENBQUMsaUJBQWlCO29CQUN0QyxZQUFZLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjtpQkFDekM7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDYixtQkFBbUIsRUFBRSxLQUFLO1NBQzNCLENBQUMsQ0FBQztLQUNKO0lBRVMscUJBQXFCLENBQUMsTUFBaUI7UUFDL0MsZ0VBQWdFO1FBQ2hFLHdDQUF3QztRQUN4QyxPQUFPLFNBQVMsQ0FBQztLQUNsQjtDQUNGO0FBRUQ7OztHQUdHO0FBQ0gsTUFBYSx3QkFBd0I7SUE0Rm5DLFlBQ0Usc0JBQThCLEVBQUUsMEJBQWtDLEtBQUssRUFDdkUsMkJBQXFDO1FBRXJDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxzQkFBc0IsQ0FBQztRQUNyRCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsdUJBQXVCLENBQUM7UUFDdkQsSUFBSSxDQUFDLDRCQUE0QixHQUFHLDJCQUEyQixDQUFDO0tBQ2pFO0lBdkNEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsRUFBRSxDQUFDLHNCQUE4QixFQUFFLHVCQUFnQztRQUMvRSxPQUFPLElBQUksd0JBQXdCLENBQUMsc0JBQXNCLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztLQUN0RjtJQUVPLE1BQU0sQ0FBQyxXQUFXLENBQUMsWUFBb0IsRUFBRSxvQkFBNkIsSUFBSTtRQUNoRixPQUFPLElBQUksd0JBQXdCLENBQUMsT0FBTyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsWUFBWSxFQUFFLENBQUMsQ0FBQztLQUN2RztJQUVPLE1BQU0sQ0FBQyxXQUFXLENBQUMsWUFBb0I7UUFDN0MsdUVBQXVFO1FBQ3ZFLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxvQkFBb0IsWUFBWSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO0tBQ3RGOztBQS9FSCw0REFvR0M7OztBQW5HQyx3QkFBd0I7QUFDRCxtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDdEYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ25GLHlDQUF5QztBQUNsQixtQ0FBVSxHQUFHLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNuRix5Q0FBeUM7QUFDbEIsbUNBQVUsR0FBRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDbkYseUNBQXlDO0FBQ2xCLG1DQUFVLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBcURyRixNQUFNLHdCQUF5QixTQUFRLHNCQUFzQjtJQUMzRCxZQUFZLE9BQWtDO1FBQzVDLEtBQUssQ0FBQztZQUNKLFVBQVUsRUFBRSxjQUFjO1lBQzFCLGFBQWEsRUFBRSxPQUFPO2dCQUNwQixDQUFDLENBQUM7b0JBQ0EsV0FBVyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0I7b0JBQzNDLFlBQVksRUFBRSxPQUFPLENBQUMsdUJBQXVCO2lCQUM5QztnQkFDRCxDQUFDLENBQUMsU0FBUztZQUNiLG1CQUFtQixFQUFFLEtBQUs7WUFDMUIsMkJBQTJCLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLDRCQUE0QjtTQUNuRSxDQUFDLENBQUM7S0FDSjtJQUVTLHFCQUFxQixDQUFDLEtBQWdCO1FBQzlDLE9BQU8sZ0NBQWMsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUUsdURBQXVELEVBQ3pHLFdBQVcsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUMsQ0FBQztLQUMzQztDQUNGO0FBcUJEOzs7R0FHRztBQUNILE1BQWEsMkJBQTJCO0lBdUd0QyxZQUFvQix5QkFBaUMsRUFBRSwwQkFBa0MsRUFBRSxzQkFBcUQ7UUFDOUksSUFBSSxDQUFDLHlCQUF5QixHQUFHLHlCQUF5QixDQUFDO1FBQzNELElBQUksQ0FBQywwQkFBMEIsR0FBRywwQkFBMEIsQ0FBQztRQUM3RCxJQUFJLENBQUMsU0FBUyxHQUFHO1lBQ2YsUUFBUSxFQUFFLENBQUEsc0JBQXNCLGFBQXRCLHNCQUFzQix1QkFBdEIsc0JBQXNCLENBQUUsUUFBUSxFQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbkUsUUFBUSxFQUFFLENBQUEsc0JBQXNCLGFBQXRCLHNCQUFzQix1QkFBdEIsc0JBQXNCLENBQUUsUUFBUSxFQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDcEUsQ0FBQztLQUNIO0lBaENEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsRUFBRSxDQUFDLHlCQUFpQyxFQUFFLDBCQUFrQyxFQUNwRixzQkFBcUQ7Ozs7Ozs7Ozs7UUFFckQsT0FBTyxJQUFJLDJCQUEyQixDQUFDLHlCQUF5QixFQUFFLDBCQUEwQixFQUFFLHNCQUFzQixDQUFDLENBQUM7S0FDdkg7O0FBMUZILGtFQStHQzs7O0FBOUdDLHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNsRix1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDbEYsd0JBQXdCO0FBQ0Qsc0NBQVUsR0FBRywyQkFBMkIsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ3BGLHdCQUF3QjtBQUNELHNDQUFVLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNwRix3QkFBd0I7QUFDRCxzQ0FBVSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDcEYsd0JBQXdCO0FBQ0Qsc0NBQVUsR0FBRywyQkFBMkIsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ3BGLHdCQUF3QjtBQUNELHNDQUFVLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNwRix3QkFBd0I7QUFDRCxzQ0FBVSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDcEYsc0JBQXNCO0FBQ0Msb0NBQVEsR0FBRywyQkFBMkIsQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQy9FLHNCQUFzQjtBQUNDLG9DQUFRLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztBQUMvRSxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDL0Usc0JBQXNCO0FBQ0Msb0NBQVEsR0FBRywyQkFBMkIsQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ25HLHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHVCQUF1QjtBQUNBLHFDQUFTLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3JILHNCQUFzQjtBQUNDLG9DQUFRLEdBQUcsMkJBQTJCLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuRyxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCx1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNySCx1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNySCx1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNySCx1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNySCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCx1QkFBdUI7QUFDQSxxQ0FBUyxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNySCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQUNuSCxzQkFBc0I7QUFDQyxvQ0FBUSxHQUFHLDJCQUEyQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztBQThDckgsTUFBTSwyQkFBNEIsU0FBUSxpQkFBaUI7SUFlekQsWUFBWSxPQUFxQztRQUMvQyxLQUFLLENBQUM7WUFDSixVQUFVLEVBQUUsbUJBQW1CO1lBQy9CLDZCQUE2QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQyw2QkFBNkI7WUFDckcsNEJBQTRCLEVBQUUsY0FBYyxDQUFDLHlCQUF5QixDQUFDLDRCQUE0QjtZQUNuRyxXQUFXLEVBQUUsSUFBSTtZQUNqQixhQUFhLEVBQUUsT0FBTztnQkFDcEIsQ0FBQyxDQUFDO29CQUNBLFdBQVcsRUFBRSxPQUFPLENBQUMseUJBQXlCO29CQUM5QyxZQUFZLEVBQUUsT0FBTyxDQUFDLDBCQUEwQjtpQkFDakQ7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDYixRQUFRLEVBQUUsT0FBTztnQkFDZixDQUFDLENBQUM7b0JBQ0EsUUFBUSxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQywyQkFBMkIsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUMsU0FBUztvQkFDckcsUUFBUSxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQywyQkFBMkIsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUMsU0FBUztpQkFDdEc7Z0JBQ0QsQ0FBQyxDQUFDO29CQUNBLFFBQVEsRUFBRSwyQkFBMkIsQ0FBQyxzQkFBc0I7b0JBQzVELFFBQVEsRUFBRSwyQkFBMkIsQ0FBQyxzQkFBc0I7aUJBQzdEO1NBQ0osQ0FBQyxDQUFDO1FBekJXLGlCQUFZLEdBQUcsWUFBWSxDQUFDO1FBQzVCLG9CQUFlLEdBQUcsVUFBVSxDQUFDO1FBQzdCLHNCQUFpQixHQUFhLENBQUMsWUFBWSxDQUFDLENBQUM7S0F3QjVEO0lBRU0sYUFBYSxDQUFDLEtBQWdCLEVBQUUsT0FBaUM7O1FBQ3RFLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ25ELGdIQUFnSDtRQUNoSCxVQUFJLElBQUksQ0FBQyxhQUFhLDBDQUFFLFdBQVcsRUFBRTtZQUNuQyxJQUFJLE9BQU8sQ0FBQyxZQUFZLElBQUksQ0FBQyxPQUFDLE1BQU0sQ0FBQyxRQUFRLDBDQUFFLFFBQVEsQ0FBQyxFQUFFO2dCQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcscURBQXFELENBQUMsQ0FBQzthQUN6SjtZQUNELElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxDQUFDLE9BQUMsTUFBTSxDQUFDLFFBQVEsMENBQUUsUUFBUSxDQUFDLEVBQUU7Z0JBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsbURBQW1ELElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxxREFBcUQsQ0FBQyxDQUFDO2FBQ3pKO1NBQ0Y7UUFDRCxPQUFPLE1BQU0sQ0FBQztLQUNmO0lBRVMscUJBQXFCLENBQUMsS0FBZ0I7UUFDOUMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtZQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLDZGQUE2RjtnQkFDM0csb0dBQW9HLENBQUMsQ0FBQztTQUN6RztRQUNELE9BQU8sZ0NBQWMsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUUsNERBQTRELEVBQzlHLFdBQVcsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUMsQ0FBQztLQUMzQzs7QUEzREQ7O0dBRUc7QUFDcUIsa0RBQXNCLEdBQUcsVUFBVSxDQUFDO0FBRTVEOztHQUVHO0FBQ3FCLGtEQUFzQixHQUFHLFVBQVUsQ0FBQztBQXNEOUQ7OztHQUdHO0FBQ0gsTUFBYSxxQkFBcUI7SUE0QmhDLDBEQUEwRDtJQUNuRCxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQStCOzs7Ozs7Ozs7O1FBQ2xELE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7S0FDL0M7SUFFRCwwREFBMEQ7SUFDbkQsTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFvQzs7Ozs7Ozs7OztRQUM1RCxPQUFPLElBQUksd0JBQXdCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQ3BEO0lBRUQsK0RBQStEO0lBQ3hELE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBdUM7Ozs7Ozs7Ozs7UUFDbEUsT0FBTyxJQUFJLDJCQUEyQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztLQUN2RDs7QUF6Q0gsc0RBMENDOzs7QUF6Q0M7Ozs7OztHQU1HO0FBQ29CLDRCQUFNLEdBQW1CLElBQUksbUJBQW1CLEVBQUUsQ0FBQztBQUUxRTs7Ozs7O0dBTUc7QUFDb0Isa0NBQVksR0FBbUIsSUFBSSx3QkFBd0IsRUFBRSxDQUFDO0FBRXJGOzs7Ozs7R0FNRztBQUNvQix1Q0FBaUIsR0FBbUIsSUFBSSwyQkFBMkIsRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCB7IElFbmdpbmUgfSBmcm9tICcuL2VuZ2luZSc7XG5pbXBvcnQgeyBFbmdpbmVWZXJzaW9uIH0gZnJvbSAnLi9lbmdpbmUtdmVyc2lvbic7XG5pbXBvcnQgeyBJUGFyYW1ldGVyR3JvdXAsIFBhcmFtZXRlckdyb3VwIH0gZnJvbSAnLi9wYXJhbWV0ZXItZ3JvdXAnO1xuXG4vLyBrZWVwIHRoaXMgaW1wb3J0IHNlcGFyYXRlIGZyb20gb3RoZXIgaW1wb3J0cyB0byByZWR1Y2UgY2hhbmNlIGZvciBtZXJnZSBjb25mbGljdHMgd2l0aCB2Mi1tYWluXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tZHVwbGljYXRlLWltcG9ydHMsIGltcG9ydC9vcmRlclxuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbi8qKlxuICogVGhlIGV4dHJhIG9wdGlvbnMgcGFzc2VkIHRvIHRoZSB7QGxpbmsgSUNsdXN0ZXJFbmdpbmUuYmluZFRvQ2x1c3Rlcn0gbWV0aG9kLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsdXN0ZXJFbmdpbmVCaW5kT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgcm9sZSB1c2VkIGZvciBTMyBpbXBvcnRpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgczNJbXBvcnRSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgcm9sZSB1c2VkIGZvciBTMyBleHBvcnRpbmcuXG4gICAqXG4gICAqICBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHMzRXhwb3J0Um9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIGN1c3RvbWVyLXByb3ZpZGVkIFBhcmFtZXRlckdyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlckdyb3VwPzogSVBhcmFtZXRlckdyb3VwO1xufVxuXG4vKipcbiAqIFRoZSB0eXBlIHJldHVybmVkIGZyb20gdGhlIHtAbGluayBJQ2x1c3RlckVuZ2luZS5iaW5kVG9DbHVzdGVyfSBtZXRob2QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2x1c3RlckVuZ2luZUNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgUGFyYW1ldGVyR3JvdXAgdG8gdXNlIGZvciB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBQYXJhbWV0ZXJHcm91cCB3aWxsIGJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlckdyb3VwPzogSVBhcmFtZXRlckdyb3VwO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCB0byB1c2UgZm9yIHRoaXMgY2x1c3RlcixcbiAgICogdW5sZXNzIHRoZSBjdXN0b21lciBzcGVjaWZpZWQgdGhlIHBvcnQgZGlyZWN0bHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdXNlIHRoZSBkZWZhdWx0IHBvcnQgZm9yIGNsdXN0ZXJzICgzMzA2KVxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogRmVhdHVyZXMgc3VwcG9ydGVkIGJ5IHRoZSBkYXRhYmFzZSBlbmdpbmUuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EQkVuZ2luZVZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGZlYXR1cmVzXG4gICAqL1xuICByZWFkb25seSBmZWF0dXJlcz86IENsdXN0ZXJFbmdpbmVGZWF0dXJlcztcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIERhdGFiYXNlIEVuZ2luZSBmZWF0dXJlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENsdXN0ZXJFbmdpbmVGZWF0dXJlcyB7XG4gIC8qKlxuICAgKiBGZWF0dXJlIG5hbWUgZm9yIHRoZSBEQiBpbnN0YW5jZSB0aGF0IHRoZSBJQU0gcm9sZSB0byBhY2Nlc3MgdGhlIFMzIGJ1Y2tldCBmb3IgaW1wb3J0XG4gICAqIGlzIHRvIGJlIGFzc29jaWF0ZWQgd2l0aC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBzM0ltcG9ydCBmZWF0dXJlIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IHMzSW1wb3J0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBGZWF0dXJlIG5hbWUgZm9yIHRoZSBEQiBpbnN0YW5jZSB0aGF0IHRoZSBJQU0gcm9sZSB0byBleHBvcnQgdG8gUzMgYnVja2V0IGlzIHRvIGJlXG4gICAqIGFzc29jaWF0ZWQgd2l0aC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBzM0V4cG9ydCBmZWF0dXJlIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IHMzRXhwb3J0Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFRoZSBpbnRlcmZhY2UgcmVwcmVzZW50aW5nIGEgZGF0YWJhc2UgY2x1c3RlciAoYXMgb3Bwb3NlZCB0byBpbnN0YW5jZSkgZW5naW5lLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElDbHVzdGVyRW5naW5lIGV4dGVuZHMgSUVuZ2luZSB7XG4gIC8qKiBUaGUgYXBwbGljYXRpb24gdXNlZCBieSB0aGlzIGVuZ2luZSB0byBwZXJmb3JtIHJvdGF0aW9uIGZvciBhIHNpbmdsZS11c2VyIHNjZW5hcmlvLiAqL1xuICByZWFkb25seSBzaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcblxuICAvKiogVGhlIGFwcGxpY2F0aW9uIHVzZWQgYnkgdGhpcyBlbmdpbmUgdG8gcGVyZm9ybSByb3RhdGlvbiBmb3IgYSBtdWx0aS11c2VyIHNjZW5hcmlvLiAqL1xuICByZWFkb25seSBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuXG4gIC8qKiBUaGUgbG9nIHR5cGVzIHRoYXQgYXJlIGF2YWlsYWJsZSB3aXRoIHRoaXMgZW5naW5lIHR5cGUgKi9cbiAgcmVhZG9ubHkgc3VwcG9ydGVkTG9nVHlwZXM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBJQU0gUm9sZXMgdXNlZCBmb3IgZGF0YSBpbXBvcnRpbmcgYW5kIGV4cG9ydGluZyBuZWVkIHRvIGJlIGNvbWJpbmVkIGZvciB0aGlzIEVuZ2luZSxcbiAgICogb3IgY2FuIHRoZXkgYmUga2VwdCBzZXBhcmF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvbWJpbmVJbXBvcnRBbmRFeHBvcnRSb2xlcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE1ldGhvZCBjYWxsZWQgd2hlbiB0aGUgZW5naW5lIGlzIHVzZWQgdG8gY3JlYXRlIGEgbmV3IGNsdXN0ZXIuXG4gICAqL1xuICBiaW5kVG9DbHVzdGVyKHNjb3BlOiBDb25zdHJ1Y3QsIG9wdGlvbnM6IENsdXN0ZXJFbmdpbmVCaW5kT3B0aW9ucyk6IENsdXN0ZXJFbmdpbmVDb25maWc7XG59XG5cbmludGVyZmFjZSBDbHVzdGVyRW5naW5lQmFzZVByb3BzIHtcbiAgcmVhZG9ubHkgZW5naW5lVHlwZTogc3RyaW5nO1xuICByZWFkb25seSBzaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcbiAgcmVhZG9ubHkgbXVsdGlVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcbiAgcmVhZG9ubHkgZGVmYXVsdFBvcnQ/OiBudW1iZXI7XG4gIHJlYWRvbmx5IGVuZ2luZVZlcnNpb24/OiBFbmdpbmVWZXJzaW9uO1xuICByZWFkb25seSBmZWF0dXJlcz86IENsdXN0ZXJFbmdpbmVGZWF0dXJlcztcbn1cblxuYWJzdHJhY3QgY2xhc3MgQ2x1c3RlckVuZ2luZUJhc2UgaW1wbGVtZW50cyBJQ2x1c3RlckVuZ2luZSB7XG4gIHB1YmxpYyByZWFkb25seSBlbmdpbmVUeXBlOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBlbmdpbmVWZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcbiAgcHVibGljIHJlYWRvbmx5IHBhcmFtZXRlckdyb3VwRmFtaWx5Pzogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb247XG4gIHB1YmxpYyByZWFkb25seSBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgc3VwcG9ydGVkTG9nVHlwZXM6IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdFBvcnQ/OiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgZmVhdHVyZXM/OiBDbHVzdGVyRW5naW5lRmVhdHVyZXM7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IENsdXN0ZXJFbmdpbmVCYXNlUHJvcHMpIHtcbiAgICB0aGlzLmVuZ2luZVR5cGUgPSBwcm9wcy5lbmdpbmVUeXBlO1xuICAgIHRoaXMuZmVhdHVyZXMgPSBwcm9wcy5mZWF0dXJlcztcbiAgICB0aGlzLnNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uID0gcHJvcHMuc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb247XG4gICAgdGhpcy5tdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uID0gcHJvcHMubXVsdGlVc2VyUm90YXRpb25BcHBsaWNhdGlvbjtcbiAgICB0aGlzLmRlZmF1bHRQb3J0ID0gcHJvcHMuZGVmYXVsdFBvcnQ7XG4gICAgdGhpcy5lbmdpbmVWZXJzaW9uID0gcHJvcHMuZW5naW5lVmVyc2lvbjtcbiAgICB0aGlzLnBhcmFtZXRlckdyb3VwRmFtaWx5ID0gdGhpcy5lbmdpbmVWZXJzaW9uID8gYCR7dGhpcy5lbmdpbmVUeXBlfSR7dGhpcy5lbmdpbmVWZXJzaW9uLm1ham9yVmVyc2lvbn1gIDogdW5kZWZpbmVkO1xuICB9XG5cbiAgcHVibGljIGJpbmRUb0NsdXN0ZXIoc2NvcGU6IENvbnN0cnVjdCwgb3B0aW9uczogQ2x1c3RlckVuZ2luZUJpbmRPcHRpb25zKTogQ2x1c3RlckVuZ2luZUNvbmZpZyB7XG4gICAgY29uc3QgcGFyYW1ldGVyR3JvdXAgPSBvcHRpb25zLnBhcmFtZXRlckdyb3VwID8/IHRoaXMuZGVmYXVsdFBhcmFtZXRlckdyb3VwKHNjb3BlKTtcbiAgICByZXR1cm4ge1xuICAgICAgcGFyYW1ldGVyR3JvdXAsXG4gICAgICBwb3J0OiB0aGlzLmRlZmF1bHRQb3J0LFxuICAgICAgZmVhdHVyZXM6IHRoaXMuZmVhdHVyZXMsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYW4gb3B0aW9uYWwgZGVmYXVsdCBQYXJhbWV0ZXJHcm91cCxcbiAgICogcG9zc2libHkgYW4gaW1wb3J0ZWQgb25lLFxuICAgKiBpZiBvbmUgd2Fzbid0IHByb3ZpZGVkIGJ5IHRoZSBjdXN0b21lciBleHBsaWNpdGx5LlxuICAgKi9cbiAgcHJvdGVjdGVkIGFic3RyYWN0IGRlZmF1bHRQYXJhbWV0ZXJHcm91cChzY29wZTogQ29uc3RydWN0KTogSVBhcmFtZXRlckdyb3VwIHwgdW5kZWZpbmVkO1xufVxuXG5pbnRlcmZhY2UgTXlzcWxDbHVzdGVyRW5naW5lQmFzZVByb3BzIHtcbiAgcmVhZG9ubHkgZW5naW5lVHlwZTogc3RyaW5nO1xuICByZWFkb25seSBlbmdpbmVWZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcbiAgcmVhZG9ubHkgZGVmYXVsdE1ham9yVmVyc2lvbjogc3RyaW5nO1xuICByZWFkb25seSBjb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXM/OiBib29sZWFuO1xufVxuXG5hYnN0cmFjdCBjbGFzcyBNeVNxbENsdXN0ZXJFbmdpbmVCYXNlIGV4dGVuZHMgQ2x1c3RlckVuZ2luZUJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgZW5naW5lRmFtaWx5ID0gJ01ZU1FMJztcbiAgcHVibGljIHJlYWRvbmx5IHN1cHBvcnRlZExvZ1R5cGVzOiBzdHJpbmdbXSA9IFsnZXJyb3InLCAnZ2VuZXJhbCcsICdzbG93cXVlcnknLCAnYXVkaXQnXTtcbiAgcHVibGljIHJlYWRvbmx5IGNvbWJpbmVJbXBvcnRBbmRFeHBvcnRSb2xlcz86IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IE15c3FsQ2x1c3RlckVuZ2luZUJhc2VQcm9wcykge1xuICAgIHN1cGVyKHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24uTVlTUUxfUk9UQVRJT05fU0lOR0xFX1VTRVIsXG4gICAgICBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLk1ZU1FMX1JPVEFUSU9OX01VTFRJX1VTRVIsXG4gICAgICBlbmdpbmVWZXJzaW9uOiBwcm9wcy5lbmdpbmVWZXJzaW9uID8gcHJvcHMuZW5naW5lVmVyc2lvbiA6IHsgbWFqb3JWZXJzaW9uOiBwcm9wcy5kZWZhdWx0TWFqb3JWZXJzaW9uIH0sXG4gICAgfSk7XG4gICAgdGhpcy5jb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXMgPSBwcm9wcy5jb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXM7XG4gIH1cblxuICBwdWJsaWMgYmluZFRvQ2x1c3RlcihzY29wZTogQ29uc3RydWN0LCBvcHRpb25zOiBDbHVzdGVyRW5naW5lQmluZE9wdGlvbnMpOiBDbHVzdGVyRW5naW5lQ29uZmlnIHtcbiAgICBjb25zdCBjb25maWcgPSBzdXBlci5iaW5kVG9DbHVzdGVyKHNjb3BlLCBvcHRpb25zKTtcbiAgICBjb25zdCBwYXJhbWV0ZXJHcm91cCA9IG9wdGlvbnMucGFyYW1ldGVyR3JvdXAgPz8gKG9wdGlvbnMuczNJbXBvcnRSb2xlIHx8IG9wdGlvbnMuczNFeHBvcnRSb2xlXG4gICAgICA/IG5ldyBQYXJhbWV0ZXJHcm91cChzY29wZSwgJ0NsdXN0ZXJQYXJhbWV0ZXJHcm91cCcsIHtcbiAgICAgICAgZW5naW5lOiB0aGlzLFxuICAgICAgfSlcbiAgICAgIDogY29uZmlnLnBhcmFtZXRlckdyb3VwKTtcbiAgICBpZiAob3B0aW9ucy5zM0ltcG9ydFJvbGUpIHtcbiAgICAgIC8vIHZlcnNpb25zIHdoaWNoIGNvbWJpbmUgdGhlIGltcG9ydCBhbmQgZXhwb3J0IFJvbGVzIChyaWdodCBub3csIHRoaXMgaXMgb25seSA4LjApXG4gICAgICAvLyByZXF1aXJlIGEgZGlmZmVyZW50IHBhcmFtZXRlciBuYW1lIChpZGVudGljYWwgZm9yIGJvdGggaW1wb3J0IGFuZCBleHBvcnQpXG4gICAgICBjb25zdCBzM0ltcG9ydFBhcmFtID0gdGhpcy5jb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXNcbiAgICAgICAgPyAnYXdzX2RlZmF1bHRfczNfcm9sZSdcbiAgICAgICAgOiAnYXVyb3JhX2xvYWRfZnJvbV9zM19yb2xlJztcbiAgICAgIHBhcmFtZXRlckdyb3VwPy5hZGRQYXJhbWV0ZXIoczNJbXBvcnRQYXJhbSwgb3B0aW9ucy5zM0ltcG9ydFJvbGUucm9sZUFybik7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnMzRXhwb3J0Um9sZSkge1xuICAgICAgY29uc3QgczNFeHBvcnRQYXJhbSA9IHRoaXMuY29tYmluZUltcG9ydEFuZEV4cG9ydFJvbGVzXG4gICAgICAgID8gJ2F3c19kZWZhdWx0X3MzX3JvbGUnXG4gICAgICAgIDogJ2F1cm9yYV9zZWxlY3RfaW50b19zM19yb2xlJztcbiAgICAgIHBhcmFtZXRlckdyb3VwPy5hZGRQYXJhbWV0ZXIoczNFeHBvcnRQYXJhbSwgb3B0aW9ucy5zM0V4cG9ydFJvbGUucm9sZUFybik7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmNvbmZpZyxcbiAgICAgIHBhcmFtZXRlckdyb3VwLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBUaGUgdmVyc2lvbnMgZm9yIHRoZSBBdXJvcmEgY2x1c3RlciBlbmdpbmVcbiAqICh0aG9zZSByZXR1cm5lZCBieSB7QGxpbmsgRGF0YWJhc2VDbHVzdGVyRW5naW5lLmF1cm9yYX0pLlxuICovXG5leHBvcnQgY2xhc3MgQXVyb3JhRW5naW5lVmVyc2lvbiB7XG4gIC8qKiBWZXJzaW9uIFwiNS42LjEwYVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMEEgPSBBdXJvcmFFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV82KCcxMGEnLCBmYWxzZSk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE3LjlcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xN185ID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xNy45Jyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE5LjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xOV8wID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xOS4wJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE5LjFcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xOV8xID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xOS4xJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE5LjJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xOV8yID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xOS4yJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE5LjVcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xOV81ID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xOS41Jyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjE5LjZcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8xOV82ID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4xOS42Jyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIwLjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMF8wID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMC4wJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIwLjFcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMF8xID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMC4xJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIxLjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMV8wID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMS4wJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIyLjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMl8wID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMi4wJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIyLjFcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMl8xID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMi4xJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIyLjEuM1wiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xXzIyXzFfMyA9IEF1cm9yYUVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzYoJzEuMjIuMS4zJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS42Lm15c3FsX2F1cm9yYS4xLjIyLjJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMV8yMl8yID0gQXVyb3JhRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNignMS4yMi4yJyk7XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBBdXJvcmFFbmdpbmVWZXJzaW9uIHdpdGggYW4gYXJiaXRyYXJ5IHZlcnNpb24uXG4gICAqXG4gICAqIEBwYXJhbSBhdXJvcmFGdWxsVmVyc2lvbiB0aGUgZnVsbCB2ZXJzaW9uIHN0cmluZyxcbiAgICogICBmb3IgZXhhbXBsZSBcIjUuNi5teXNxbF9hdXJvcmEuMS43OC4zLjZcIlxuICAgKiBAcGFyYW0gYXVyb3JhTWFqb3JWZXJzaW9uIHRoZSBtYWpvciB2ZXJzaW9uIG9mIHRoZSBlbmdpbmUsXG4gICAqICAgZGVmYXVsdHMgdG8gXCI1LjZcIlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvZihhdXJvcmFGdWxsVmVyc2lvbjogc3RyaW5nLCBhdXJvcmFNYWpvclZlcnNpb24/OiBzdHJpbmcpOiBBdXJvcmFFbmdpbmVWZXJzaW9uIHtcbiAgICByZXR1cm4gbmV3IEF1cm9yYUVuZ2luZVZlcnNpb24oYXVyb3JhRnVsbFZlcnNpb24sIGF1cm9yYU1ham9yVmVyc2lvbik7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBidWlsdEluXzVfNihtaW5vclZlcnNpb246IHN0cmluZywgYWRkU3RhbmRhcmRQcmVmaXg6IGJvb2xlYW4gPSB0cnVlKTogQXVyb3JhRW5naW5lVmVyc2lvbiB7XG4gICAgcmV0dXJuIG5ldyBBdXJvcmFFbmdpbmVWZXJzaW9uKGA1LjYuJHthZGRTdGFuZGFyZFByZWZpeCA/ICdteXNxbF9hdXJvcmEuJyA6ICcnfSR7bWlub3JWZXJzaW9ufWApO1xuICB9XG5cbiAgLyoqIFRoZSBmdWxsIHZlcnNpb24gc3RyaW5nLCBmb3IgZXhhbXBsZSwgXCI1LjYubXlzcWxfYXVyb3JhLjEuNzguMy42XCIuICovXG4gIHB1YmxpYyByZWFkb25seSBhdXJvcmFGdWxsVmVyc2lvbjogc3RyaW5nO1xuICAvKiogVGhlIG1ham9yIHZlcnNpb24gb2YgdGhlIGVuZ2luZS4gQ3VycmVudGx5LCBpdCdzIGFsd2F5cyBcIjUuNlwiLiAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXVyb3JhTWFqb3JWZXJzaW9uOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihhdXJvcmFGdWxsVmVyc2lvbjogc3RyaW5nLCBhdXJvcmFNYWpvclZlcnNpb246IHN0cmluZyA9ICc1LjYnKSB7XG4gICAgdGhpcy5hdXJvcmFGdWxsVmVyc2lvbiA9IGF1cm9yYUZ1bGxWZXJzaW9uO1xuICAgIHRoaXMuYXVyb3JhTWFqb3JWZXJzaW9uID0gYXVyb3JhTWFqb3JWZXJzaW9uO1xuICB9XG59XG5cbi8qKlxuICogQ3JlYXRpb24gcHJvcGVydGllcyBvZiB0aGUgcGxhaW4gQXVyb3JhIGRhdGFiYXNlIGNsdXN0ZXIgZW5naW5lLlxuICogVXNlZCBpbiB7QGxpbmsgRGF0YWJhc2VDbHVzdGVyRW5naW5lLmF1cm9yYX0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXVyb3JhQ2x1c3RlckVuZ2luZVByb3BzIHtcbiAgLyoqIFRoZSB2ZXJzaW9uIG9mIHRoZSBBdXJvcmEgY2x1c3RlciBlbmdpbmUuICovXG4gIHJlYWRvbmx5IHZlcnNpb246IEF1cm9yYUVuZ2luZVZlcnNpb247XG59XG5cbmNsYXNzIEF1cm9yYUNsdXN0ZXJFbmdpbmUgZXh0ZW5kcyBNeVNxbENsdXN0ZXJFbmdpbmVCYXNlIHtcbiAgY29uc3RydWN0b3IodmVyc2lvbj86IEF1cm9yYUVuZ2luZVZlcnNpb24pIHtcbiAgICBzdXBlcih7XG4gICAgICBlbmdpbmVUeXBlOiAnYXVyb3JhJyxcbiAgICAgIGVuZ2luZVZlcnNpb246IHZlcnNpb25cbiAgICAgICAgPyB7XG4gICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24uYXVyb3JhRnVsbFZlcnNpb24sXG4gICAgICAgICAgbWFqb3JWZXJzaW9uOiB2ZXJzaW9uLmF1cm9yYU1ham9yVmVyc2lvbixcbiAgICAgICAgfVxuICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIGRlZmF1bHRNYWpvclZlcnNpb246ICc1LjYnLFxuICAgIH0pO1xuICB9XG5cbiAgcHJvdGVjdGVkIGRlZmF1bHRQYXJhbWV0ZXJHcm91cChfc2NvcGU6IENvbnN0cnVjdCk6IElQYXJhbWV0ZXJHcm91cCB8IHVuZGVmaW5lZCB7XG4gICAgLy8gdGhlIGRlZmF1bHQuYXVyb3JhNS42IFBhcmFtZXRlckdyb3VwIGlzIGFjdHVhbGx5IHRoZSBkZWZhdWx0LFxuICAgIC8vIHNvIGp1c3QgcmV0dXJuIHVuZGVmaW5lZCBpbiB0aGlzIGNhc2VcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG59XG5cbi8qKlxuICogVGhlIHZlcnNpb25zIGZvciB0aGUgQXVyb3JhIE15U1FMIGNsdXN0ZXIgZW5naW5lXG4gKiAodGhvc2UgcmV0dXJuZWQgYnkge0BsaW5rIERhdGFiYXNlQ2x1c3RlckVuZ2luZS5hdXJvcmFNeXNxbH0pLlxuICovXG5leHBvcnQgY2xhc3MgQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uIHtcbiAgLyoqIFZlcnNpb24gXCI1LjcuMTJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfNV83XzEyID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcxMicsIGZhbHNlKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDMuMlwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzAzXzIgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDMuMicpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wMy4zXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDNfMyA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wMy4zJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjAzLjRcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wM180ID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjAzLjQnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDQuMFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA0XzAgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDQuMCcpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wNC4xXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDRfMSA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wNC4xJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA0LjJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wNF8yID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA0LjInKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDQuM1wiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA0XzMgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDQuMycpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wNC40XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDRfNCA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wNC40Jyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA0LjVcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wNF81ID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA0LjUnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDQuNlwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA0XzYgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDQuNicpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wNC43XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDRfNyA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wNC43Jyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA0LjhcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wNF84ID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA0LjgnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDUuMFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA1XzAgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDUuMCcpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wNi4wXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDZfMCA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wNi4wJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA3LjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wN18wID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA3LjAnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDcuMVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA3XzEgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDcuMScpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wNy4yXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDdfMiA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wNy4yJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA4LjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wOF8wID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA4LjAnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDguMVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA4XzEgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDguMScpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wOC4yXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDhfMiA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wOC4yJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA5LjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wOV8wID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA5LjAnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMDkuMVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzA5XzEgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMDkuMScpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4wOS4yXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMDlfMiA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4wOS4yJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjA5LjNcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8wOV8zID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjA5LjMnKTtcbiAgLyoqIFZlcnNpb24gXCI1LjcubXlzcWxfYXVyb3JhLjIuMTAuMFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8yXzEwXzAgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl81XzcoJzIuMTAuMCcpO1xuICAvKiogVmVyc2lvbiBcIjUuNy5teXNxbF9hdXJvcmEuMi4xMC4xXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzJfMTBfMSA9IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5idWlsdEluXzVfNygnMi4xMC4xJyk7XG4gIC8qKiBWZXJzaW9uIFwiNS43Lm15c3FsX2F1cm9yYS4yLjEwLjJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMl8xMF8yID0gQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLmJ1aWx0SW5fNV83KCcyLjEwLjInKTtcbiAgLyoqIFZlcnNpb24gXCI4LjAubXlzcWxfYXVyb3JhLjMuMDEuMFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8zXzAxXzAgPSBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uYnVpbHRJbl84XzAoJzMuMDEuMCcpO1xuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uIHdpdGggYW4gYXJiaXRyYXJ5IHZlcnNpb24uXG4gICAqXG4gICAqIEBwYXJhbSBhdXJvcmFNeXNxbEZ1bGxWZXJzaW9uIHRoZSBmdWxsIHZlcnNpb24gc3RyaW5nLFxuICAgKiAgIGZvciBleGFtcGxlIFwiNS43Lm15c3FsX2F1cm9yYS4yLjc4LjMuNlwiXG4gICAqIEBwYXJhbSBhdXJvcmFNeXNxbE1ham9yVmVyc2lvbiB0aGUgbWFqb3IgdmVyc2lvbiBvZiB0aGUgZW5naW5lLFxuICAgKiAgIGRlZmF1bHRzIHRvIFwiNS43XCJcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgb2YoYXVyb3JhTXlzcWxGdWxsVmVyc2lvbjogc3RyaW5nLCBhdXJvcmFNeXNxbE1ham9yVmVyc2lvbj86IHN0cmluZyk6IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbiB7XG4gICAgcmV0dXJuIG5ldyBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24oYXVyb3JhTXlzcWxGdWxsVmVyc2lvbiwgYXVyb3JhTXlzcWxNYWpvclZlcnNpb24pO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgYnVpbHRJbl81XzcobWlub3JWZXJzaW9uOiBzdHJpbmcsIGFkZFN0YW5kYXJkUHJlZml4OiBib29sZWFuID0gdHJ1ZSk6IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbiB7XG4gICAgcmV0dXJuIG5ldyBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24oYDUuNy4ke2FkZFN0YW5kYXJkUHJlZml4ID8gJ215c3FsX2F1cm9yYS4nIDogJyd9JHttaW5vclZlcnNpb259YCk7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBidWlsdEluXzhfMChtaW5vclZlcnNpb246IHN0cmluZyk6IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbiB7XG4gICAgLy8gOC4wIG9mIHRoZSBNeVNRTCBlbmdpbmUgbmVlZHMgdG8gY29tYmluZSB0aGUgaW1wb3J0IGFuZCBleHBvcnQgUm9sZXNcbiAgICByZXR1cm4gbmV3IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbihgOC4wLm15c3FsX2F1cm9yYS4ke21pbm9yVmVyc2lvbn1gLCAnOC4wJywgdHJ1ZSk7XG4gIH1cblxuICAvKiogVGhlIGZ1bGwgdmVyc2lvbiBzdHJpbmcsIGZvciBleGFtcGxlLCBcIjUuNy5teXNxbF9hdXJvcmEuMS43OC4zLjZcIi4gKi9cbiAgcHVibGljIHJlYWRvbmx5IGF1cm9yYU15c3FsRnVsbFZlcnNpb246IHN0cmluZztcbiAgLyoqIFRoZSBtYWpvciB2ZXJzaW9uIG9mIHRoZSBlbmdpbmUuIEN1cnJlbnRseSwgaXQncyBlaXRoZXIgXCI1LjdcIiwgb3IgXCI4LjBcIi4gKi9cbiAgcHVibGljIHJlYWRvbmx5IGF1cm9yYU15c3FsTWFqb3JWZXJzaW9uOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgdmVyc2lvbiByZXF1aXJlcyBjb21iaW5pbmcgdGhlIGltcG9ydCBhbmQgZXhwb3J0IElBTSBSb2xlcy5cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgX2NvbWJpbmVJbXBvcnRBbmRFeHBvcnRSb2xlcz86IGJvb2xlYW47XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihcbiAgICBhdXJvcmFNeXNxbEZ1bGxWZXJzaW9uOiBzdHJpbmcsIGF1cm9yYU15c3FsTWFqb3JWZXJzaW9uOiBzdHJpbmcgPSAnNS43JyxcbiAgICBjb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXM/OiBib29sZWFuLFxuICApIHtcbiAgICB0aGlzLmF1cm9yYU15c3FsRnVsbFZlcnNpb24gPSBhdXJvcmFNeXNxbEZ1bGxWZXJzaW9uO1xuICAgIHRoaXMuYXVyb3JhTXlzcWxNYWpvclZlcnNpb24gPSBhdXJvcmFNeXNxbE1ham9yVmVyc2lvbjtcbiAgICB0aGlzLl9jb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXMgPSBjb21iaW5lSW1wb3J0QW5kRXhwb3J0Um9sZXM7XG4gIH1cbn1cblxuLyoqXG4gKiBDcmVhdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSBBdXJvcmEgTXlTUUwgZGF0YWJhc2UgY2x1c3RlciBlbmdpbmUuXG4gKiBVc2VkIGluIHtAbGluayBEYXRhYmFzZUNsdXN0ZXJFbmdpbmUuYXVyb3JhTXlzcWx9LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1cm9yYU15c3FsQ2x1c3RlckVuZ2luZVByb3BzIHtcbiAgLyoqIFRoZSB2ZXJzaW9uIG9mIHRoZSBBdXJvcmEgTXlTUUwgY2x1c3RlciBlbmdpbmUuICovXG4gIHJlYWRvbmx5IHZlcnNpb246IEF1cm9yYU15c3FsRW5naW5lVmVyc2lvbjtcbn1cblxuY2xhc3MgQXVyb3JhTXlzcWxDbHVzdGVyRW5naW5lIGV4dGVuZHMgTXlTcWxDbHVzdGVyRW5naW5lQmFzZSB7XG4gIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBBdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24pIHtcbiAgICBzdXBlcih7XG4gICAgICBlbmdpbmVUeXBlOiAnYXVyb3JhLW15c3FsJyxcbiAgICAgIGVuZ2luZVZlcnNpb246IHZlcnNpb25cbiAgICAgICAgPyB7XG4gICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24uYXVyb3JhTXlzcWxGdWxsVmVyc2lvbixcbiAgICAgICAgICBtYWpvclZlcnNpb246IHZlcnNpb24uYXVyb3JhTXlzcWxNYWpvclZlcnNpb24sXG4gICAgICAgIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBkZWZhdWx0TWFqb3JWZXJzaW9uOiAnNS43JyxcbiAgICAgIGNvbWJpbmVJbXBvcnRBbmRFeHBvcnRSb2xlczogdmVyc2lvbj8uX2NvbWJpbmVJbXBvcnRBbmRFeHBvcnRSb2xlcyxcbiAgICB9KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBkZWZhdWx0UGFyYW1ldGVyR3JvdXAoc2NvcGU6IENvbnN0cnVjdCk6IElQYXJhbWV0ZXJHcm91cCB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIFBhcmFtZXRlckdyb3VwLmZyb21QYXJhbWV0ZXJHcm91cE5hbWUoc2NvcGUsICdBdXJvcmFNeVNxbERhdGFiYXNlQ2x1c3RlckVuZ2luZURlZmF1bHRQYXJhbWV0ZXJHcm91cCcsXG4gICAgICBgZGVmYXVsdC4ke3RoaXMucGFyYW1ldGVyR3JvdXBGYW1pbHl9YCk7XG4gIH1cbn1cblxuLyoqXG4gKiBGZWF0dXJlcyBzdXBwb3J0ZWQgYnkgdGhpcyB2ZXJzaW9uIG9mIHRoZSBBdXJvcmEgUG9zdGdyZXMgY2x1c3RlciBlbmdpbmUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXVyb3JhUG9zdGdyZXNFbmdpbmVGZWF0dXJlcyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgdmVyc2lvbiBvZiB0aGUgQXVyb3JhIFBvc3RncmVzIGNsdXN0ZXIgZW5naW5lIHN1cHBvcnRzIHRoZSBTMyBkYXRhIGltcG9ydCBmZWF0dXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgczNJbXBvcnQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgdmVyc2lvbiBvZiB0aGUgQXVyb3JhIFBvc3RncmVzIGNsdXN0ZXIgZW5naW5lIHN1cHBvcnRzIHRoZSBTMyBkYXRhIGV4cG9ydCBmZWF0dXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgczNFeHBvcnQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFRoZSB2ZXJzaW9ucyBmb3IgdGhlIEF1cm9yYSBQb3N0Z3JlU1FMIGNsdXN0ZXIgZW5naW5lXG4gKiAodGhvc2UgcmV0dXJuZWQgYnkge0BsaW5rIERhdGFiYXNlQ2x1c3RlckVuZ2luZS5hdXJvcmFQb3N0Z3Jlc30pLlxuICovXG5leHBvcnQgY2xhc3MgQXVyb3JhUG9zdGdyZXNFbmdpbmVWZXJzaW9uIHtcbiAgLyoqIFZlcnNpb24gXCI5LjYuOFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfOCA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignOS42LjgnLCAnOS42Jyk7XG4gIC8qKiBWZXJzaW9uIFwiOS42LjlcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfOV82XzkgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi45JywgJzkuNicpO1xuICAvKiogVmVyc2lvbiBcIjkuNi4xMVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTEgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xMScsICc5LjYnKTtcbiAgLyoqIFZlcnNpb24gXCI5LjYuMTJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfOV82XzEyID0gQXVyb3JhUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuMTInLCAnOS42Jyk7XG4gIC8qKiBWZXJzaW9uIFwiOS42LjE2XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl8xNiA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignOS42LjE2JywgJzkuNicpO1xuICAvKiogVmVyc2lvbiBcIjkuNi4xN1wiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTcgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xNycsICc5LjYnKTtcbiAgLyoqIFZlcnNpb24gXCI5LjYuMThcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfOV82XzE4ID0gQXVyb3JhUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuMTgnLCAnOS42Jyk7XG4gIC8qKiBWZXJzaW9uIFwiOS42LjE5XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl8xOSA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignOS42LjE5JywgJzkuNicpO1xuICAvKiogVmVyc2lvbiBcIjEwLjRcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNCA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNCcsICcxMCcpO1xuICAvKiogVmVyc2lvbiBcIjEwLjVcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNSA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNScsICcxMCcpO1xuICAvKiogVmVyc2lvbiBcIjEwLjZcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNiA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNicsICcxMCcpO1xuICAvKiogVmVyc2lvbiBcIjEwLjdcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNyA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNycsICcxMCcsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTFcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTEgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjExJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTJcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTIgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjEyJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTNcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTMgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjEzJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTRcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTQgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjE0JywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTZcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTYgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjE2JywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMThcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTggPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjE4JywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMTlcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTkgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjE5JywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTAuMjBcIi4gKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMjAgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjIwJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSwgczNFeHBvcnQ6IHRydWUgfSk7XG4gIC8qKiBWZXJzaW9uIFwiMTEuNFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV80ID0gQXVyb3JhUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMS40JywgJzExJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS42XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzYgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjYnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS43XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzcgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjcnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS44XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzggPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjgnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS45XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzkgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjknLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS4xMVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8xMSA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuMTEnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS4xM1wiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8xMyA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuMTMnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS4xNFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8xNCA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuMTQnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMS4xNVwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8xNSA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuMTUnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMi40XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzQgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyLjQnLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMi42XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzYgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyLjYnLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMi44XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzggPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyLjgnLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMi44XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzkgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyLjknLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMi4xMFwiLiAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xMCA9IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTIuMTAnLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMy4zXCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEzXzMgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEzLjMnLCAnMTMnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMy40XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEzXzQgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEzLjQnLCAnMTMnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMy41XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEzXzUgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEzLjUnLCAnMTMnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcbiAgLyoqIFZlcnNpb24gXCIxMy42XCIuICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEzXzYgPSBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEzLjYnLCAnMTMnLCB7IHMzSW1wb3J0OiB0cnVlLCBzM0V4cG9ydDogdHJ1ZSB9KTtcblxuICAvKipcbiAgICogQ3JlYXRlIGEgbmV3IEF1cm9yYVBvc3RncmVzRW5naW5lVmVyc2lvbiB3aXRoIGFuIGFyYml0cmFyeSB2ZXJzaW9uLlxuICAgKlxuICAgKiBAcGFyYW0gYXVyb3JhUG9zdGdyZXNGdWxsVmVyc2lvbiB0aGUgZnVsbCB2ZXJzaW9uIHN0cmluZyxcbiAgICogICBmb3IgZXhhbXBsZSBcIjkuNi4yNS4xXCJcbiAgICogQHBhcmFtIGF1cm9yYVBvc3RncmVzTWFqb3JWZXJzaW9uIHRoZSBtYWpvciB2ZXJzaW9uIG9mIHRoZSBlbmdpbmUsXG4gICAqICAgZm9yIGV4YW1wbGUgXCI5LjZcIlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvZihhdXJvcmFQb3N0Z3Jlc0Z1bGxWZXJzaW9uOiBzdHJpbmcsIGF1cm9yYVBvc3RncmVzTWFqb3JWZXJzaW9uOiBzdHJpbmcsXG4gICAgYXVyb3JhUG9zdGdyZXNGZWF0dXJlcz86IEF1cm9yYVBvc3RncmVzRW5naW5lRmVhdHVyZXMpOiBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ge1xuXG4gICAgcmV0dXJuIG5ldyBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24oYXVyb3JhUG9zdGdyZXNGdWxsVmVyc2lvbiwgYXVyb3JhUG9zdGdyZXNNYWpvclZlcnNpb24sIGF1cm9yYVBvc3RncmVzRmVhdHVyZXMpO1xuICB9XG5cbiAgLyoqIFRoZSBmdWxsIHZlcnNpb24gc3RyaW5nLCBmb3IgZXhhbXBsZSwgXCI5LjYuMjUuMVwiLiAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXVyb3JhUG9zdGdyZXNGdWxsVmVyc2lvbjogc3RyaW5nO1xuICAvKiogVGhlIG1ham9yIHZlcnNpb24gb2YgdGhlIGVuZ2luZSwgZm9yIGV4YW1wbGUsIFwiOS42XCIuICovXG4gIHB1YmxpYyByZWFkb25seSBhdXJvcmFQb3N0Z3Jlc01ham9yVmVyc2lvbjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHN1cHBvcnRlZCBmZWF0dXJlcyBmb3IgdGhlIERCIGVuZ2luZVxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBfZmVhdHVyZXM6IENsdXN0ZXJFbmdpbmVGZWF0dXJlcztcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGF1cm9yYVBvc3RncmVzRnVsbFZlcnNpb246IHN0cmluZywgYXVyb3JhUG9zdGdyZXNNYWpvclZlcnNpb246IHN0cmluZywgYXVyb3JhUG9zdGdyZXNGZWF0dXJlcz86IEF1cm9yYVBvc3RncmVzRW5naW5lRmVhdHVyZXMpIHtcbiAgICB0aGlzLmF1cm9yYVBvc3RncmVzRnVsbFZlcnNpb24gPSBhdXJvcmFQb3N0Z3Jlc0Z1bGxWZXJzaW9uO1xuICAgIHRoaXMuYXVyb3JhUG9zdGdyZXNNYWpvclZlcnNpb24gPSBhdXJvcmFQb3N0Z3Jlc01ham9yVmVyc2lvbjtcbiAgICB0aGlzLl9mZWF0dXJlcyA9IHtcbiAgICAgIHMzSW1wb3J0OiBhdXJvcmFQb3N0Z3Jlc0ZlYXR1cmVzPy5zM0ltcG9ydCA/ICdzM0ltcG9ydCcgOiB1bmRlZmluZWQsXG4gICAgICBzM0V4cG9ydDogYXVyb3JhUG9zdGdyZXNGZWF0dXJlcz8uczNFeHBvcnQgPyAnczNFeHBvcnQnIDogdW5kZWZpbmVkLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBDcmVhdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSBBdXJvcmEgUG9zdGdyZVNRTCBkYXRhYmFzZSBjbHVzdGVyIGVuZ2luZS5cbiAqIFVzZWQgaW4ge0BsaW5rIERhdGFiYXNlQ2x1c3RlckVuZ2luZS5hdXJvcmFQb3N0Z3Jlc30uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXVyb3JhUG9zdGdyZXNDbHVzdGVyRW5naW5lUHJvcHMge1xuICAvKiogVGhlIHZlcnNpb24gb2YgdGhlIEF1cm9yYSBQb3N0Z3JlU1FMIGNsdXN0ZXIgZW5naW5lLiAqL1xuICByZWFkb25seSB2ZXJzaW9uOiBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb247XG59XG5cbmNsYXNzIEF1cm9yYVBvc3RncmVzQ2x1c3RlckVuZ2luZSBleHRlbmRzIENsdXN0ZXJFbmdpbmVCYXNlIHtcbiAgLyoqXG4gICAqIGZlYXR1cmUgbmFtZSBmb3IgdGhlIFMzIGRhdGEgaW1wb3J0IGZlYXR1cmVcbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFMzX0lNUE9SVF9GRUFUVVJFX05BTUUgPSAnczNJbXBvcnQnO1xuXG4gIC8qKlxuICAgKiBmZWF0dXJlIG5hbWUgZm9yIHRoZSBTMyBkYXRhIGV4cG9ydCBmZWF0dXJlXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTM19FWFBPUlRfRkVBVFVSRV9OQU1FID0gJ3MzRXhwb3J0JztcblxuICBwdWJsaWMgcmVhZG9ubHkgZW5naW5lRmFtaWx5ID0gJ1BPU1RHUkVTUUwnO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdFVzZXJuYW1lID0gJ3Bvc3RncmVzJztcbiAgcHVibGljIHJlYWRvbmx5IHN1cHBvcnRlZExvZ1R5cGVzOiBzdHJpbmdbXSA9IFsncG9zdGdyZXNxbCddO1xuXG4gIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBBdXJvcmFQb3N0Z3Jlc0VuZ2luZVZlcnNpb24pIHtcbiAgICBzdXBlcih7XG4gICAgICBlbmdpbmVUeXBlOiAnYXVyb3JhLXBvc3RncmVzcWwnLFxuICAgICAgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24uUE9TVEdSRVNfUk9UQVRJT05fU0lOR0xFX1VTRVIsXG4gICAgICBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLlBPU1RHUkVTX1JPVEFUSU9OX01VTFRJX1VTRVIsXG4gICAgICBkZWZhdWx0UG9ydDogNTQzMixcbiAgICAgIGVuZ2luZVZlcnNpb246IHZlcnNpb25cbiAgICAgICAgPyB7XG4gICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24uYXVyb3JhUG9zdGdyZXNGdWxsVmVyc2lvbixcbiAgICAgICAgICBtYWpvclZlcnNpb246IHZlcnNpb24uYXVyb3JhUG9zdGdyZXNNYWpvclZlcnNpb24sXG4gICAgICAgIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBmZWF0dXJlczogdmVyc2lvblxuICAgICAgICA/IHtcbiAgICAgICAgICBzM0ltcG9ydDogdmVyc2lvbi5fZmVhdHVyZXMuczNJbXBvcnQgPyBBdXJvcmFQb3N0Z3Jlc0NsdXN0ZXJFbmdpbmUuUzNfSU1QT1JUX0ZFQVRVUkVfTkFNRSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBzM0V4cG9ydDogdmVyc2lvbi5fZmVhdHVyZXMuczNFeHBvcnQgPyBBdXJvcmFQb3N0Z3Jlc0NsdXN0ZXJFbmdpbmUuUzNfRVhQT1JUX0ZFQVRVUkVfTkFNRSA6IHVuZGVmaW5lZCxcbiAgICAgICAgfVxuICAgICAgICA6IHtcbiAgICAgICAgICBzM0ltcG9ydDogQXVyb3JhUG9zdGdyZXNDbHVzdGVyRW5naW5lLlMzX0lNUE9SVF9GRUFUVVJFX05BTUUsXG4gICAgICAgICAgczNFeHBvcnQ6IEF1cm9yYVBvc3RncmVzQ2x1c3RlckVuZ2luZS5TM19FWFBPUlRfRkVBVFVSRV9OQU1FLFxuICAgICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIGJpbmRUb0NsdXN0ZXIoc2NvcGU6IENvbnN0cnVjdCwgb3B0aW9uczogQ2x1c3RlckVuZ2luZUJpbmRPcHRpb25zKTogQ2x1c3RlckVuZ2luZUNvbmZpZyB7XG4gICAgY29uc3QgY29uZmlnID0gc3VwZXIuYmluZFRvQ2x1c3RlcihzY29wZSwgb3B0aW9ucyk7XG4gICAgLy8gc2tpcCB2YWxpZGF0aW9uIGZvciB1bnZlcnNpb25lZCBhcyBpdCBtaWdodCBiZSBzdXBwb3J0ZWQvdW5zdXBwb3J0ZWQuIHdlIGNhbm5vdCByZWxpYWJseSB0ZWxsIGF0IGNvbXBpbGUtdGltZVxuICAgIGlmICh0aGlzLmVuZ2luZVZlcnNpb24/LmZ1bGxWZXJzaW9uKSB7XG4gICAgICBpZiAob3B0aW9ucy5zM0ltcG9ydFJvbGUgJiYgIShjb25maWcuZmVhdHVyZXM/LnMzSW1wb3J0KSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHMzSW1wb3J0IGlzIG5vdCBzdXBwb3J0ZWQgZm9yIFBvc3RncmVzIHZlcnNpb246ICR7dGhpcy5lbmdpbmVWZXJzaW9uLmZ1bGxWZXJzaW9ufS4gVXNlIGEgdmVyc2lvbiB0aGF0IHN1cHBvcnRzIHRoZSBzM0ltcG9ydCBmZWF0dXJlLmApO1xuICAgICAgfVxuICAgICAgaWYgKG9wdGlvbnMuczNFeHBvcnRSb2xlICYmICEoY29uZmlnLmZlYXR1cmVzPy5zM0V4cG9ydCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBzM0V4cG9ydCBpcyBub3Qgc3VwcG9ydGVkIGZvciBQb3N0Z3JlcyB2ZXJzaW9uOiAke3RoaXMuZW5naW5lVmVyc2lvbi5mdWxsVmVyc2lvbn0uIFVzZSBhIHZlcnNpb24gdGhhdCBzdXBwb3J0cyB0aGUgczNFeHBvcnQgZmVhdHVyZS5gKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGNvbmZpZztcbiAgfVxuXG4gIHByb3RlY3RlZCBkZWZhdWx0UGFyYW1ldGVyR3JvdXAoc2NvcGU6IENvbnN0cnVjdCk6IElQYXJhbWV0ZXJHcm91cCB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCF0aGlzLnBhcmFtZXRlckdyb3VwRmFtaWx5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvdWxkIG5vdCBjcmVhdGUgYSBuZXcgUGFyYW1ldGVyR3JvdXAgZm9yIGFuIHVudmVyc2lvbmVkIGF1cm9yYS1wb3N0Z3Jlc3FsIGNsdXN0ZXIgZW5naW5lLiAnICtcbiAgICAgICAgJ1BsZWFzZSBlaXRoZXIgdXNlIGEgdmVyc2lvbmVkIGVuZ2luZSwgb3IgcGFzcyBhbiBleHBsaWNpdCBQYXJhbWV0ZXJHcm91cCB3aGVuIGNyZWF0aW5nIHRoZSBjbHVzdGVyJyk7XG4gICAgfVxuICAgIHJldHVybiBQYXJhbWV0ZXJHcm91cC5mcm9tUGFyYW1ldGVyR3JvdXBOYW1lKHNjb3BlLCAnQXVyb3JhUG9zdGdyZVNxbERhdGFiYXNlQ2x1c3RlckVuZ2luZURlZmF1bHRQYXJhbWV0ZXJHcm91cCcsXG4gICAgICBgZGVmYXVsdC4ke3RoaXMucGFyYW1ldGVyR3JvdXBGYW1pbHl9YCk7XG4gIH1cbn1cblxuLyoqXG4gKiBBIGRhdGFiYXNlIGNsdXN0ZXIgZW5naW5lLiBQcm92aWRlcyBtYXBwaW5nIHRvIHRoZSBzZXJ2ZXJsZXNzIGFwcGxpY2F0aW9uXG4gKiB1c2VkIGZvciBzZWNyZXQgcm90YXRpb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBEYXRhYmFzZUNsdXN0ZXJFbmdpbmUge1xuICAvKipcbiAgICogVGhlIHVudmVyc2lvbmVkICdhdXJvcmEnIGNsdXN0ZXIgZW5naW5lLlxuICAgKlxuICAgKiAqKk5vdGUqKjogd2UgZG8gbm90IHJlY29tbWVuZCB1c2luZyB1bnZlcnNpb25lZCBlbmdpbmVzIGZvciBub24tc2VydmVybGVzcyBDbHVzdGVycyxcbiAgICogICBhcyB0aGF0IGNhbiBwb3NlIGFuIGF2YWlsYWJpbGl0eSByaXNrLlxuICAgKiAgIFdlIHJlY29tbWVuZCB1c2luZyB2ZXJzaW9uZWQgZW5naW5lcyBjcmVhdGVkIHVzaW5nIHRoZSB7QGxpbmsgYXVyb3JhKCl9IG1ldGhvZFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBBVVJPUkE6IElDbHVzdGVyRW5naW5lID0gbmV3IEF1cm9yYUNsdXN0ZXJFbmdpbmUoKTtcblxuICAvKipcbiAgICogVGhlIHVudmVyc2lvbmVkICdhdXJvcmEtbXNxbCcgY2x1c3RlciBlbmdpbmUuXG4gICAqXG4gICAqICoqTm90ZSoqOiB3ZSBkbyBub3QgcmVjb21tZW5kIHVzaW5nIHVudmVyc2lvbmVkIGVuZ2luZXMgZm9yIG5vbi1zZXJ2ZXJsZXNzIENsdXN0ZXJzLFxuICAgKiAgIGFzIHRoYXQgY2FuIHBvc2UgYW4gYXZhaWxhYmlsaXR5IHJpc2suXG4gICAqICAgV2UgcmVjb21tZW5kIHVzaW5nIHZlcnNpb25lZCBlbmdpbmVzIGNyZWF0ZWQgdXNpbmcgdGhlIHtAbGluayBhdXJvcmFNeXNxbCgpfSBtZXRob2RcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQVVST1JBX01ZU1FMOiBJQ2x1c3RlckVuZ2luZSA9IG5ldyBBdXJvcmFNeXNxbENsdXN0ZXJFbmdpbmUoKTtcblxuICAvKipcbiAgICogVGhlIHVudmVyc2lvbmVkICdhdXJvcmEtcG9zdGdyZXNxbCcgY2x1c3RlciBlbmdpbmUuXG4gICAqXG4gICAqICoqTm90ZSoqOiB3ZSBkbyBub3QgcmVjb21tZW5kIHVzaW5nIHVudmVyc2lvbmVkIGVuZ2luZXMgZm9yIG5vbi1zZXJ2ZXJsZXNzIENsdXN0ZXJzLFxuICAgKiAgIGFzIHRoYXQgY2FuIHBvc2UgYW4gYXZhaWxhYmlsaXR5IHJpc2suXG4gICAqICAgV2UgcmVjb21tZW5kIHVzaW5nIHZlcnNpb25lZCBlbmdpbmVzIGNyZWF0ZWQgdXNpbmcgdGhlIHtAbGluayBhdXJvcmFQb3N0Z3JlcygpfSBtZXRob2RcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQVVST1JBX1BPU1RHUkVTUUw6IElDbHVzdGVyRW5naW5lID0gbmV3IEF1cm9yYVBvc3RncmVzQ2x1c3RlckVuZ2luZSgpO1xuXG4gIC8qKiBDcmVhdGVzIGEgbmV3IHBsYWluIEF1cm9yYSBkYXRhYmFzZSBjbHVzdGVyIGVuZ2luZS4gKi9cbiAgcHVibGljIHN0YXRpYyBhdXJvcmEocHJvcHM6IEF1cm9yYUNsdXN0ZXJFbmdpbmVQcm9wcyk6IElDbHVzdGVyRW5naW5lIHtcbiAgICByZXR1cm4gbmV3IEF1cm9yYUNsdXN0ZXJFbmdpbmUocHJvcHMudmVyc2lvbik7XG4gIH1cblxuICAvKiogQ3JlYXRlcyBhIG5ldyBBdXJvcmEgTXlTUUwgZGF0YWJhc2UgY2x1c3RlciBlbmdpbmUuICovXG4gIHB1YmxpYyBzdGF0aWMgYXVyb3JhTXlzcWwocHJvcHM6IEF1cm9yYU15c3FsQ2x1c3RlckVuZ2luZVByb3BzKTogSUNsdXN0ZXJFbmdpbmUge1xuICAgIHJldHVybiBuZXcgQXVyb3JhTXlzcWxDbHVzdGVyRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICB9XG5cbiAgLyoqIENyZWF0ZXMgYSBuZXcgQXVyb3JhIFBvc3RncmVTUUwgZGF0YWJhc2UgY2x1c3RlciBlbmdpbmUuICovXG4gIHB1YmxpYyBzdGF0aWMgYXVyb3JhUG9zdGdyZXMocHJvcHM6IEF1cm9yYVBvc3RncmVzQ2x1c3RlckVuZ2luZVByb3BzKTogSUNsdXN0ZXJFbmdpbmUge1xuICAgIHJldHVybiBuZXcgQXVyb3JhUG9zdGdyZXNDbHVzdGVyRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICB9XG59XG4iXX0=