"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const s3assets = require("@aws-cdk/aws-s3-assets");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_kinesis_1 = require("@aws-cdk/aws-kinesis");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-solutions-constructs/core");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Pattern minimal deployment
// --------------------------------------------------------------
test('Pattern minimal deployment', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const props = {
        glueJobProps: {
            command: {
                name: 'glueetl',
                pythonVersion: '3',
                scriptLocation: 's3://fakebucket/fakefolder/fakefolder/fakefile.py'
            }
        },
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }],
    };
    const id = 'test-kinesisstreams-lambda';
    const construct = new lib_1.KinesisstreamsToGluejob(stack, id, props);
    // check for role creation
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com",
                        },
                    }],
                Version: "2012-10-17",
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction",
        },
        Type: "AWS::IAM::Role"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check for Kinesis Stream
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Properties: {
            RetentionPeriodHours: 24,
            ShardCount: 1,
            StreamEncryption: {
                EncryptionType: "KMS",
                KeyId: "alias/aws/kinesis",
            },
        },
        Type: "AWS::Kinesis::Stream"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check policy to allow read access to Kinesis Stream
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        Type: "AWS::IAM::Policy",
        Properties: {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "glue:GetJob",
                        Effect: "Allow",
                        Resource: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":glue:",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ":",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":job/",
                                    {
                                        Ref: "testkinesisstreamslambdaKinesisETLJobF9454612"
                                    }
                                ]
                            ]
                        }
                    },
                    {
                        Action: "glue:GetSecurityConfiguration",
                        Effect: "Allow",
                        Resource: "*"
                    },
                    {
                        Action: "glue:GetTable",
                        Effect: "Allow",
                        Resource: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":table/",
                                        {
                                            Ref: "GlueDatabase"
                                        },
                                        "/",
                                        {
                                            Ref: "GlueTable"
                                        }
                                    ]
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":database/",
                                        {
                                            Ref: "GlueDatabase"
                                        }
                                    ]
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":catalog"
                                    ]
                                ]
                            }
                        ]
                    },
                    {
                        Action: "cloudwatch:PutMetricData",
                        Condition: {
                            StringEquals: {
                                "cloudwatch:namespace": "Glue"
                            },
                            Bool: {
                                "aws:SecureTransport": "true"
                            }
                        },
                        Effect: "Allow",
                        Resource: "*"
                    },
                    {
                        Action: [
                            "kinesis:DescribeStream",
                            "kinesis:DescribeStreamSummary",
                            "kinesis:GetRecords",
                            "kinesis:GetShardIterator",
                            "kinesis:ListShards",
                            "kinesis:SubscribeToShard"
                        ],
                        Effect: "Allow",
                        Resource: {
                            "Fn::GetAtt": [
                                "testkinesisstreamslambdaKinesisStream374D6D56",
                                "Arn"
                            ]
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            PolicyName: "testkinesisstreamslambdaGlueJobPolicy10DEE7DE",
            Roles: [
                {
                    Ref: "testkinesisstreamslambdaJobRole42199B9C"
                }
            ]
        },
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "Glue Security Configuration does not have an ARN, and the policy only allows reading the configuration.            CloudWatch metrics also do not have an ARN but adding a namespace condition to the policy to allow it to            publish metrics only for AWS Glue"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // Check for cloudwatch alarm
    expect(stack).toCountResources('AWS::CloudWatch::Alarm', 2);
    // Check for properties
    expect(construct.database).toBeDefined();
    expect(construct.glueJob).toBeDefined();
    expect(construct.table).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.glueJobRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
// --------------------------------------------------------------
// Test if existing Glue Job is provided
// --------------------------------------------------------------
test('Test if existing Glue Job is provided', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const _jobRole = defaults.createGlueJobRole(stack);
    const existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: 's3://fakebucket/fakepath/fakepath/fakefile.py'
        },
        role: _jobRole.roleArn,
        securityConfiguration: 'testSecConfig'
    });
    new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', {
        existingGlueJob: existingCfnJob,
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }],
    });
    // check for Kinesis Stream
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Properties: {
            RetentionPeriodHours: 24,
            ShardCount: 1,
            StreamEncryption: {
                EncryptionType: "KMS",
                KeyId: "alias/aws/kinesis",
            },
        },
        Type: "AWS::Kinesis::Stream"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test if existing S3 bucket location for script is provided
// --------------------------------------------------------------
test('When S3 bucket location for script exists', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const _s3ObjectUrl = 's3://fakelocation/etl/fakefile.py';
    const props = {
        glueJobProps: {
            command: {
                name: 'pythonshell',
                pythonVersion: '3',
                scriptLocation: _s3ObjectUrl
            }
        },
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }],
        outputDataStore: {
            datastoreType: core_2.SinkStoreType.S3
        }
    };
    new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', props);
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: 'AWS::Glue::Job',
        Properties: {
            Command: {
                Name: "pythonshell",
                PythonVersion: "3",
                ScriptLocation: "s3://fakelocation/etl/fakefile.py",
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test when the construct is supplied with an existing stream
// --------------------------------------------------------------
test('create glue job with existing kinesis stream', () => {
    const stack = new core_1.Stack();
    const _kinesisStream = new aws_kinesis_1.Stream(stack, 'FakeStream', {
        streamName: 'fakename',
        encryption: aws_kinesis_1.StreamEncryption.UNENCRYPTED,
        shardCount: 3,
        retentionPeriod: core_1.Duration.hours(30)
    });
    new lib_1.KinesisstreamsToGluejob(stack, 'existingStreamJob', {
        glueJobProps: {
            command: {
                name: 'pythonshell',
                pythonVersion: '3',
                scriptLocation: 's3://fakes3bucket/fakepath/fakefile.py'
            }
        },
        existingStreamObj: _kinesisStream,
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }],
        outputDataStore: {
            datastoreType: core_2.SinkStoreType.S3
        }
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Type: 'AWS::Kinesis::Stream',
        Properties: {
            Name: 'fakename',
            RetentionPeriodHours: 30,
            ShardCount: 3
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test if no script loocation is provided
// --------------------------------------------------------------
test('Do not pass s3ObjectUrlForScript or scriptLocationPath, error out', () => {
    const stack = new core_1.Stack();
    try {
        const _kinesisStream = defaults.buildKinesisStream(stack, {});
        new lib_1.KinesisstreamsToGluejob(stack, 'existingStreamJob', {
            glueJobProps: {
                command: {
                    name: 'pythonshell',
                    scriptLocation: 's3://fakebucket/fakepath/fakefile.py'
                }
            },
            existingStreamObj: _kinesisStream,
            fieldSchema: [{
                    name: "id",
                    type: "int",
                    comment: "Identifier for the record"
                }, {
                    name: "name",
                    type: "string",
                    comment: "The name of the record"
                }, {
                    name: "type",
                    type: "string",
                    comment: "The type of the record"
                }, {
                    name: "numericvalue",
                    type: "int",
                    comment: "Some value associated with the record"
                }],
            outputDataStore: {
                datastoreType: core_2.SinkStoreType.S3
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Test when neither CfnTable nor Table schem structure is provided
// --------------------------------------------------------------
test('Do not pass fieldSchame or table (CfnTable), error out', () => {
    const stack = new core_1.Stack();
    try {
        const props = {
            glueJobProps: {
                command: {
                    name: 'glueetl',
                    pythonVersion: '3',
                    scriptPath: `s3://fakebucket/fakepath/fakefile.py`
                }
            },
            outputDataStore: {
                datastoreType: core_2.SinkStoreType.S3
            }
        };
        new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', props);
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Provide a database and table
// --------------------------------------------------------------
test('When database and table are provided', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const _database = new aws_glue_1.CfnDatabase(stack, 'fakedb', {
        catalogId: 'fakecatalogId',
        databaseInput: {
            description: 'a fake glue db'
        }
    });
    const props = {
        glueJobProps: {
            command: {
                name: 'glueetl',
                pythonVersion: '3',
                scriptLocation: 's3://fakebucket/fakefolder/fakefolder/fakefile.py'
            }
        },
        existingDatabase: _database,
        existingTable: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    };
    new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', props);
    expect(stack).toHaveResourceLike('AWS::Glue::Database', {
        Type: "AWS::Glue::Database",
        Properties: {
            CatalogId: "fakecatalogId",
            DatabaseInput: {
                Description: "a fake glue db"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// When database and table are not provided & cloudwatch alarms set to false
// --------------------------------------------------------------
test('When database and table are not provided & cloudwatch alarms set to false', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const props = {
        createCloudWatchAlarms: false,
        glueJobProps: {
            command: {
                name: 'glueetl',
                pythonVersion: '3',
                scriptLocation: 's3://fakebucket/fakefolder/fakefolder/fakefile.py'
            }
        },
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }]
    };
    const construct = new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', props);
    expect(stack).toHaveResourceLike('AWS::Glue::Database', {
        Type: "AWS::Glue::Database",
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseInput: {
                Description: "An AWS Glue database generated by AWS Solutions Construct"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Type: 'AWS::Glue::Table',
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "GlueDatabase"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Type: "AWS::Glue::Table",
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "GlueDatabase"
            },
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Comment: "Identifier for the record",
                            Name: "id",
                            Type: "int"
                        },
                        {
                            Comment: "The name of the record",
                            Name: "name",
                            Type: "string"
                        },
                        {
                            Comment: "The type of the record",
                            Name: "type",
                            Type: "string"
                        },
                        {
                            Comment: "Some value associated with the record",
                            Name: "numericvalue",
                            Type: "int"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        Ref: "testkinesisstreamslambdaKinesisStream374D6D56"
                    },
                    NumberOfBuckets: -1,
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: {
                            Ref: "testkinesisstreamslambdaKinesisStream374D6D56"
                        },
                        typeOfData: "kinesis"
                    },
                    SerdeInfo: {
                        Parameters: {
                            paths: "id,name,type,numericvalue"
                        },
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    }
                },
                TableType: "EXTERNAL_TABLE"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // Cloudwatch alarms is set to false, no CFN def should exist
    expect(stack).not.toHaveResource('AWS::CloudWatch::Alarm');
    // Since alarms is set to false, cloudwatch alarms property should be undefined
    expect(construct.cloudwatchAlarms).toBeUndefined();
});
test('When Asset for local file is defined', () => {
    const stack = new core_1.Stack();
    const etlAsset = new s3assets.Asset(stack, 'ETL', {
        path: path.join(__dirname, 'fakefile.py')
    });
    const props = {
        etlCodeAsset: etlAsset,
        glueJobProps: {
            command: {
                name: 'glueetl',
                pythonVersion: '3'
            }
        },
        fieldSchema: [{
                name: "id",
                type: "int",
                comment: "Identifier for the record"
            }, {
                name: "name",
                type: "string",
                comment: "The name of the record"
            }, {
                name: "type",
                type: "string",
                comment: "The type of the record"
            }, {
                name: "numericvalue",
                type: "int",
                comment: "Some value associated with the record"
            }],
    };
    const id = 'test-kinesisstreams-lambda';
    const construct = new lib_1.KinesisstreamsToGluejob(stack, id, props);
    // Check for properties
    expect(construct.database).toBeDefined();
    expect(construct.glueJob).toBeDefined();
    expect(construct.table).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.glueJobRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testkinesisstreamslambdaS3Bucket54759F5C",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "testkinesisstreamslambdaS3Bucket54759F5C",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                },
                {
                    Action: [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":s3:::", {}
                                ]
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":s3:::", {},
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testkinesisstreamslambdaJobRoleDefaultPolicy943FFA49",
        Roles: [
            {
                Ref: "testkinesisstreamslambdaJobRole42199B9C"
            }
        ]
    }, assert_1.ResourcePart.Properties);
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: 'AWS::Glue::Job',
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: {}
            },
            Role: {},
            DefaultArguments: {},
            GlueVersion: "2.0",
            NumberOfWorkers: 2,
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "G.1X"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
//# sourceMappingURL=data:application/json;base64,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