"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
const lib_1 = require("../lib");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_2.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-kinesisstream-gluejob';
const scriptBucket = new aws_s3_1.Bucket(stack, 'existingScriptLocation', {
    versioned: false,
    lifecycleRules: [{
            expiration: core_1.Duration.days(30)
        }],
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryption: aws_s3_1.BucketEncryption.S3_MANAGED
});
scriptBucket.node.defaultChild.cfnOptions.metadata = {
    cfn_nag: {
        rules_to_suppress: [{
                id: 'W51',
                reason: 'This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct implementation'
            }, {
                id: 'W35',
                reason: 'This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct implementation'
            }, {
                id: 'W41',
                reason: 'This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct'
            }]
    }
};
const job = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
    command: {
        name: 'glueetl',
        pythonVersion: '3',
        scriptLocation: scriptBucket.bucketArn,
    },
    role: new aws_iam_1.Role(stack, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    }).roleArn
});
// Definitions
new lib_1.KinesisstreamsToGluejob(stack, 'test-kinesisstreams-lambda', {
    existingGlueJob: job,
    fieldSchema: [{
            name: "id",
            type: "int",
            comment: "Identifier for the record"
        }, {
            name: "name",
            type: "string",
            comment: "The name of the record"
        }, {
            name: "type",
            type: "string",
            comment: "The type of the record"
        }, {
            name: "numericvalue",
            type: "int",
            comment: "Some value associated with the record"
        }],
    outputDataStore: {
        datastoreType: core_2.SinkStoreType.S3
    }
});
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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