"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisstreamsToGluejob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
/**
 * @summary = This construct either creates or uses the existing construct provided that can be deployed
 * to perform streaming ETL operations using:
 *    - AWS Glue Database
 *    - AWS Glue Table
 *    - AWS Glue Job
 *    - Amazon Kinesis Data Streams
 *    - Amazon S3 Bucket (output datastore).
 * The construct also configures the required role policies so that AWS Glue Job can read data from
 * the streams, process it, and write to an output store.
 */
class KinesisstreamsToGluejob extends core_1.Construct {
    /**
     * Constructs a new instance of KinesisstreamsToGluejob.Based on the values set in the @props
     *
     * @param scope
     * @param id
     * @param props
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // custom props check
        if (props.existingGlueJob && props.glueJobProps) {
            throw Error("Either existingGlueJob instance or glueJobProps should be set, but found both");
        }
        if (!props.existingGlueJob) {
            if (!props.glueJobProps.command.scriptLocation && !props.etlCodeAsset) {
                throw Error('Either one of CfnJob.JobCommandProperty.scriptLocation or KinesisstreamsToGluejobProps.etlCodeAsset has ' +
                    'to be provided. If the ETL Job code file exists in a local filesystem, please set ' +
                    'KinesisstreamsToGluejobProps.etlCodeAsset. If the ETL Job is available in an S3 bucket, set the ' +
                    'CfnJob.JobCommandProperty.scriptLocation property');
            }
            if (!props.etlCodeAsset) {
                const s3Url = props.glueJobProps.command.scriptLocation;
                const found = s3Url.match(/^s3:\/\/\S+\/\S+/g);
                if (!(found && found.length > 0 && found[0].length === s3Url.length)) {
                    throw Error("Invalid S3 URL provided");
                }
            }
        }
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps,
        });
        this.database = props.existingDatabase !== undefined ? props.existingDatabase : defaults.createGlueDatabase(scope, props.databaseProps);
        if (props.fieldSchema === undefined && props.existingTable === undefined && props.tableProps === undefined) {
            throw Error("Either fieldSchema or table property has to be set, both cannot be optional");
        }
        if (props.existingTable !== undefined) {
            this.table = props.existingTable;
        }
        else {
            this.table = defaults.createGlueTable(scope, this.database, props.tableProps, props.fieldSchema, "kinesis", {
                STREAM_NAME: this.kinesisStream.streamName,
            });
        }
        [this.glueJob, this.glueJobRole, this.outputBucket] = defaults.buildGlueJob(this, {
            existingCfnJob: props.existingGlueJob,
            glueJobProps: props.glueJobProps,
            table: this.table,
            database: this.database,
            outputDataStore: props.outputDataStore,
            etlCodeAsset: props.etlCodeAsset
        });
        this.glueJobRole = this.buildRolePolicy(scope, id, this.database, this.table, this.glueJob, this.glueJobRole);
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
    /**
     * Updates the AWS Glue Job role to include additional policies required for the ETL job to execute
     *
     * @param scope
     * @param glueDatabase
     * @param glueTable
     * @param glueJob
     * @param role
     */
    buildRolePolicy(scope, id, glueDatabase, glueTable, glueJob, role) {
        const _glueJobPolicy = new aws_iam_1.Policy(scope, `${id}GlueJobPolicy`, {
            statements: [new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ["glue:GetJob"],
                    resources: [
                        `arn:${core_1.Aws.PARTITION}:glue:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:job/${glueJob.ref}`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ["glue:GetSecurityConfiguration"],
                    resources: ["*"],
                }),
                new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ["glue:GetTable"],
                    resources: [
                        `arn:${core_1.Aws.PARTITION}:glue:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:table/${glueDatabase.ref}/${glueTable.ref}`,
                        `arn:${core_1.Aws.PARTITION}:glue:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:database/${glueDatabase.ref}`,
                        `arn:${core_1.Aws.PARTITION}:glue:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:catalog`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ["cloudwatch:PutMetricData"],
                    resources: ["*"],
                    conditions: {
                        StringEquals: {
                            "cloudwatch:namespace": "Glue",
                        },
                        Bool: {
                            "aws:SecureTransport": "true",
                        },
                    },
                }),
                new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ["kinesis:DescribeStream", "kinesis:DescribeStreamSummary", "kinesis:GetRecords",
                        "kinesis:GetShardIterator", "kinesis:ListShards", "kinesis:SubscribeToShard"],
                    resources: [this.kinesisStream.streamArn],
                })],
        });
        defaults.addCfnSuppressRules(_glueJobPolicy, [
            {
                id: "W12",
                reason: "Glue Security Configuration does not have an ARN, and the policy only allows reading the configuration.            CloudWatch metrics also do not have an ARN but adding a namespace condition to the policy to allow it to            publish metrics only for AWS Glue",
            },
        ]);
        role.attachInlinePolicy(_glueJobPolicy);
        return role;
    }
}
exports.KinesisstreamsToGluejob = KinesisstreamsToGluejob;
_a = JSII_RTTI_SYMBOL_1;
KinesisstreamsToGluejob[_a] = { fqn: "@aws-solutions-constructs/aws-kinesisstreams-gluejob.KinesisstreamsToGluejob", version: "1.154.0" };
//# sourceMappingURL=data:application/json;base64,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