"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisFirehoseToS3 = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const kinesisfirehose = require("@aws-cdk/aws-kinesisfirehose");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-constructs/core");
const cdk = require("@aws-cdk/core");
const kms = require("@aws-cdk/aws-kms");
class KinesisFirehoseToS3 extends core_1.Construct {
    /**
     * Constructs a new instance of the KinesisFirehoseToS3 class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {KinesisFirehoseToS3Props} props - user provided props for the construct.
     * @since 0.8.0-beta
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        let bucket;
        // Setup S3 Bucket
        if (!props.existingBucketObj) {
            let bucketProps = props.bucketProps ?? {};
            bucketProps = props.existingLoggingBucketObj ?
                core_2.overrideProps(bucketProps, { serverAccessLogsBucket: props.existingLoggingBucketObj }) :
                bucketProps;
            // Setup logging S3 Bucket
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps,
                loggingBucketProps: props.loggingBucketProps,
                logS3AccessLogs: props.logS3AccessLogs,
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        this.s3BucketInterface = bucket;
        // Setup Cloudwatch Log group & stream for Kinesis Firehose
        this.kinesisFirehoseLogGroup = defaults.buildLogGroup(this, "firehose-log-group", props.logGroupProps);
        const cwLogStream = this.kinesisFirehoseLogGroup.addStream("firehose-log-stream");
        // Setup the IAM Role for Kinesis Firehose
        this.kinesisFirehoseRole = new iam.Role(this, "KinesisFirehoseRole", {
            assumedBy: new iam.ServicePrincipal("firehose.amazonaws.com"),
        });
        // Setup the IAM policy for Kinesis Firehose
        const firehosePolicy = new iam.Policy(this, "KinesisFirehosePolicy", {
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        "s3:AbortMultipartUpload",
                        "s3:GetBucketLocation",
                        "s3:GetObject",
                        "s3:ListBucket",
                        "s3:ListBucketMultipartUploads",
                        "s3:PutObject",
                    ],
                    resources: [`${bucket.bucketArn}`, `${bucket.bucketArn}/*`],
                }),
                new iam.PolicyStatement({
                    actions: ["logs:PutLogEvents"],
                    resources: [
                        `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${this.kinesisFirehoseLogGroup.logGroupName}:log-stream:${cwLogStream.logStreamName}`,
                    ],
                }),
            ],
        });
        // Attach policy to role
        firehosePolicy.attachToRole(this.kinesisFirehoseRole);
        const awsManagedKey = kms.Alias.fromAliasName(scope, "aws-managed-key", "alias/aws/s3");
        // Setup the default Kinesis Firehose props
        let defaultKinesisFirehoseProps = defaults.DefaultCfnDeliveryStreamProps(bucket.bucketArn, this.kinesisFirehoseRole.roleArn, this.kinesisFirehoseLogGroup.logGroupName, cwLogStream.logStreamName, awsManagedKey);
        // if the client didn't explicity say it was a Kinesis client, then turn on encryption
        if (!props.kinesisFirehoseProps ||
            !props.kinesisFirehoseProps.deliveryStreamType ||
            props.kinesisFirehoseProps.deliveryStreamType !== 'KinesisStreamAsSource') {
            defaultKinesisFirehoseProps = defaults.overrideProps(defaultKinesisFirehoseProps, {
                deliveryStreamEncryptionConfigurationInput: {
                    keyType: "AWS_OWNED_CMK",
                },
            });
        }
        const kinesisFirehoseProps = core_2.consolidateProps(defaultKinesisFirehoseProps, props.kinesisFirehoseProps);
        this.kinesisFirehose = new kinesisfirehose.CfnDeliveryStream(this, "KinesisFirehose", kinesisFirehoseProps);
    }
}
exports.KinesisFirehoseToS3 = KinesisFirehoseToS3;
_a = JSII_RTTI_SYMBOL_1;
KinesisFirehoseToS3[_a] = { fqn: "@aws-solutions-constructs/aws-kinesisfirehose-s3.KinesisFirehoseToS3", version: "1.175.0" };
//# sourceMappingURL=data:application/json;base64,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